<?php
namespace App\Http\Controllers\API;
use App\Http\Controllers\Controller;
use App\Models\IncomeWallet;
use App\Models\User;
use Illuminate\Http\Request;
use App\Models\Cart;
use App\Models\DeliveryFeeMaster;
use App\Models\OrderDetails;
use App\Models\Product;
use App\Models\ProductVarient;
use App\Models\Address;
use Illuminate\Support\Facades\Validator;
use Illuminate\Support\Facades\Log;
use App\Models\shopWalletIncome;
use Illuminate\Support\Facades\DB;
class CheckoutController extends Controller
{
      public function create_order(Request $request)
{
    $validator = Validator::make($request->all(), [
        'amount' => 'required|numeric',
    ]);
    if ($validator->fails()) {
        return response()->json([
            'status' => 'error',
            'message' => $validator->errors()->first(),
        ], 400);
    }
    $api = new \Razorpay\Api\Api(env('RAZORPAY_KEY'), env('RAZORPAY_SECRET'));
    $order = $api->order->create([
        'receipt' => 'rcptid_' . time(),
        'amount' => $request->amount * 100,
        'currency' => 'INR',
        'payment_capture' => 1
    ]);
    return response()->json([
        'status' => 'success',
        'message' => 'Order created successfully',
        'data' => [
            'order_id' => $order->id,
            'amount_paise' => $order->amount,
            'currency' => $order->currency,
        ],
    ]);
}
public function order_place(Request $request)
    {
        if (!auth()->guard('sanctum')->check()) {
            return response()->json([
                'status' => 'error',
                'message' => 'Please login!',
                'data' => [],
                'code' => 401,
            ]);
        }
        Log::info($request->all());
        $user = auth()->guard('sanctum')->user();
        $validator = Validator::make($request->all(), [
        'cart_items'              => 'required|array',
        'cart_items.*.product_id' => 'required|exists:products,product_id',
        'cart_items.*.quantity'   => 'required|integer|min:1',
        'cart_items.*.variant_id' => 'nullable|exists:product_varients,variant_id',
        'cash_wallet'             => 'nullable',
        'shopp_wallet'            => 'nullable',
        'packing_instruction'     => 'nullable',
        'delivery_address'        => 'nullable|string',
        'razorpay_payment_id'     => 'nullable|string',
        'razorpay_order_id'       => 'nullable|string',
        ]);
        if ($validator->fails()) {
            return response()->json([
                'status'  => 'error',
                'message' => 'Validation failed.',
                'errors'  => $validator->errors(),
                'code'    => 422,
            ], 422);
        }
        try {
            DB::beginTransaction();
            do {
                $orderRandomId = 'ORD' . rand(1000, 9999);
            } while (OrderDetails::where('order_random_id', $orderRandomId)->exists());
            $incomeWallet = IncomeWallet::where('user_id', $user->user_id)->first();
            if (!$incomeWallet) {
                return response()->json([
                    'status' => 'error',
                    'message' => 'Wallet not found.',
                    'code' => 404,
                ]);
            }
            $cashUsed = 0;
            $shoppUsed = 0;
            if (!empty($request->cart_items)) {
                foreach ($request->cart_items as $item) {
                    $cashUsed += (float) ($item['cash_wallet'] ?? 0);
                    $shoppUsed += (float) ($item['shopp_wallet'] ?? 0);
                }
            }
            Log::info("Total Cash Used: $cashUsed, Total Shopping Used: $shoppUsed");
            if ($cashUsed > 0 && $cashUsed > $incomeWallet->cash_wallet) {
                return response()->json(['status' => 'error', 'message' => 'Insufficient cash wallet balance.', 'code' => 400]);
            }
            if ($shoppUsed > 0 && $shoppUsed > $incomeWallet->shopping_wallet) {
                return response()->json(['status' => 'error', 'message' => 'Insufficient shopping wallet balance.', 'code' => 400]);
            }
            $incomeWallet->cash_wallet -= $cashUsed;
            Log::info("After Cash Deduction: {$incomeWallet->cash_wallet}");
            $incomeWallet->shopping_wallet -= $shoppUsed;
            Log::info("After Shopping Deduction: {$incomeWallet->shopping_wallet}");
            $incomeWallet->cash_wallet_deduction += $cashUsed;
            Log::info("cash_wallet_deduction: {$incomeWallet->cash_wallet_deduction}");
            $incomeWallet->shopping_wallet_deduction += $shoppUsed;
            Log::info("shopping_wallet_deduction: {$incomeWallet->shopping_wallet_deduction}");
            $incomeWallet->saveOrFail();
            Log::info("Wallet updated successfully.");
            $subtotal = 0;
            $address = Address::getCurrentAddress($user->id);
            $houseNo = null;
            $roadName = null;
            $landmark = null;
            $district = null;
            $state = null;
            $pin = null;
            $deliveryAddress = null;
            if ($address) {
                $houseNo  = $address->house_no;
                $roadName = $address->road_name;
                $landmark = $address->landmark;
                $district = $address->district;
                $state    = $address->state;
                $pin      = $address->pin;
                $deliveryAddress = implode(', ', array_filter([
                    $address->name, 
                    $address->house_no,
                    $address->road_name,
                    $address->landmark,
                    $address->district,
                    $address->state,
                    $address->pin,
                ]));
            }
            if ($request->filled('delivery_address')) {
                $houseNo         = $request->house_no ?? null;
                $roadName        = $request->road_name ?? null;
                $landmark        = $request->landmark ?? null;
                $district        = $request->district ?? null;
                $state           = $request->state ?? null;
                $pin             = $request->pin ?? null;
                $deliveryAddress = $request->delivery_address;
            } else {
                $deliveryAddress = $address ? implode(', ', array_filter([
                    $address->name, 
                    $address->house_no,
                    $address->road_name,
                    $address->landmark,
                    $address->district,
                    $address->state,
                    $address->pin,
                ])) : null;
            }
            if (!$deliveryAddress) {
                return response()->json([
                    'status' => 'error',
                    'message' => 'Delivery address is required.',
                    'code' => 422,
                ], 422);
            }
            $cartItems    = [];
            $orderDetails = [];
            $hsn_code     = [];
            $totalGst     = 0;
            $instruction  = $request->input('packing_instruction');
            Log::info("Packing Instruction: " . $instruction);
            $imageArray = [];
            foreach ($request->cart_items as $item) {
                $product = Product::with('category')->where('product_id', $item['product_id'])->first();
                if (!$product) {
                    return response()->json([
                        'status' => 'error',
                        'message' => 'Product not found.',
                        'code' => 404,
                    ], 404);
                }
                $productVariant = null;
                if (!empty($item['variant_id'])) {
                    $productVariant = ProductVarient::where('variant_id', $item['variant_id'])
                        ->where('product_id', $item['product_id'])
                        ->first();
                    if (!$productVariant) {
                        return response()->json([
                            'status' => 'error',
                            'message' => 'Product variant not found.',
                            'code' => 404,
                        ], 404);
                    }
                }
                // Check for sufficient stock before proceeding
                $quantityToOrder = $item['quantity'];
                if ($productVariant) {
                    if ($productVariant->stock < $quantityToOrder) {
                        return response()->json([
                            'status' => 'error',
                            'message' => 'Insufficient stock for product variant: ' . $product->title . ' (Variant: ' . $productVariant->variant_name . ')',
                            'code' => 400,
                        ], 400);
                    }
                } else {
                    if ($product->stock < $quantityToOrder) {
                        return response()->json([
                            'status' => 'error',
                            'message' => 'Insufficient stock for product: ' . $product->title,
                            'code' => 400,
                        ], 400);
                    }
                }
                // Update stock
                if ($productVariant) {
                    $productVariant->stock -= $quantityToOrder;
                    $productVariant->save();
                } else {
                    $product->stock -= $quantityToOrder;
                    $product->save();
                }
                $originalPrice  = $productVariant ? $productVariant->sell_price : $product->sell_price;
                $itemSubtotal   = $originalPrice * $item['quantity'];
                $subtotal      += $itemSubtotal;
                $priceDetails   = $productVariant ? $productVariant->sell_price : $product->sell_price;
               $hsn_code[]      = $productVariant ? $productVariant->hsn_code : $product->hsn_code;
                $gstPrice       = $productVariant ? $productVariant->gst_price : $product->gst_price;
                Log::info($gstPrice);
                $imagePath = $product->product_image ? json_decode($product->product_image, true)[0] ?? null : null;
                if ($imagePath) {
                    $imageArray[] = $imagePath;
                }
                $deliveryFee = 0;
                if ($subtotal <= 1000) {
                    $deliveryFeeMaster = DeliveryFeeMaster::first();
                    if ($deliveryFeeMaster) {
                        $deliveryFee = $deliveryFeeMaster->delivery_fee;
                    }
                }
                $orderDetails[] = [
                     'razorpay_payment_id'  => $request->razorpay_payment_id,
            'razorpay_order_id'    => $request->razorpay_order_id,
                    'product_details_id'   => $product->product_id,
                    'product_details_name' => $product->title,
                    'quantity'             => $item['quantity'],
                    'amount'               => $itemSubtotal,
                    'category_id'          => $product->category_id ?? null,
                    'variant_id'           => $item['variant_id'] ?? null,
                    'category_name'        => $product->category_name ?? 'Not Available',
                    'subcategory_id'       => $product->subcategory_id ?? null,
                    'subcategory_name'     => $product->subcategory_name ?? 'Not Available',
                    'image'                => $imagePath,
                    'priceDetails'         => $priceDetails,
                    'cash_wallet'          => $item['cash_wallet'] ?? 0,
                    'shopp_wallet'         => $item['shopp_wallet'] ?? 0,
                    'gst_price'            => $gstPrice,
                    'packing_instruction'  => $instruction,
                    'hsn_code'             => $hsn_code,
                ];
                $cartItems[] = [
                    'product_details_id'   => $product->product_id,
                    'product_details_name' => $product->title,
                    'category_id'          => $product->category_id ?? null,
                    'category_name'        => $product->category_name ?? 'Not Available',
                    'subcategory_id'       => $product->subcategory_id ?? null,
                    'subcategory_name'     => $product->subcategory_name ?? 'Not Available',
                    'variant_id'           => $item['variant_id'] ?? null,
                    'quantity'             => $item['quantity'],
                    'price'                => number_format($originalPrice, 2),
                    'subtotal'             => number_format($itemSubtotal, 2),
                    'priceDetails'         => $priceDetails,
                    'cash_wallet'          => $item['cash_wallet'] ?? 0,
                    'shopp_wallet'         => $item['shopp_wallet'] ?? 0,
                    'gst_price'            => $gstPrice,
                    'amount'               => $subtotal + $gstPrice + $deliveryFee - ($cashUsed + $shoppUsed),
                    'instruction'          => $instruction,
                      'hsn_code'             => $hsn_code,
                ];
                $totalGst += (float) $gstPrice;
            }
            OrderDetails::create([
            'order_date'           => now(),
            'user_id'              => $user->user_id,
            'user_name'            => $user->name,
            'category_id'          => json_encode(array_column($orderDetails, 'category_id')),
            'category_name'        => json_encode(array_column($orderDetails, 'category_name')),
            'subcategory_id'       => json_encode(array_column($orderDetails, 'subcategory_id')),
            'subcategory_name'     => json_encode(array_column($orderDetails, 'subcategory_name')),
            'order_random_id'      => $orderRandomId,
            'product_details_id'   => json_encode(array_column($orderDetails, 'product_details_id')),
            'product_details_name' => json_encode(array_column($orderDetails, 'product_details_name')),
            'quantity'             => json_encode(array_column($orderDetails, 'quantity')),
            'amount'               => $subtotal + $deliveryFee  - ($cashUsed + $shoppUsed),
            'delivery_address'     => $deliveryAddress,
            'variant_id'           => json_encode(array_column($orderDetails, 'variant_id')),
            'image'                => json_encode($imageArray),
            'price_details'        => json_encode(array_column($orderDetails, 'priceDetails')),
            'cash_wallet'          => $cashUsed,
            'shopp_wallet'         => $shoppUsed,
            'gst_price'            => $totalGst,
            'instruction'          => $instruction,
            'delivery_fee'         => $deliveryFee,
            'house_no'             => $houseNo,
            'road_name'            => $roadName,
            'landmark'             => $landmark,
            'district'             => $district,
            'state'                => $state,
            'pin'                  => $pin,
            'hsn_code'             => json_encode($hsn_code),
            'razorpay_payment_id'  => $request->razorpay_payment_id,
            'razorpay_order_id'    => $request->razorpay_order_id,
            ]);
            $welcomeIncome               = new shopWalletIncome();
            $welcomeIncome->user_id      = $user->user_id;
            $welcomeIncome->amount       = $subtotal + $totalGst - ($cashUsed + $shoppUsed);
            $welcomeIncome->remark       = "product purchase";
            $welcomeIncome->description  = json_encode(array_column($orderDetails, 'product_details_id'));
            $welcomeIncome->type         = "debit";
            $welcomeIncome->payment_mode = "upi";
            $welcomeIncome->save();
            Cart::where('user_id', $user->user_id)->delete();
            DB::commit();
            return response()->json([
                'status'  => 'success',
                'message' => 'Order placed successfully',
                'Bill_Id' => $orderRandomId,
                'data'    => [
                        'cart_items'        => $cartItems,
                        'delivery_address'  => $deliveryAddress,
                        'subtotal'          => number_format($subtotal, 2),
                        'gst_price'         => number_format($totalGst, 2),
                        'cash_wallet_used'  => number_format($cashUsed, 2),
                        'shopp_wallet_used' => number_format($shoppUsed, 2),
                        'delivery_fee'      => number_format($deliveryFee, 2),
                        'total_amount'      => number_format($subtotal + $deliveryFee - ($cashUsed + $shoppUsed), 2),
                    ],
            ], 200);
        } catch (\Exception $e) {
            DB::rollBack();
            Log::error('Error processing order placement: ' . $e->getMessage());
            return response()->json([
                'status'  => 'error',
                'message' => 'Failed to place order.',
                'error'   => $e->getMessage(),
                'code'    => 500,
            ], 500);
        }
    }
public function get_checkout_details(Request $request)
    {
        $user = auth()->guard('sanctum')->user();
        $validator = Validator::make($request->all(), [
            'wallet_type' => 'nullable|in:cash,shopping,both',
        ]);
        if ($validator->fails()) {
            return response()->json([
                'status' => 'error',
                'message' => 'Validation failed.',
                'errors' => $validator->errors(),
                'code' => 422
            ], 422);
        }
        $wallet_type = $request->input('wallet_type');
        $cartItems = Cart::where('user_id', $user->user_id)
            ->with(['product', 'variant'])
            ->where('type', 'normal')
            ->get();
        if ($cartItems->isEmpty()) {
            return response()->json([
                'status' => 'error',
                'message' => 'Your cart is empty!',
                'data' => [],
                'code' => 400
            ]);
        }
        $wallet = IncomeWallet::where('user_id', $user->user_id)->first();
        $cashWallet = $wallet ? (float)$wallet->cash_wallet : 0;
        $shoppingWallet = $wallet ? (float)$wallet->shopping_wallet : 0;
        $totalPriceIncludingGst = 0; 
        $totalShopWalletEligible = 0;
        $cartData = [];
        foreach ($cartItems as $cart) {
            $product = $cart->product;
            $variant = $cart->variant;
            if (!$product) {
                Log::warning("Product missing for cart item: {$cart->cart_id}");
                continue;
            }
            $pricePerUnit = $variant && isset($variant->sell_price) ? $variant->sell_price : $product->sell_price;
            $pricePerUnit = is_numeric($pricePerUnit) ? (float)$pricePerUnit : 0;
            $quantity = $cart->quantity;
            $shopWalletPerUnit = is_numeric($product->shop_wallet) ? (float)$product->shop_wallet : 0;
            $subtotalIncludingGst = $pricePerUnit * $quantity;
            $totalPriceIncludingGst += $subtotalIncludingGst; 
            $eligibleShopWallet = $shopWalletPerUnit * $quantity;
            $totalShopWalletEligible += $eligibleShopWallet;
            $gstPercent = is_numeric($product->gst) ? (float)$product->gst : 0;
            $subtotalBeforeGstForProduct = round($subtotalIncludingGst / (1 + ($gstPercent / 100)), 2);
            $cgstForProduct = round($subtotalBeforeGstForProduct * ($gstPercent / 200), 2);
            $sgstForProduct = round($subtotalBeforeGstForProduct * ($gstPercent / 200), 2);
            $cartData[] = [
                'product_id'   => $product->product_id,
                'variant_id'   => $variant ? $variant->variant_id : null,
                'product_name' => $product->title,
                'quantity'     => $quantity,
                // 'price'        => number_format($pricePerUnit, 2),
                 'price'      => number_format($pricePerUnit * $quantity, 2),
                'subtotal'     => number_format($subtotalIncludingGst, 2),
            ];
        }
        $deliveryFee = 0;
        if ($totalPriceIncludingGst <= 1000) {
            $deliveryFeeRow = DeliveryFeeMaster::first();
            $deliveryFee = $deliveryFeeRow ? (float)$deliveryFeeRow->delivery_fee : 0;
        }
        $subTotalWithDelivery = $totalPriceIncludingGst + $deliveryFee; 
        $currentAddress = Address::where('user_id', $user->id)->where('is_current', 1)->first();
        $shoppingWalletUsed = 0;
        $cashWalletUsed = 0;
        $remainingAmount = $subTotalWithDelivery;
        if ($wallet_type === 'shopping') {
            $shoppingWalletUsed = min($shoppingWallet, $totalShopWalletEligible);
            $remainingAmount -= $shoppingWalletUsed;
        } elseif ($wallet_type === 'cash') {
            $cashWalletUsed = min($cashWallet, $subTotalWithDelivery); // Can use cash wallet up to full bill
            $remainingAmount -= $cashWalletUsed;
        } elseif ($wallet_type === 'both') {
            $shoppingWalletUsed = min($shoppingWallet, $totalShopWalletEligible);
            $remainingAmount -= $shoppingWalletUsed;
            if ($remainingAmount > 0) {
                $cashWalletUsed = min($cashWallet, $remainingAmount);
                $remainingAmount -= $cashWalletUsed;
            }
        }
        // Ensure remaining amount is not negative due to rounding or edge cases
        $remainingAmount = max(0, $remainingAmount);
        // 8. GST flag from admin
        $is_gst = User::where('role', 'admin')->value('is_gst');
        $effectiveGstRateForSummary = 18; 
        $summaryTotalBeforeGst = 0;
        $summaryTotalGstAmount = 0;
        if ($is_gst == 1 && $effectiveGstRateForSummary > 0) {
            $summaryTotalBeforeGst = round($remainingAmount / (1 + ($effectiveGstRateForSummary / 100)), 2);
            $summaryTotalGstAmount = round($remainingAmount - $summaryTotalBeforeGst, 2);
        } else {
            $summaryTotalBeforeGst = $remainingAmount;
            $summaryTotalGstAmount = 0;
        }
        $data = [
            'order_summary' => $cartData,
            'is_gst' => $is_gst,
            'delivery_fee' => number_format($deliveryFee, 2),
            'sub_total' => number_format($subTotalWithDelivery, 2), // This is the total bill before wallet use (e.g., 880.00)
            'shopping_wallet' => number_format($shoppingWalletUsed, 2),
            'cash_wallet' => number_format($cashWalletUsed, 2),
            'total' => number_format($remainingAmount, 2), // This is the final amount to be paid (e.g., 580.00)
            'address' => $currentAddress ? [
                'name'      => $currentAddress->name,
                'house_no'  => $currentAddress->house_no,
                'road_name' => $currentAddress->road_name,
                'landmark'  => $currentAddress->landmark,
                'district'  => $currentAddress->district,
                'state'     => $currentAddress->state,
                'pin'       => $currentAddress->pin,
            ] : null,
        ];
        if ($is_gst == 1) {
            $halfGstForSummary = round($summaryTotalGstAmount / 2, 2);
            $data['total_before_gst_summary'] = number_format($summaryTotalBeforeGst, 2); // Calculated from 'total' (e.g., 580 / 1.18 = 491.53)
            $data['total_cgst'] = number_format($halfGstForSummary, 2); // Calculated from 'total' (e.g., 44.23)
            $data['total_sgst'] = number_format($halfGstForSummary, 2); // Calculated from 'total' (e.g., 44.23)
            $data['total_gst'] = number_format($summaryTotalGstAmount, 2); // Calculated from 'total' (e.g., 88.47)
        } else {
            $data['total_before_gst_summary'] = number_format($remainingAmount, 2); // If GST is off, total before GST is same as remaining amount
            $data['total_cgst'] = null;
            $data['total_sgst'] = null;
            $data['total_gst'] = number_format(0, 2); 
        }
        $data['total_bill_incl_gst_summary'] = number_format($remainingAmount, 2);
        return response()->json([
            'status' => 'success',
            'message' => 'Checkout details fetched successfully.',
            'data' => $data,
            'code' => 200,
        ]);
    }
// public function order_place(Request $request)
//     {
//         if (!auth()->guard('sanctum')->check()) {
//             return response()->json([
//                 'status' => 'error',
//                 'message' => 'Please login!',
//                 'data' => [],
//                 'code' => 401,
//             ]);
//         }
//         Log::info($request->all());
//         $user = auth()->guard('sanctum')->user();
//         $validator = Validator::make($request->all(), [
//             'cart_items'              => 'required|array',
//             'cart_items.*.product_id' => 'required|exists:products,product_id',
//             'cart_items.*.quantity'   => 'required|integer|min:1',
//             'cart_items.*.variant_id' => 'nullable|exists:product_varients,variant_id',
//             'cash_wallet'             => 'nullable',
//             'shopp_wallet'            => 'nullable',
//             'packing_instruction'     => 'nullable',
//             'delivery_address'        => 'nullable|string',
//         ]);
//         if ($validator->fails()) {
//             return response()->json([
//                 'status'  => 'error',
//                 'message' => 'Validation failed.',
//                 'errors'  => $validator->errors(),
//                 'code'    => 422,
//             ], 422);
//         }
//         try {
//             DB::beginTransaction();
//             do {
//                 $orderRandomId = 'ORD' . rand(1000, 9999);
//             } while (OrderDetails::where('order_random_id', $orderRandomId)->exists());
//             $incomeWallet = IncomeWallet::where('user_id', $user->user_id)->first();
//             if (!$incomeWallet) {
//                 return response()->json([
//                     'status' => 'error',
//                     'message' => 'Wallet not found.',
//                     'code' => 404,
//                 ]);
//             }
//             $cashUsed = 0;
//             $shoppUsed = 0;
//             if (!empty($request->cart_items)) {
//                 foreach ($request->cart_items as $item) {
//                     $cashUsed += (float) ($item['cash_wallet'] ?? 0);
//                     $shoppUsed += (float) ($item['shopp_wallet'] ?? 0);
//                 }
//             }
//             Log::info("Total Cash Used: $cashUsed, Total Shopping Used: $shoppUsed");
//             if ($cashUsed > 0 && $cashUsed > $incomeWallet->cash_wallet) {
//                 return response()->json(['status' => 'error', 'message' => 'Insufficient cash wallet balance.', 'code' => 400]);
//             }
//             if ($shoppUsed > 0 && $shoppUsed > $incomeWallet->shopping_wallet) {
//                 return response()->json(['status' => 'error', 'message' => 'Insufficient shopping wallet balance.', 'code' => 400]);
//             }
//             $incomeWallet->cash_wallet -= $cashUsed;
//             Log::info("After Cash Deduction: {$incomeWallet->cash_wallet}");
//             $incomeWallet->shopping_wallet -= $shoppUsed;
//             Log::info("After Shopping Deduction: {$incomeWallet->shopping_wallet}");
//             $incomeWallet->cash_wallet_deduction += $cashUsed;
//             Log::info("cash_wallet_deduction: {$incomeWallet->cash_wallet_deduction}");
//             $incomeWallet->shopping_wallet_deduction += $shoppUsed;
//             Log::info("shopping_wallet_deduction: {$incomeWallet->shopping_wallet_deduction}");
//             $incomeWallet->saveOrFail();
//             Log::info("Wallet updated successfully.");
//             $subtotal = 0;
//             $address = Address::getCurrentAddress($user->id);
//             $houseNo = null;
//             $roadName = null;
//             $landmark = null;
//             $district = null;
//             $state = null;
//             $pin = null;
//             $deliveryAddress = null;
//             if ($address) {
//                 $houseNo  = $address->house_no;
//                 $roadName = $address->road_name;
//                 $landmark = $address->landmark;
//                 $district = $address->district;
//                 $state    = $address->state;
//                 $pin      = $address->pin;
//                 $deliveryAddress = implode(', ', array_filter([
//                     $address->name, 
//                     $address->house_no,
//                     $address->road_name,
//                     $address->landmark,
//                     $address->district,
//                     $address->state,
//                     $address->pin,
//                 ]));
//             }
//             if ($request->filled('delivery_address')) {
//                 $houseNo         = $request->house_no ?? null;
//                 $roadName        = $request->road_name ?? null;
//                 $landmark        = $request->landmark ?? null;
//                 $district        = $request->district ?? null;
//                 $state           = $request->state ?? null;
//                 $pin             = $request->pin ?? null;
//                 $deliveryAddress = $request->delivery_address;
//             } else {
//                 $deliveryAddress = $address ? implode(', ', array_filter([
//                     $address->name, 
//                     $address->house_no,
//                     $address->road_name,
//                     $address->landmark,
//                     $address->district,
//                     $address->state,
//                     $address->pin,
//                 ])) : null;
//             }
//             if (!$deliveryAddress) {
//                 return response()->json([
//                     'status' => 'error',
//                     'message' => 'Delivery address is required.',
//                     'code' => 422,
//                 ], 422);
//             }
//             $cartItems    = [];
//             $orderDetails = [];
//             $hsn_code     = [];
//             $totalGst     = 0;
//             $instruction  = $request->input('packing_instruction');
//             Log::info("Packing Instruction: " . $instruction);
//             $imageArray = [];
//             foreach ($request->cart_items as $item) {
//                 $product = Product::with('category')->where('product_id', $item['product_id'])->first();
//                 if (!$product) {
//                     return response()->json([
//                         'status' => 'error',
//                         'message' => 'Product not found.',
//                         'code' => 404,
//                     ], 404);
//                 }
//                 $productVariant = null;
//                 if (!empty($item['variant_id'])) {
//                     $productVariant = ProductVarient::where('variant_id', $item['variant_id'])
//                         ->where('product_id', $item['product_id'])
//                         ->first();
//                     if (!$productVariant) {
//                         return response()->json([
//                             'status' => 'error',
//                             'message' => 'Product variant not found.',
//                             'code' => 404,
//                         ], 404);
//                     }
//                 }
//                 // Check for sufficient stock before proceeding
//                 $quantityToOrder = $item['quantity'];
//                 if ($productVariant) {
//                     if ($productVariant->stock < $quantityToOrder) {
//                         return response()->json([
//                             'status' => 'error',
//                             'message' => 'Insufficient stock for product variant: ' . $product->title . ' (Variant: ' . $productVariant->variant_name . ')',
//                             'code' => 400,
//                         ], 400);
//                     }
//                 } else {
//                     if ($product->stock < $quantityToOrder) {
//                         return response()->json([
//                             'status' => 'error',
//                             'message' => 'Insufficient stock for product: ' . $product->title,
//                             'code' => 400,
//                         ], 400);
//                     }
//                 }
//                 // Update stock
//                 if ($productVariant) {
//                     $productVariant->stock -= $quantityToOrder;
//                     $productVariant->save();
//                 } else {
//                     $product->stock -= $quantityToOrder;
//                     $product->save();
//                 }
//                 $originalPrice  = $productVariant ? $productVariant->sell_price : $product->sell_price;
//                 $itemSubtotal   = $originalPrice * $item['quantity'];
//                 $subtotal      += $itemSubtotal;
//                 $priceDetails   = $productVariant ? $productVariant->sell_price : $product->sell_price;
//                $hsn_code[]      = $productVariant ? $productVariant->hsn_code : $product->hsn_code;
//                 $gstPrice       = $productVariant ? $productVariant->gst_price : $product->gst_price;
//                 Log::info($gstPrice);
//                 $imagePath = $product->product_image ? json_decode($product->product_image, true)[0] ?? null : null;
//                 if ($imagePath) {
//                     $imageArray[] = $imagePath;
//                 }
//                 $deliveryFee = 0;
//                 if ($subtotal <= 1000) {
//                     $deliveryFeeMaster = DeliveryFeeMaster::first();
//                     if ($deliveryFeeMaster) {
//                         $deliveryFee = $deliveryFeeMaster->delivery_fee;
//                     }
//                 }
//                 $orderDetails[] = [
//                     'product_details_id'   => $product->product_id,
//                     'product_details_name' => $product->title,
//                     'quantity'             => $item['quantity'],
//                     'amount'               => $itemSubtotal,
//                     'category_id'          => $product->category_id ?? null,
//                     'variant_id'           => $item['variant_id'] ?? null,
//                     'category_name'        => $product->category_name ?? 'Not Available',
//                     'subcategory_id'       => $product->subcategory_id ?? null,
//                     'subcategory_name'     => $product->subcategory_name ?? 'Not Available',
//                     'image'                => $imagePath,
//                     'priceDetails'         => $priceDetails,
//                     'cash_wallet'          => $item['cash_wallet'] ?? 0,
//                     'shopp_wallet'         => $item['shopp_wallet'] ?? 0,
//                     'gst_price'            => $gstPrice,
//                     'packing_instruction'  => $instruction,
//                     'hsn_code'             => $hsn_code,
//                 ];
//                 $cartItems[] = [
//                     'product_details_id'   => $product->product_id,
//                     'product_details_name' => $product->title,
//                     'category_id'          => $product->category_id ?? null,
//                     'category_name'        => $product->category_name ?? 'Not Available',
//                     'subcategory_id'       => $product->subcategory_id ?? null,
//                     'subcategory_name'     => $product->subcategory_name ?? 'Not Available',
//                     'variant_id'           => $item['variant_id'] ?? null,
//                     'quantity'             => $item['quantity'],
//                     'price'                => number_format($originalPrice, 2),
//                     'subtotal'             => number_format($itemSubtotal, 2),
//                     'priceDetails'         => $priceDetails,
//                     'cash_wallet'          => $item['cash_wallet'] ?? 0,
//                     'shopp_wallet'         => $item['shopp_wallet'] ?? 0,
//                     'gst_price'            => $gstPrice,
//                     'amount'               => $subtotal + $gstPrice + $deliveryFee - ($cashUsed + $shoppUsed),
//                     'instruction'          => $instruction,
//                       'hsn_code'             => $hsn_code,
//                 ];
//                 $totalGst += (float) $gstPrice;
//             }
//             OrderDetails::create([
//                 'order_date'           => now(),
//                 'user_id'              => $user->user_id,
//                 'user_name'            => $user->name,
//                 'category_id'          => json_encode(array_column($orderDetails, 'category_id')),
//                 'category_name'        => json_encode(array_column($orderDetails, 'category_name')),
//                 'subcategory_id'       => json_encode(array_column($orderDetails, 'subcategory_id')),
//                 'subcategory_name'     => json_encode(array_column($orderDetails, 'subcategory_name')),
//                 'order_random_id'      => $orderRandomId,
//                 'product_details_id'   => json_encode(array_column($orderDetails, 'product_details_id')),
//                 'product_details_name' => json_encode(array_column($orderDetails, 'product_details_name')),
//                 'quantity'             => json_encode(array_column($orderDetails, 'quantity')),
//                 'amount'               => $subtotal + $deliveryFee  - ($cashUsed + $shoppUsed),
//                 'delivery_address'     => $deliveryAddress,
//                 'variant_id'           => json_encode(array_column($orderDetails, 'variant_id')),
//                 'image'                => json_encode($imageArray),
//                 'price_details'        => json_encode(array_column($orderDetails, 'priceDetails')),
//                 'cash_wallet'          => $cashUsed,
//                 'shopp_wallet'         => $shoppUsed,
//                 'gst_price'            => $totalGst,
//                 'instruction'          => $instruction,
//                 'delivery_fee'         => $deliveryFee,
//                 'house_no'             => $houseNo,
//                 'road_name'            => $roadName,
//                 'landmark'             => $landmark,
//                 'district'             => $district,
//                 'state'                => $state,
//                 'pin'                  => $pin,
//                 'hsn_code'             => json_encode($hsn_code),
//             ]);
//             $welcomeIncome               = new shopWalletIncome();
//             $welcomeIncome->user_id      = $user->user_id;
//             $welcomeIncome->amount       = $subtotal + $totalGst - ($cashUsed + $shoppUsed);
//             $welcomeIncome->remark       = "product purchase";
//             $welcomeIncome->description  = json_encode(array_column($orderDetails, 'product_details_id'));
//             $welcomeIncome->type         = "debit";
//             $welcomeIncome->payment_mode = "upi";
//             $welcomeIncome->save();
//             Cart::where('user_id', $user->user_id)->delete();
//             DB::commit();
//             return response()->json([
//                 'status'  => 'success',
//                 'message' => 'Order placed successfully',
//                 'Bill_Id' => $orderRandomId,
//                 'data'    => [
//                         'cart_items'        => $cartItems,
//                         'delivery_address'  => $deliveryAddress,
//                         'subtotal'          => number_format($subtotal, 2),
//                         'gst_price'         => number_format($totalGst, 2),
//                         'cash_wallet_used'  => number_format($cashUsed, 2),
//                         'shopp_wallet_used' => number_format($shoppUsed, 2),
//                         'delivery_fee'      => number_format($deliveryFee, 2),
//                         'total_amount'      => number_format($subtotal + $deliveryFee - ($cashUsed + $shoppUsed), 2),
//                     ],
//             ], 200);
//         } catch (\Exception $e) {
//             DB::rollBack();
//             Log::error('Error processing order placement: ' . $e->getMessage());
//             return response()->json([
//                 'status'  => 'error',
//                 'message' => 'Failed to place order.',
//                 'error'   => $e->getMessage(),
//                 'code'    => 500,
//             ], 500);
//         }
//     }
    public function orderdetails($order_random_id)
    {
        try {
            $user = auth()->guard('sanctum')->user();
            if (!$user) {
                Log::warning('User not authenticated.');
                return response()->json([
                    'status' => 'error',
                    'message' => 'User not authenticated.',
                    'code' => 401,
                ], 401);
            }
            $order = OrderDetails::where('order_random_id', $order_random_id)->first();
            if (!$order) {
                return response()->json([
                    'status'  => 'error',
                    'message' => 'Order not found.',
                    'data'    => [],
                    'code'    => 404,
                ], 404);
            }
            $productNames = json_decode($order->product_details_name, true) ?? [];
            $quantities   = json_decode($order->quantity, true) ?? [];
            $prices       = json_decode($order->price_details, true) ?? [];
            $gstPrices    = json_decode($order->gst_price_individual, true) ?? [];  // Assuming you'll store individual GST in OrderDetails
            $orderSummary = [];
            $subTotal = 0;
            $totalGst = 0;
            foreach ($productNames as $index => $productName) {
                $price        = isset($prices[$index]) ? (float) $prices[$index] : 0;
                $quantity     = isset($quantities[$index]) ? (int) $quantities[$index] : 1;
                $itemSubtotal = $price * $quantity;
                $itemGst      = isset($gstPrices[$index]) ? (float) $gstPrices[$index] : 0;
                $orderSummary[] = [
                    'product_name' => $productName,
                    'quantity'     => $quantity,
                    'price'        => number_format($price, 2),
                    'subtotal'     => number_format($itemSubtotal, 2),
                    // 'gst_price' => number_format($itemGst, 2),
                ];
                $subTotal += $itemSubtotal;
                $totalGst += $itemGst;
            }
            return response()->json([
                'status' => 'success',
                'message' => 'Order details fetched successfully.',
                'data' => [
                        'order_id'             => $order->order_random_id,
                        'order_date'           => $order->created_at,
                        'status'               => $order->status,
                        'order_summary'        => $orderSummary,
                        'sub_total'            => number_format($subTotal, 2),
                        'gst_total'            => number_format($order->gst_price ?? $totalGst, 2),                                                                                                                                // Use stored total GST
                        'delivery_fee'         => number_format($order->delivery_fee ?? 0, 2),
                        'shopping_wallet_used' => number_format($order->shopp_wallet ?? 0, 2),
                        'cash_wallet_used'     => number_format($order->cash_wallet ?? 0, 2),
                        'total_amount'         => number_format($order->amount ?? ($subTotal + ($order->gst_price ?? $totalGst) + ($order->delivery_fee ?? 0) - (($order->cash_wallet ?? 0) + ($order->shopp_wallet ?? 0))), 2),
                        'address'              => $order->delivery_address ?? '',
                        'packing_instruction'  => $order->instruction ?? '',
                    ],
                'code' => 200,
            ], 200);
        } catch (\Exception $e) {
            Log::error('Error fetching order details: ' . $e->getMessage());
            return response()->json([
                'status' => 'error',
                'message' => 'Something went wrong. Please try again.',
                'code' => 500,
            ], 500);
        }
    }
}
