<?php

namespace App\Http\Controllers\API;

use App\Http\Controllers\BrevoMailController;
use App\Http\Controllers\Controller;
use App\Mail\PasswordResetMail;
use Illuminate\Http\Request;
use App\Models\User;
use Illuminate\Support\Facades\Hash;
use Illuminate\Support\Facades\Crypt;
use Illuminate\Support\Facades\Validator;
use Illuminate\Support\Facades\Mail;
use Illuminate\Support\Facades\Log;
use Exception;
use Carbon\Carbon;
use SendinBlue\Client\Api\TransactionalEmailsApi;
use SendinBlue\Client\Configuration;
use GuzzleHttp\Client;

class ForgetPasswordController extends Controller
{
    public function create_password(Request $request)
    {
        $validated = $request->validate([
            'email'    => 'required|email',
            'password' => 'required|string|min:8|confirmed'
        ]);
        $user = User::where('email', $request->email)->first();
        if (!$user) {
            Log::error('User not found:', ['email' => $request->email]);
            return response()->json([
                'status'  => 'error',
                'message' => 'User not found.',
                'code'    => 404,
            ], 404);
        }
        $user->password = Hash::make($request->password);
        $user->save();
        Log::info('Password reset successfully for:', ['email' => $request->email]);
        return response()->json([
            'status'  => 'success',
            'message' => 'Password has been reset successfully.',
            'code'    => 200,
        ], 200);
    }

    public function forget_sendotp(Request $request)
    {
        $request->validate([
            'email' => 'required|email',
        ]);

        $user = User::where('email', $request->email)->where('role', 'user')->first();

        if (!$user) {
            Log::info("OTP request failed - User not found for email: {$request->email}");
            return response()->json(['status' => 'error', 'message' => 'User not found.'], 404);
        }

       $otp                  = rand(10000, 99999);
       $encryptedOtp         = Crypt::encryptString($otp);
       $expiryTime           = Carbon::now()->addMinutes(2);
       $user->otp            = $encryptedOtp;
       $user->otp_expires_at = $expiryTime;
        $user->save();

        try {
            Log::info("Sending OTP to: {$user->email}");
            $sendMailResponse = $this->send_mail_api($user->email, $otp);

            if ($sendMailResponse['status'] === 'success') {
                Log::info("OTP sent successfully to {$user->email}");

                return response()->json([
                    'status'  => 'success',
                    'message' => 'OTP sent successfully.',
                    'email'   => $user->email
                ]);
            } else {
                Log::info("OTP send failed for {$user->email}: " . $sendMailResponse['error']);

                return response()->json([
                    'status'  => 'error',
                    'message' => 'Failed to send OTP. Please try again later.'
                ], 500);
            }
        } catch (\Exception $e) {
            Log::info('Exception while sending OTP', ['email' => $user->email, 'error' => $e->getMessage()]);

            return response()->json([
                'status' => 'error',
                'message' => 'An error occurred. Please try again later.'
            ], 500);
        }
    }

    public function send_mail_api($email, $otp)
    {
        $apiKey      = env('BREVO_API_KEY');
        $config      = Configuration::getDefaultConfiguration()->setApiKey('api-key', $apiKey);
        $apiInstance = new TransactionalEmailsApi(new Client(), $config);

        $sendSmtpEmail = new \SendinBlue\Client\Model\SendSmtpEmail([
            'subject'     => 'Your OTP Code',
            'htmlContent' => '
<div style="font-family: Arial, sans-serif; padding: 20px; background-color: #f9f9f9; color: #333;">
    <div style="max-width: 600px; margin: auto; background-color: #ffffff; border-radius: 8px; padding: 30px; box-shadow: 0 0 10px rgba(0,0,0,0.05);">
        <h2 style="color: #2c3e50; text-align: center;">Your One-Time Password (OTP)</h2>
        <p>Hello,</p>
        <p>Use the following OTP to complete your password reset process. This code is valid for a limited time only:</p>
        <div style="text-align: center; margin: 20px 0;">
            <span style="font-size: 28px; font-weight: bold; background: #f0f0f0; padding: 10px 20px; border-radius: 6px; display: inline-block; letter-spacing: 2px; color: #34495e;">' . $otp . '</span>
        </div>
        <p style="margin-top: 30px;">If you did not request this OTP, please ignore this email or contact support.</p>
        <p>Thank you,<br><strong>Adbazzar Team</strong></p>
    </div>
</div>',
            'sender'      => ['name' => 'Adbazzar', 'email' => 'smartworld2025@gmail.com'],
            'to'          => [['email' => $email]],
        ]);

        try {
            $apiInstance->sendTransacEmail($sendSmtpEmail);
            return ['status' => 'success'];
        } catch (\Exception $e) {
            Log::info('Failed to send email via Brevo', ['email' => $email, 'error' => $e->getMessage()]);
            return ['status' => 'error', 'error' => $e->getMessage()];
        }
    }

    public function forget_verifyotp(Request $request)
    {
        try {
            $request->validate([
                'email' => 'required|email',
                'otp'   => 'required|string|min:5|max:5',
            ]);

            $user = User::where('email', $request->email)->where('role', 'user')->first();

            if (!$user) {
                return response()->json(['status' => 'error', 'message' => 'Invalid User'], 404);
            }

            if (Crypt::decryptString($user->otp) !== $request->otp) {
                return response()->json(['status' => 'error', 'message' => 'The verification code is incorrect.'], 400);
            }

            return response()->json([
                'status'  => 'success',
                'message' => 'OTP verified successfully.',
                'email'   => $request->email,
                'otp'     => $request->otp,
            ]);
        } catch (\Exception $e) {
            Log::error('Error verifying OTP', [
                'email' => $request->email,
                'error' => $e->getMessage(),
            ]);
            return response()->json(['status' => 'error', 'message' => 'Something went wrong! Please try again.'], 500);
        }
    }
    public function reset_password_update(Request $request)
    {
        Log::info('Reset Password Request Data:', $request->all());

        $validator = Validator::make($request->all(), [
            'password'              => 'required|string|min:8|confirmed',
            'password_confirmation' => 'required|string|min:8',
            'email'                 => 'required|email',
            'otp'                   => 'required|string|min:5|max:5',
        ]);

        if ($validator->fails()) {
            return response()->json(['status' => 'error', 'message' => $validator->errors()->first()], 400);
        }

        try {
            $user = User::where('email', $request->email)->where('role', 'user')->first();

            if (!$user) {
                return response()->json(['status' => 'error', 'message' => 'Invalid email or OTP.'], 400);
            }

            try {
                $storedOtp = Crypt::decryptString($user->otp);
                if ($storedOtp !== $request->otp) {
                    return response()->json(['status' => 'error', 'message' => 'The verification code is incorrect.'], 400);
                }
            } catch (\Exception $e) {
                Log::error('Error decrypting OTP', ['email' => $request->email, 'error' => $e->getMessage()]);
                return response()->json(['status' => 'error', 'message' => 'Invalid OTP.'], 400);
            }

            $user->password = Hash::make($request->password);
            $user->otp = null;
            $user->save();

            return response()->json(['status' => 'success', 'message' => 'Password has been successfully reset.']);
        } catch (\Exception $e) {
            Log::error('Error resetting password', ['email' => $request->email, 'error' => $e->getMessage()]);
            return response()->json(['status' => 'error', 'message' => 'An error occurred. Please try again.'], 500);
        }
    }
}
