<?php

namespace App\Http\Controllers\API;

use App\Http\Controllers\Controller;
use App\Http\Resources\SearchProductResources;
use App\Http\Resources\WishlistResource;
use Illuminate\Http\Request;
use App\Models\Wishlist;
use App\Models\Product;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Log;

class WishlistController extends Controller
{
    public function wishlist_display()
    {
        try {
            $user = Auth::user();
            if (!$user) {
                return response()->json([
                    'status' => 'error',
                    'message' => 'User not authenticated!',
                    'data' => [],
                    'code' => 401,
                ], 401);
            }

            $wishlistItems = Wishlist::with(['product.variants'])
                ->where('user_id', $user->user_id)
                ->whereHas('product')
                ->get();

            return response()->json([
                'status' => 'success',
                'data' => WishlistResource::collection($wishlistItems),
                'message' => 'Wishlist items retrieved successfully',
                'code' => 200
            ], 200);
        } catch (\Exception $e) {
            return response()->json([
                'status' => 'error',
                'message' => 'Failed to retrieve wishlist items.',
                'code' => 500,
            ], 500);
        }
    }


    public function wishlist_delete($product_id)
    {
        $user = Auth::user();
        if (!$user) {
            return response()->json([
                'status' => 'error',
                'message' => 'User not authenticated!',
                'data' => [],
                'code' => 401,
            ], 401);
        }

        $wishlist = Wishlist::where('user_id', $user->user_id)
            ->where('product_id', $product_id)
            ->first();

        if ($wishlist) {
            $wishlist->delete();
            return response()->json([
                'status' => 'success',
                'message' => 'Product removed from wishlist',
                'data' => [],
                'code' => 200,
            ]);
        }

        return response()->json([
            'status' => 'error',
            'message' => 'Product not found in wishlist',
            'data' => [],
            'code' => 404,
        ], 404);
    }

    public function wishlist(Request $request)
    {
        try {
            if (!auth()->guard('sanctum')->check()) {
                return response()->json([
                    'status' => 'error',
                    'message' => 'Please login!',
                    'data' => [],
                    'code' => 401,
                ]);
            }
            $user = auth()->guard('sanctum')->user();
            $productId = $request->input('product_id');
            if (is_null($productId)) {
                return response()->json([
                    'status' => 'error',
                    'message' => 'Product ID is required',
                    'data' => [],
                    'code' => 400,
                ]);
            }
            $product = Product::where('product_id', $productId)->first();
            if (!$product) {
                return response()->json([
                    'status' => 'error',
                    'message' => 'Product not found',
                    'data' => [],
                    'code' => 404,
                ]);
            }
            $existingWishlistItem = Wishlist::where('user_id', $user->id)
                ->where('product_id', $productId)
                ->first();

            if ($existingWishlistItem) {
                return response()->json([
                    'status' => 'error',
                    'message' => 'Product is already in your wishlist',
                    'data' => [],
                    'code' => 409,
                ]);
            }
            $wishlistItem = Wishlist::create([
                'user_id' => $user->user_id,
                'product_id' => $product->product_id,
                'product_name' => $product->title,
            ]);

            return response()->json([
                'status' => 'success',
                'message' => 'Product added to wishlist',
                'data' => [
                    'user_id' => $wishlistItem->user_id,
                    'wishlist_id' => $wishlistItem->id,
                    'product_id' => $wishlistItem->product_id,
                    'product_name' => $wishlistItem->product_name,
                ],
                'code' => 201,
            ]);
        } catch (\Exception $e) {
            Log::error('Failed to add product to wishlist', ['error' => $e->getMessage()]);
            return response()->json([
                'status' => 'error',
                'message' => 'Failed to add product to wishlist',
                'data' => [],
                'code' => 500,
            ]);
        }
    }

    private function getFirstProductImage($product)
    {
        $images = json_decode($product->product_image, true);
        return is_array($images) && !empty($images) ? url($images[0]) : null;
    }




    public function search_product(Request $request)
    {
        $searchTerm = $request->input('search_term');
        $sortBy = $request->input('sort_by', 'az');

        try {
            $query = Product::with(['variants', 'category']);



            $query = Product::with(['variants', 'category'])
                ->where('stock', '>', 0);
            if ($searchTerm) {
                $query->where('title', 'LIKE', "%{$searchTerm}%");
            }


            if ($sortBy === 'az') {
                $query->orderBy('title', 'asc');
            } else {
                $query->orderBy('id', 'desc');
            }

            $perPage = $request->input('limit', 8);
            $page = $request->input('page', 1);

            $products = $query->paginate($perPage, ['*'], 'page', $page);

            if ($products->isEmpty()) {
                return response()->json([
                    'status' => 'error',
                    'message' => 'No products found.',
                    'code' => 404,
                ], 404);
            }

            $user = auth()->guard('sanctum')->user();

            $formattedProducts = $products->getCollection()->map(function ($product) use ($user) {
                return [
                    'id' => $product->id,
                    'product_id' => $product->product_id,
                    'category_id' => $product->category_id,
                    'subcategory_id' => $product->subcategory_id,
                    'product_image' => $this->getFirstProductImage($product),
                    'product_video' => $product->product_video ? url($product->product_video) : null,
                    'brand' => $product->brand,
                    'title' => $product->title,
                    'description' => $product->description,
                    'stock' => $product->stock,
                    'unit' => $product->unit,
                    'mrp' => $product->mrp,
                    'sell_price' => $product->sell_price,
                    'gst' => $product->gst,
                    'discount' => $product->discount,
                    'shop_wallet' => $product->shop_wallet,
                    'payble_amount' => $product->payble_amount,
                    'gst_price' => $product->gst_price,
                    'is_active' => $product->is_active,
                    'is_wishlist' => $user ? Wishlist::where('product_id', $product->product_id)
                        ->where('user_id', $user->user_id)
                        ->exists() : false,
                    'category_name' => $product->category->name ?? null,
                    'subcategory_name' => $product->subcategory_name ?? null,
                    'variants' => $product->variants,
                ];
            });

            $paginatedResponse = [
                'status' => 'success',
                'data' => $formattedProducts,
                'meta' => [
                    'current_page' => $products->currentPage(),
                    'last_page' => $products->lastPage(),
                    'per_page' => $products->perPage(),
                    'total' => $products->total(),
                ],
                'message' => $searchTerm ? 'Products found.' : 'All products.',
                'code' => 200,
            ];

            return response()->json($paginatedResponse, 200);

        } catch (\Exception $e) {
            Log::error('Error during product search', ['exception' => $e]);

            return response()->json([
                'status' => 'error',
                'message' => 'An error occurred while searching for products.',
                'error' => $e->getMessage(),
                'code' => 500,
            ], 500);
        }
    }


}