<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;

use App\Models\Branch;

use App\Models\ProductVarient;

use App\Models\StockSummary;

use Illuminate\Http\Request;

use App\Models\Category;

use App\Models\Product;

use App\Models\DeliveryFeeMaster;

use Exception;

use Illuminate\Support\Facades\File;

use Illuminate\Support\Facades\Validator;

use Illuminate\Support\Facades\Log;

use Illuminate\Http\JsonResponse;

use Illuminate\Support\Facades\Session;

use Illuminate\Support\Facades\DB;

class CategoryController extends Controller

{

// public function updateStock(Request $request)

// {

//     $request->validate([

//         'id' => 'required|integer|exists:products,id',

//         'stock' => 'required|integer|min:0',

//     ]);

//     $product = Product::find($request->id);

//     $product->stock = $request->stock;

//     $product->save();

//     return redirect()->back()->with('success', 'Stock updated successfully!');

// }

 public function updateStock(Request $request)

    {

        $request->validate([

            'id' => 'required|integer|exists:products,id',

            'stock' => 'required|integer|min:0',

        ]);

        DB::beginTransaction();

        try {

            $product = Product::lockForUpdate()->find($request->id);

            $old_stock = $product->stock;

            $new_stock = (int) $request->stock;

            if ($old_stock != $new_stock) {

                $quantity_change = abs($new_stock - $old_stock);

                if ($new_stock > $old_stock) {

                    $transaction_type = 'Admin Add';

                } else {

                    $transaction_type = 'Admin Reduce';

                }

                // 1. Log the Admin Stock Adjustment

                StockSummary::create([

                    'product_id' => $product->id, // Use the product ID (primary key)

                    'transaction_type' => $transaction_type,

                    'quantity' => $quantity_change,

                    'order_id' => 'Admin Update',

                    'closing_stock' => $new_stock,

                    'stock' => $old_stock,

                ]);

                // 2. Update the product stock

                $product->stock = $new_stock;

                $product->save();

            }

            DB::commit();

            return redirect()->back()->with('success', 'Stock updated successfully!');

        } catch (\Exception $e) {

            DB::rollBack();

            Log::error("Stock update failed: " . $e->getMessage());

            return redirect()->back()->with('error', 'Failed to update stock: ' . $e->getMessage());

        }

    }

    public function category_delete($id)

    {

        $category = Category::findOrFail($id);

        if ($category) {

            $subcategories = Category::where('link_to', $id)->get();

            foreach ($subcategories as $subcategory) {

                $products = Product::where('subcategory_id', $subcategory->id)->get();

                foreach ($products as $product) {

                    ProductVarient::where('product_id', $product->product_id)->delete();

                    $product->delete();

                }

                $subcategory->delete();

            }

            $products = Product::where('category_id', $id)->get();

            foreach ($products as $product) {

                ProductVarient::where('product_id', $product->product_id)->delete();

                $product->delete();

            }

            $category->delete();

            return redirect()->back()->with('success', 'Category and its related data deleted successfully.');

        }

        return redirect()->back()->with('error', 'Category not found.');

    }

    public function delete_product($id)

    {

        $product = Product::findOrFail($id);

        if ($product) {

            $product->delete();

            return redirect()->back()->with('success', 'Product deleted successfully.');

        }

        return redirect()->back()->with('error', 'Error not found');

    }

    public function uploadTempImages(Request $request)

    {

        $validator = Validator::make($request->all(), [

            'images' => 'array|max:10', // Max 10 images total

            'images.*' => 'image|mimes:jpg,jpeg,png,webp,gif,svg,bmp|max:5120', // 5MB max

        ]);

        if ($validator->fails()) {

            return response()->json(['status' => 'error', 'message' => $validator->errors()->first('images.*')], 422);

        }

        $uploadedPaths = [];

        $tempImages = Session::get('newtemp_images', []);

        foreach ($request->file('images') as $image) {

            // Check current count before uploading

            if (count($tempImages) >= 10) {

                return response()->json(['status' => 'error', 'message' => 'You can upload a maximum of 10 images.'], 400);

            }

            $extension = $image->getClientOriginalExtension();

            $fileName = uniqid('temp_') . '.' . $extension; // Generate unique name

            $path = 'temp_uploads/products'; // Temporary directory

            // Store the image in the public disk

            $image->move(public_path($path), $fileName);

            $fullPath = $path . '/' . $fileName;

            $uploadedPaths[] = $fullPath;

            $tempImages[] = $fullPath; // Add to session array

        }

        Session::put('newtemp_images', $tempImages); // Update session with new paths

        return response()->json(['status' => 'success', 'images' => $uploadedPaths]);

    }

    public function removeTempImage(Request $request)

    {

        $imageSrc = $request->input('image_src');

        if (!preg_match('/^temp_uploads\/products\/temp_[a-zA-Z0-9]+\.(jpg|jpeg|png|webp|gif|svg|bmp)$/', $imageSrc)) {

            return response()->json(['status' => 'error', 'message' => 'Invalid image source.'], 400);

        }

        $tempImages = Session::get('newtemp_images', []);

        $key = array_search($imageSrc, $tempImages);

        if ($key !== false) {

            unset($tempImages[$key]);

            Session::put('newtemp_images', array_values($tempImages));

            $filePath = public_path($imageSrc);

            if (file_exists($filePath)) {

                unlink($filePath);

            }

            return response()->json(['status' => 'success', 'message' => 'Image removed.']);

        }

        return response()->json(['status' => 'error', 'message' => 'Image not found in session.'], 404);

    }

    public function store_product(Request $request)

    {

        try {

            Log::info('--- store_product START ---');

            Log::info('Request Data:', $request->all());

            $validator = Validator::make($request->all(), [

                'category_id'         => 'required|integer',

                'subcategory_id'      => 'integer',

                'brand'               => 'required|string',

                'title'               => 'required|string',

                'description'         => 'nullable|string',

                'stock'               => 'required|integer',

                'youtube_url'         => 'nullable|url',

                'unit'                => 'nullable',

                'mrp'                 => 'required|numeric',

                'purchase_price'      => 'required|numeric',

                'sell_price'          => 'required|numeric|min:0',

                'gst'                 => 'required|in:none,0,5,12,18',

                'discount'            => 'nullable|numeric',

                'shop_wallet'         => 'nullable|numeric',

                'payable_amount'      => 'nullable|numeric|min:0',

                'variants.*.name'     => 'nullable|string',

                'variants.*.mrp'      => 'nullable|numeric|min:0',

                'variants.*.discount' => 'nullable|numeric|min:0',

                'variants.*.price'    => 'nullable|numeric|min:0',

                'item_code'           => 'required|numeric|unique:products,item_code',

                'hsn_code'            => 'nullable',

                'barcode1'           => 'nullable|unique:products,barcode1',

                'barcode2'            => 'nullable|unique:products,barcode2',

                'is_note'             => 'nullable|boolean',

                'branch_id'           => 'nullable|string|exists:branches,id',

                    'role' => 'nullable|string',

                //  'user_role' => 'required',

            ]);

$userRole = $request->role ?? auth()->user()->role;

            $tempImages = Session::get('newtemp_images', []);

            Log::info('Temp Images from Session:', $tempImages);

            if (empty($tempImages)) {

                Log::warning('No temp images found.');

                $validator->errors()->add('images', 'At least one image is required.');

            } elseif (count($tempImages) > 10) {

                Log::warning('More than 10 temp images.');

                $validator->errors()->add('images', 'You can upload up to 10 images.');

            }

            if ($validator->fails()) {

                Log::error('Validation failed:', $validator->errors()->toArray());

                return redirect()->back()->withErrors($validator)->withInput();

            }

            $category = Category::find($request->category_id);

            Log::info('Fetched Category:', $category ? $category->toArray() : []);

            if (!$category) {

                Log::error('Category not found.');

                return redirect()->back()->with('error', 'Selected category not found.')->withInput();

            }

            $categoryId   = $category->id;

            $categoryName = $category->category_name;

            $subcategoryName = null;

            if ($request->filled('subcategory_id')) {

                $subcategory = Category::find($request->subcategory_id);

                Log::info('Fetched Subcategory:', $subcategory ? $subcategory->toArray() : []);

                if (!$subcategory || $subcategory->link_to != $category->id) {

                    Log::error('Subcategory invalid or not linked.');

                    return redirect()->back()->with('error', 'Selected subcategory is invalid or does not belong to the selected category.')->withInput();

                }

                $subcategoryName = $subcategory->subcategory_name ?? $subcategory->category_name;

            }

            $imagePaths = [];

            $destinationDir = public_path('assets/images/products');

            Log::info('Destination Dir: ' . $destinationDir);

            if (!file_exists($destinationDir)) {

                mkdir($destinationDir, 0777, true);

                Log::info('Created directory: ' . $destinationDir);

            }

            foreach ($tempImages as $tempImage) {

                $sourcePath = public_path($tempImage);

                Log::info('Processing temp image: ' . $tempImage);

                if (file_exists($sourcePath)) {

                    $fileName = basename($tempImage);

                    $newPath  = 'assets/images/products/' . $fileName;

                    rename($sourcePath, $destinationDir . '/' . $fileName);

                    $imagePaths[] = $newPath;

                    Log::info('Moved image to: ' . $newPath);

                }

            }

            Session::forget('newtemp_images');

            Log::info('Temp images cleared from session.');

            $proid = (int) $request->item_code;

            Log::info('Generated Product ID: ' . $proid);

            if (Product::where('product_id', $proid)->exists()) {

                Log::error('Product ID already exists.');

                Session::put('newtemp_images', $imagePaths);

                return redirect()->back()->with('error', 'The item code is already used as a product ID. Please enter a unique item code.')->withInput();

            }

            $gstInput = $request->gst;

            $gst      = ($gstInput === 'none') ? 0 : (float) $gstInput;

            $gstPrice = ($request->sell_price * $gst) / 100;

            $cgst     = $gstPrice / 2;

            $sgst     = $gstPrice / 2;

            Log::info('GST:', ['gst' => $gst, 'gstPrice' => $gstPrice, 'cgst' => $cgst, 'sgst' => $sgst]);

            $branch = Branch::where('id', $request->branch_id)->first();

            Log::info('Fetched Branch:', $branch ? $branch->toArray() : []);

            if (!$branch) {

                Log::error('Branch not found.');

                return redirect()->back()->with('error', 'Branch not found!')->withInput();

            }

            $product = Product::create([

                'product_id'       => $proid,

                'category_id'      => $categoryId,

                'subcategory_id'   => $request->subcategory_id,

                'brand'            => $request->brand,

                'title'            => $request->title,

                'purchase_price'   => $request->purchase_price,

                'description'      => $request->description,

                'stock'            => $request->stock,

                'youtube_url'      => $request->youtube_url,

                'unit'             => $request->unit,

                'mrp'              => $request->mrp,

                'sell_price'       => $request->sell_price,

                'gst'              => $gst,

                'discount'         => $request->discount,

                'shop_wallet'      => $request->shop_wallet,

                'is_active'        => 1,

                'product_image'    => json_encode($imagePaths),

                'category_name'    => $categoryName,

                'subcategory_name' => $subcategoryName,

                'gst_price'        => $gstPrice,

                'cgst'             => $cgst,

                'sgst'             => $sgst,

                'payble_amount'    => $request->payable_amount,

                'barcode1'         => $request->barcode1,

                'barcode2'         => $request->barcode2,

                'is_gst'           => $request->boolean('is_gst'),

                'item_code'        => $request->item_code,

                'hsn_code'         => $request->hsn_code,

                'is_note'          => $request->boolean('is_note'),

                'branch_id'        => $branch->id,

                'branch_name'      => $branch->branch_name,

                'user_role'        => $userRole,

            ]);

            Log::info('Product created:', $product->toArray());

            if ($request->has('variants') && is_array($request->variants)) {

                foreach ($request->variants as $variantData) {

                    Log::info('Processing Variant:', $variantData);

                    if (!empty($variantData['name']) || !empty($variantData['mrp']) || !empty($variantData['discount']) || !empty($variantData['price'])) {

                        do {

                            $variantId = 'VAR' . rand(10000, 99999);

                        } while (ProductVarient::where('variant_id', $variantId)->exists());

                        $variantGstPrice = 0;

                        $variantCgst = 0;

                        $variantSgst = 0;

                        if (isset($variantData['price'])) {

                            $variantGstPrice = ($variantData['price'] * $gst) / 100;

                            $variantCgst = $variantGstPrice / 2;

                            $variantSgst = $variantGstPrice / 2;

                        }

                        $variant = ProductVarient::create([

                            'product_id'   => $proid,

                            'variant_id'   => $variantId,

                            'variant_name' => $variantData['name'] ?? null,

                            'mrp'          => $variantData['mrp'] ?? null,

                            'discount'     => $variantData['discount'] ?? null,

                            'sell_price'   => $variantData['price'] ?? null,

                            'gst_price'    => $variantGstPrice,

                            'cgst'         => $variantCgst,

                            'sgst'         => $variantSgst,

                        ]);

                        Log::info('Variant created:', $variant->toArray());

                    }

                }

            }

            Log::info('--- store_product END SUCCESS ---');

            return redirect()->route('products')->with('success', 'Product added successfully!');

        } catch (\Exception $e) {

            Log::error('Exception occurred:', ['message' => $e->getMessage(), 'trace' => $e->getTraceAsString()]);

            Session::put('newtemp_images', Session::get('newtemp_images', []));

            return redirect()->back()->with('error', 'An error occurred while adding the product. Please try again.')->withInput();

        }

    }

    public function toggleNote(Request $request)

    {

        $request->validate([

            'product_id' => 'required|exists:products,id',

            'is_note' => 'required|boolean',

        ]);

        $product = Product::find($request->product_id);

        $product->is_note = $request->is_note;

        $product->save();

        return response()->json(['status' => 'success', 'message' => 'Note status updated.']);

    }

    public function variant_destroy($id)

    {

        try {

            $variant = ProductVarient::find($id);

            if (!$variant) {

                return response()->json(['success' => false, 'message' => 'Variant not found']);

            }

            $variant->delete();

            return response()->json(['success' => true, 'message' => 'Variant deleted successfully']);

        } catch (\Exception $e) {

            return response()->json(['success' => false, 'message' => 'Error deleting variant']);

        }

    }

    public function getSubcategories(Request $request)

    {

        $categoryId = $request->input('categoryId');

        if (empty($categoryId)) {

            return response()->json(['errormessage' => 'No categoryId provided'], 400);

        }

        $subcategories = Category::where('link_to', $categoryId)->get();

        return response()->json(['subcategories' => $subcategories]);

    }

    public function update_product(Request $request, $id)

    {

        try {

            $product = Product::findOrFail($id);

            $validator = Validator::make($request->all(), [

                'category_id'      => 'required|integer',

                'subcategory_id'   => 'nullable|integer',

                'images'           => 'nullable|array',

                'images.*'         => 'mimes:jpg,jpeg,png,webp,gif,svg,bmp|max:5120',

                'remove_images'    => 'nullable|array',

                'remove_images.*'  => 'string',

                'product_video'    => 'nullable|file|mimes:mp4,mov,avi|max:5120',

                'brand'            => 'required|string',

                'title'            => 'required|string',

                'description'      => 'nullable|string',

                'stock'            => 'required|integer',

                'youtube_url'      => 'nullable|url',

                'unit'             => 'nullable',

                'purchase_price'   => 'required|numeric',

                'mrp'              => 'required|numeric',

                'sell_price'       => 'required|numeric|min:0|lte:mrp',

                // 'gst'              => 'required|numeric',

                'gst'                 => 'required|in:none,0,5,12,18',

                'discount'         => 'nullable|numeric',

                'shop_wallet'      => 'nullable|numeric',

                'payable_amount'   => 'nullable|numeric|min:0',

                'variants.*.name'  => 'nullable|string',

                'variants.*.price' => 'nullable|numeric|min:0|lte:mrp',

                'item_code'        => 'nullable|string|max:255',

                'hsn_code'         => 'nullable|string|max:255',

                'is_note'          => 'nullable|boolean',

                'branch_id'        => 'nullable|integer|exists:branches,id',

                'barcode1'  => 'nullable',

                'barcode2'  => 'nullable',

            ]);

            if ($validator->fails()) {

                return redirect()->back()->with('error', implode(', ', $validator->errors()->all()))->withInput();

            }

            $itemCode = $request->item_code;

            // Optionally ensure item_code is numeric if product_id is integer

            if (!is_numeric($itemCode)) {

                return redirect()->back()->with('error', 'Item code must be numeric to match product ID.');

            }

            $category = Category::find($request->category_id);

            if (!$category) {

                return redirect()->back()->with('error', 'Selected category not found.');

            }

            $categoryName = $category->category_name;

            $subcategoryName = null;

            if ($request->filled('subcategory_id')) {

                $subcategory = Category::find($request->subcategory_id);

                if (!$subcategory || $subcategory->link_to != $category->id) {

                    return redirect()->back()->with('error', 'Selected subcategory is invalid or does not belong to the selected category.');

                }

                $subcategoryName = $subcategory->subcategory_name;

            }

            $existingImages = json_decode($product->product_image, true) ?? [];

            if ($request->has('remove_images')) {

                foreach ($request->remove_images as $removeImage) {

                    $imagePath = public_path($removeImage);

                    if (file_exists($imagePath)) {

                        unlink($imagePath);

                    }

                    $existingImages = array_filter($existingImages, fn($img) => basename($img) !== $removeImage);

                }

            }

            if ($request->hasFile('images')) {

                foreach ($request->file('images') as $image) {

                    if (count($existingImages) >= 4)

                        break;

                    $fileName = time() . '_' . $image->getClientOriginalName();

                    $destinationPath = public_path('assets/images/products');

                    $image->move($destinationPath, $fileName);

                    $existingImages[] = 'assets/images/products/' . $fileName;

                }

            }

            $videoPath = $product->product_video;

            if ($request->hasFile('product_video')) {

                $fileName = time() . '_' . $request->file('product_video')->getClientOriginalName();

                $destinationPath = public_path('assets/images/products');

                $request->file('product_video')->move($destinationPath, $fileName);

                $videoPath = 'assets/images/products/' . $fileName;

            }

            // $gst = $request->gst;

            // $gstPrice = ($request->sell_price * $gst) / 100;

            // $cgst = $gstPrice / 2;

            // $sgst = $gstPrice / 2;

 $gstInput = $request->gst;

            $gst      = ($gstInput === 'none') ? 0 : (float) $gstInput;

            $gstPrice = ($request->sell_price * $gst) / 100;

            $cgst     = $gstPrice / 2;

            $sgst     = $gstPrice / 2;

            $branch = null;

            if ($request->filled('branch_id')) {

                $branch = Branch::where('id', $request->branch_id)->first();

                if (!$branch) {

                    return redirect()->back()->with('error', 'Branch not found!')->withInput();

                }

            }

            // Update product with item_code and product_id same

            $userRole = $request->role ?? auth()->user()->role;

            $product->update([

                'product_id'       => $itemCode,

                'category_id'      => $request->category_id,

                'subcategory_id'   => $request->subcategory_id,

                'brand'            => $request->brand,

                'title'            => $request->title,

                'description'      => $request->description,

                'stock'            => $request->stock,

                'youtube_url'      => $request->youtube_url,

                'unit'             => $request->unit,

                'mrp'              => $request->mrp,

                'sell_price'       => $request->sell_price,

                'purchase_price'   => $request->purchase_price,

                'gst'              => $gst,

                'discount'         => $request->discount,

                'shop_wallet'      => $request->shop_wallet,

                'product_image'    => json_encode(array_values($existingImages)),

                'product_video'    => $videoPath,

                'category_name'    => $categoryName,

                'subcategory_name' => $subcategoryName,

                'gst_price'        => $gstPrice,

                'cgst'             => $cgst,

                'sgst'             => $sgst,

                'payble_amount'    => $request->payable_amount,

                'is_gst'           => $request->is_gst,

                'item_code'        => $itemCode,

                'hsn_code'         => $request->hsn_code,

                'is_note'          => $request->boolean('is_note'),

                'branch_id'        => $branch ? $branch->id : null,

                'branch_name'      => $branch ? $branch->branch_name : null,

                'barcode1'         => $request->barcode1,

                'barcode2'         => $request->barcode2,

                'user_role'        => $userRole,

            ]);

            // Delete old variants

            ProductVarient::where('product_id', $id)->delete();

            if ($request->has('variants') && is_array($request->variants)) {

                foreach ($request->variants as $variantData) {

                    if (empty($variantData['name']))

                        continue;

                    do {

                        $variantId = 'VAR' . rand(10000, 99999);

                    } while (ProductVarient::where('variant_id', $variantId)->exists());

                    $variantGstPrice = ($variantData['price'] * $gst) / 100;

                    $variantCgst = $variantGstPrice / 2;

                    $variantSgst = $variantGstPrice / 2;

                    ProductVarient::create([

                        'product_id'   => $itemCode,                      

                        'variant_id'   => $variantId,

                        'variant_name' => $variantData['name'],

                        'mrp'          => $variantData['mrp'] ?? 0,

                        'discount'     => $variantData['discount'] ?? 0,

                        'sell_price'   => $variantData['price'],

                        'gst_price'    => $variantGstPrice,

                        'cgst'         => $variantCgst,

                        'sgst'         => $variantSgst,

                    ]);

                }

            }

            return redirect()->route('products')->with('success', 'Product updated successfully!');

        } catch (\Exception $e) {

            return redirect()->back()->with('error', 'An error occurred while updating the product.');

        }

    }

    public function removeImage(Request $request)

    {

        $request->validate([

            'image' => 'required|string',

            'product_id' => 'required|exists:products,id'

        ]);

        $product = Product::findOrFail($request->product_id);

        $images = json_decode($product->product_image, true) ?? [];

        $imagePath = "assets/images/products/{$request->image}";

        $updatedImages = array_values(array_filter($images, function ($img) use ($imagePath) {

            return $img !== $imagePath;

        }));

        $product->update(['product_image' => empty($updatedImages) ? json_encode([]) : json_encode($updatedImages)]);

        $fullImagePath = public_path($imagePath);

        if (File::exists($fullImagePath)) {

            File::delete($fullImagePath);

        } else {

        }

        return response()->json([

            'success' => true,

            'remaining_images' => $updatedImages

        ]);

    }

    public function add_category_post(Request $request)

    {

        $rules = [

            'category_img' => 'required|image|mimes:jpeg,png,jpg,gif|max:2048',

            'category_type' => 'required|in:1,2',

        ];

        if ($request->category_type == 1) {

            $rules['category_name'] = 'required|string|max:255';

        } elseif ($request->category_type == 2) {

            $rules['subcategory_name'] = 'required|string|max:255';

            $rules['link_to'] = 'required|exists:categories,id';

        }

        $messages = [

            'category_img.required' => 'Please choose an image.',

            'category_name.required' => 'The category name field is required.',

            'subcategory_name.required' => 'The subcategory name field is required when creating a subcategory.',

            'link_to.required' => 'Please select a category to link the subcategory.',

        ];

        $validator = Validator::make($request->all(), $rules, $messages);

        if ($validator->fails()) {

            return redirect()->back()->withErrors($validator)->withInput();

        }

        $duplicate = Category::where(function ($query) use ($request) {

            if ($request->category_type == 1) {

                $query->where('category_name', $request->category_name);

            } elseif ($request->category_type == 2) {

                $query->where('subcategory_name', $request->subcategory_name)

                    ->where('link_to', $request->link_to);

            }

        })->exists();

        if ($duplicate) {

            return redirect()->back()->with('error', 'Combination already exists')->withInput();

        }

        $imagePath = null;

        if ($request->hasFile('category_img')) {

            $fileName = time() . '_' . $request->file('category_img')->getClientOriginalName();

            $destinationPath = public_path('assets/images/products');

            $request->file('category_img')->move($destinationPath, $fileName);

            $imagePath = 'assets/images/products/' . $fileName;

        }

        do {

            $catid = 'CAT' . rand(1000, 9999);

        } while (Category::where('cat_id', $catid)->exists());

        Category::create([

            'category_name' => $request->category_type == 1 ? $request->category_name : null,

            'subcategory_name' => $request->category_type == 2 ? $request->subcategory_name : null,

            'link_to' => $request->category_type == 2 ? $request->link_to : null,

            'category_img' => $imagePath,

            'category_type' => $request->category_type,

            'cat_id' => $catid,

        ]);

        return redirect()->route('category')->with('success', 'Category added successfully.');

    }

    public function categories_update(Request $request, $id)

    {

        $rules = [

            'category_type' => 'required|in:1,2',

            'category_img' => 'nullable|image|mimes:jpeg,png,jpg,gif,svg|max:2048',

        ];

        if ($request->category_type == 1) { // If it's a main category

            $rules['category_name'] = 'required|string|max:255';

        } elseif ($request->category_type == 2) { // If it's a subcategory

            $rules['subcategory_name'] = 'required|string|max:255';

            $rules['link_to'] = 'required|exists:categories,id';

        }

        $messages = [

            'category_name.required' => 'The category name is required for categories.',

            'subcategory_name.required' => 'Subcategory name is required when creating a subcategory.',

            'link_to.required' => 'Please select a category to link the subcategory.',

            'link_to.exists' => 'The selected category to link to does not exist.',

        ];

        $validator = Validator::make($request->all(), $rules, $messages);

        if ($validator->fails()) {

            return redirect()->back()->withErrors($validator)->withInput();

        }

        $category = Category::findOrFail($id);

        // Handle Image Upload

        if ($request->hasFile('category_img')) {

            // Delete old image if it exists and is not the default placeholder

            if ($category->category_img && File::exists(public_path($category->category_img))) {

                File::delete(public_path($category->category_img));

            }

            $fileName = time() . '_' . $request->file('category_img')->getClientOriginalName();

            $path = public_path('assets/images/products');

            $request->file('category_img')->move($path, $fileName);

            $category->category_img = 'assets/images/products/' . $fileName;

        }

        // Update Category fields based on type

        $category->category_type = $request->category_type;

        if ($request->category_type == 1) {

            $category->category_name = $request->category_name;

            $category->subcategory_name = null; // Clear subcategory name if changing to category

            $category->link_to = null; // Clear link_to if changing to category

        } elseif ($request->category_type == 2) {

            $category->subcategory_name = $request->subcategory_name;

            $category->link_to = $request->link_to;

            $category->category_name = null; // Clear category name if changing to subcategory

        }

        $category->save();

        return redirect()->route('category')->with('success', 'Category updated successfully.');

    }

    public function getProductDetails($product_id)

    {

        $product = Product::with('variants')->where('product_id', $product_id)->first();

        if (!$product) {

            return response()->json(['message' => 'Product not found'], 404);

        }

        return response()->json([

            'title' => $product->title,

            'sell_price' => $product->sell_price,

            'variants' => $product->variants->map(function ($variant) {

                return [

                    'id' => $variant->id,

                    'variant_name' => $variant->variant_name,

                    'sell_price' => $variant->sell_price

                ];

            }),

        ]);

    }

}

