<?php
namespace App\Http\Controllers\Admin;
use App\Http\Controllers\Controller;
use App\Models\Branch;
use App\Models\Employee;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Facades\Hash;
use Illuminate\Support\Facades\Auth;
class EmployeeController extends Controller
{
    public function employee_store(Request $request)
    {
        $request->validate([
            'designation' => 'required|string|max:255',
            'name' => 'required|string|max:255',
            'phone' => 'required|digits:10',
            'address' => 'required|string',
            'email' => 'required|max:255',
            'password' => 'required|string|min:6',
        ]);
        $user = auth()->user();
        try {
            if ($user->role === 'admin') {
                $request->validate([
                    'branch_id' => 'required|exists:branches,id',
                ]);
                $branch = Branch::find($request->branch_id);
                Log::info("Admin selected branch ID: {$request->branch_id}");
            } else {
                $roleToBranchName = [
                    'sakthi' => 'Sakthi Super Mart',
                    'smart_world' => 'SMARTWORLD',
                    'chendur_traders' => 'CHENDUR TRADERS',
                    'yoga_naturals' => 'Yogaa Naturals',
                ];
                $branchName = $roleToBranchName[$user->role] ?? null;
                if (!$branchName) {
                    return redirect()->back()->with('error', 'Invalid role or branch not mapped.');
                }
                $branch = Branch::where('branch_name', $branchName)->first();
                if (!$branch) {
                    return redirect()->back()->with('error', 'Branch not found for your role.');
                }
            }
            $employee = Employee::create([
                'branch_id' => $branch->id,
                'branch' => $branch->branch_name,
                'role' => $branch->role, // ← Here is the key change
                'designation' => $request->designation,
                'name' => $request->name,
                'phone' => $request->phone,
                'address' => $request->address,
                'email' => $request->email,
'password_hint' => $request->password,
                'password' => Hash::make($request->password),
            ]);
            return redirect()->route('add_employee')->with('success', 'Employee added successfully');
        } catch (\Exception $e) {
            return redirect()->back()->with('error', 'Something went wrong. Please try again.');
        }
    }
    public function employee_update(Request $request, $id): mixed
{
    $request->validate([
        'designation' => 'required|string|max:255',
        'name'        => 'required|string|max:255',
        'phone'       => 'required|digits:10',
        'address'     => 'required|string',
        'email'       => 'nullable|email|max:255|unique:employees,email,' . $id, 
        'password'    => 'nullable|string|min:6', 
    ]);
    $employee = Employee::findOrFail($id);
    $employee->designation = $request->designation;
    $employee->name        = $request->name;
    $employee->phone       = $request->phone;
    $employee->address     = $request->address;
      $employee->password_hint       = $request->password;
    if ($request->email) {
        $employee->email = $request->email;
    }
    if ($request->filled('password')) {
        $employee->password = Hash::make($request->password); 
    }
    $employee->save();
   return redirect()->route('add_employee')
                 ->with('success', 'Employee updated successfully.');
}

    public function employee_destroy($id)
    {
        Employee::findOrFail($id)->delete();
        return redirect()->route('add_employee')->with('success', 'Employee deleted successfully.');
    }
    public function employee_delete($id)
    {
        $employee = Employee::find($id);
        $employee->delete();
        return redirect()->back()->with('success', 'Employee deleted successfully.');
    }
    public function edit_employee($id)
    {
        $employee = Employee::findOrFail($id);
        $branches = Branch::all();
        return view('admin.edit_employee', compact('employee', 'branches'));
    }
    public function employee_login()
    {
        return view('admin.employee_login');
    }
    public function employee_login_post(Request $request)
    {
        $request->validate([
            'email' => 'required|email',
            'password' => 'required|string',
        ]);
        $employee = Employee::where('email', $request->email)->first();
        if (!$employee || !Hash::check($request->password, $employee->password)) {
            return back()->with('error', 'Invalid email or password')->withInput();
        }
        Auth::guard('employee')->login($employee);
        switch ($employee->role) {
            case 'sakthi':
                return redirect()->route('dashboard');
            case 'smart_world':
                return redirect()->route('dashboard');
            case 'smart_world_gift':
                return redirect()->route('dashboard');
            case 'yoga_naturals':
                return redirect()->route('dashboard');
            default:
                return redirect()->route('dashboard');
        }
    }
}
