<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use App\Models\Address;
use App\Models\ContactUs;
use App\Models\IncomeWallet;
use App\Models\CashWalletIncome;
use App\Models\Product;
use App\Models\ProductVarient;
use App\Models\shopWalletIncome;
use App\Models\User;
use App\Models\OfflineSale;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Log;
use Illuminate\Http\Request;

class OfflinesaleController extends Controller
{

public function getProductDetails($identifier)
    {
        $product = Product::with('variants')->where('product_id', $identifier)->first();

        if (!$product) {
            $product = Product::with('variants')->where('HSN_code', $identifier)->first();
        }

        if (!$product) {
            return response()->json(['message' => 'Product not found'], 404);
        }

        return response()->json([
            'product_id'  => $product->product_id,
            'title'       => $product->title,
            'sell_price'  => (float) $product->sell_price,
            'shop_wallet' => (float) $product->shop_wallet,
            'variants'    => $product->variants->map(function ($variant) {
                return [
                    'id'           => $variant->id,
                    'variant_name' => $variant->variant_name,
                    'sell_price'   => (float) $variant->sell_price,
                ];
            }),
        ]);
    }

public function getUserDetailsAndWallet($userId)
    {
        $user = User::where('user_id', $userId)->first();

        if (!$user) {
            return response()->json(['message' => 'User not found.'], 404);
        }
        $address = $user->currentAddress;
        $wallet  = $user->incomeWallet;

        return response()->json([
            'user' => [
                'user_id'      => $user->user_id,
                'name'         => $user->name,
                'phone_number' => $user->phone_number ?? 'N/A',
                'address'      => $address ? [
                    'house_no'  => $address->house_no,
                    'road_name' => $address->road_name,
                    'landmark'  => $address->landmark,
                    'district'  => $address->district,
                    'state'     => $address->state,
                    'pin'       => $address->pin,
                ] : null
            ],
            'wallet' => [
                'cash_wallet'     => $wallet ? (float) $wallet->cash_wallet : 0.00,
                'shopping_wallet' => $wallet ? (float) $wallet->shopping_wallet : 0.00,
            ]
        ]);
    }


public function getUserDetailsAndWalletByPhone($phoneNumber)
    {
        $user = User::where('phone_number', $phoneNumber)->first();

        if (!$user) {
            return response()->json(['message' => 'User with this phone number not found.'], 404);
        }

        $address = $user->currentAddress;
        $wallet  = $user->incomeWallet;
        return response()->json([
            'user' => [
                'user_id'      => $user->user_id,
                'name'         => $user->name,
                'phone_number' => $user->phone_number ?? 'N/A',
                'address'      => $address ? [
                    'house_no'  => $address->house_no,
                    'road_name' => $address->road_name,
                    'landmark'  => $address->landmark,
                    'district'  => $address->district,
                    'state'     => $address->state,
                    'pin'       => $address->pin,
                ] : null
            ],
            'wallet' => [
                'cash_wallet'     => $wallet ? (float) $wallet->cash_wallet : 0.00,
                'shopping_wallet' => $wallet ? (float) $wallet->shopping_wallet : 0.00,
            ]
        ]);
    }

public function generateInvoice($orderId)
    {
        Log::info("Generating invoice for Order ID: $orderId");

        try {
            $offlineSale = OfflineSale::where('order_id', $orderId)
                ->firstOrFail();

            $saleItems = json_decode($offlineSale->sale_items, true);
            $customerDetails = json_decode($offlineSale->customer_details, true) ?? [];

            foreach ($saleItems as &$item) {
                $product = Product::where('product_id', $item['product_id'])->first();
                $item['full_product_name'] = $product ? $product->title : ($item['product_title'] ?? 'N/A');

                if (isset($item['variant_id']) && $item['variant_id']) {
                    $variant = ProductVarient::find($item['variant_id']);
                    $item['full_variant_name'] = $variant ? $variant->variant_name : ($item['variant_name'] ?? 'N/A');
                } else {
                    $item['full_variant_name'] = 'N/A';
                }
            }
            unset($item); 
            $user = User::where('user_id', $offlineSale->user_id)->first();

            $address = null;
            if ($user) {
                $address = Address::where('userid', $user->user_id)
                    ->where('is_current', 1)
                    ->first();
            }

            if ($address) {
                $customerDetails['address'] = [
                    'house_no'  => $address->house_no ?? '',
                    'road_name' => $address->road_name ?? '',
                    'landmark'  => $address->landmark ?? '',
                    'district'  => $address->district ?? '',
                    'state'     => $address->state ?? '',
                    'pin'       => $address->pin ?? '',
                ];
            } else {
                $customerDetails['address'] = [
                    'house_no'  => '',
                    'road_name' => '',
                    'landmark'  => '',
                    'district'  => '',
                    'state'     => '',
                    'pin'       => '',
                ];
            }

            $contact = ContactUs::where('is_active', 1)->first();

              $is_gst = User::where('role', 'admin')->where('is_gst', 1)->exists();
              $gst_rate = 18;

            $data = [
                'order_id'                  => $offlineSale->order_id,
                'name'                      => $user->name ?? 'N/A',
                'phone_number'              => $user->phone_number ?? 'N/A',
                'sale_id'                   => $offlineSale->id,
                'sale_date'                 => $offlineSale->created_at->format('Y-m-d H:i:s'),
                'customer'                  => $customerDetails,
                'items'                     => $saleItems,
                'total_amount'              => $offlineSale->total_amount,
                'cash_wallet_deduction'     => $offlineSale->cash_wallet_deduction,
                'shopping_wallet_deduction' => $offlineSale->shopping_wallet_deduction,
                'discount'                  => $offlineSale->discount,
                'amount_payable'            => $offlineSale->amount_payable,
                'payment_mode'              => $offlineSale->payment_mode,
                'customer_name'             => $user->name ?? 'ADBAZZAR Customer',
                'company_name'              => $contact->company_name ?? 'ADBAZZAR',
                'company_address'           => $contact->company_address ?? 'N/A',
                'company_phone'             => $contact->phone_number ?? 'N/A',
                'company_email'             => $contact->email ?? 'N/A',
                'gst_rate'                  => $gst_rate, // Pass the GST rate
                'is_gst'                   => $is_gst, // Pass the user's GST status
            ];

            return view('admin.offline_sale_invoice', $data);

        } catch (\Exception $e) {
            Log::error('Error generating invoice: ' . $e->getMessage(), [
                'order_id' => $orderId,
                'trace' => $e->getTraceAsString()
            ]);
            return response()->json(['message' => 'Failed to generate invoice. The order might not exist or there was an internal error.'], 500);
        }
    }

public function place_offlineorder(Request $request)
{
    Log::info('--- place_offlineorder called ---');

    try {
        Log::info('Step 1: Validating request inputs...');
        $request->validate([
            'user_id'              => 'required|exists:users,user_id',
            'items'                => 'required|array|min:1',
            'items.*.product_id'   => 'required|string',
            'items.*.variant_id'   => 'nullable|exists:product_varients,id',
            'items.*.sell_price'   => 'required|numeric|min:0',
            'items.*.quantity'     => 'required|integer|min:1',
            'items.*.amount'       => 'required|numeric|min:0',
            'items.*.description'  => 'nullable|string',
            'total_amount'         => 'required|numeric|min:0',
            'cash_wallet_used'     => 'nullable|numeric|min:0',
            'shopping_wallet_used' => 'nullable|numeric|min:0',
            'discount'             => 'nullable|numeric|min:0',
            'amount_payable'       => 'required|numeric|min:0',
            'payment_mode'         => 'required|string|in:cash,card,upi',
            'admin_role'           => 'required',
        ]);
        Log::info('Step 2: Validation passed.');

        $userId             = strtoupper($request->input('user_id'));
        $cashWalletUsed     = (float) $request->input('cash_wallet_used', 0);
        $shoppingWalletUsed = (float) $request->input('shopping_wallet_used', 0);
        $totalAmount        = (float) $request->input('total_amount');
        $amountPayable      = (float) $request->input('amount_payable');
        $rawSaleItems       = $request->input('items');
        $paymentMode        = $request->input('payment_mode');
        $admin_role          = $request->input('admin_role');

        // Build cleaned saleItems with description fallback
        $saleItems = array_map(function ($item) {
            return [
                'product_id'  => $item['product_id'],
                'variant_id'  => $item['variant_id'] ?? null,
                'sell_price'  => $item['sell_price'],
                'quantity'    => $item['quantity'],
                'amount'      => $item['amount'],
                'description' => $item['description'] ?? ''
            ];
        }, $rawSaleItems);

        Log::info('Step 3: Generating order ID...');
        $lastOfflineSale = OfflineSale::latest('id')->first();
        $newOrderIdNumber = 101;
        if ($lastOfflineSale && preg_match('/^store(\d+)$/', $lastOfflineSale->order_id, $matches)) {
            $newOrderIdNumber = (int) $matches[1] + 1;
        }
        $orderId = 'store' . $newOrderIdNumber;

        Log::info('Step 3: Parsed request inputs.', compact(
            'userId', 'cashWalletUsed', 'shoppingWalletUsed',
            'totalAmount', 'amountPayable', 'saleItems', 'orderId', 'paymentMode'
        ));

        Log::info("Step 4: Fetching user wallet with locking (user_id: $userId)...");
        $userWallet = IncomeWallet::where('user_id', $userId)->lockForUpdate()->first();

        if (($cashWalletUsed > 0 || $shoppingWalletUsed > 0) && !$userWallet) {
            Log::warning("Wallet not found but required for deduction (user_id: $userId)");
            return response()->json(['message' => 'User wallet not found. Cannot deduct wallet amount.'], 400);
        }

        Log::info('Step 5: User wallet check passed.');

        if ($userWallet) {
            Log::info('Step 7: Verifying wallet balances...');
            if ($cashWalletUsed > $userWallet->cash_wallet) {
                return response()->json(['message' => 'Insufficient cash wallet balance.'], 400);
            }
            if ($shoppingWalletUsed > $userWallet->shopping_wallet) {
                return response()->json(['message' => 'Insufficient shopping wallet balance.'], 400);
            }
        }

        $discount        = (float) $request->input('discount', 0);
        $calculatedTotal = $cashWalletUsed + $shoppingWalletUsed + $amountPayable + $discount;

        Log::info("Step 8: Validating amount consistency", [
            'calculated_total' => $calculatedTotal,
            'provided_total' => $totalAmount
        ]);
        if (abs($calculatedTotal - $totalAmount) > 0.01) {
            return response()->json(['message' => 'Order amount calculation mismatch. Please try again.'], 400);
        }

        Log::info('Step 8.5: Checking stock availability...');
        foreach ($saleItems as $item) {
            $product = Product::where('product_id', $item['product_id'])->first();
            if (!$product) {
                return response()->json(['message' => "Product '{$item['product_id']}' not found."], 400);
            }
            if (!empty($item['variant_id'])) {
                $variant = ProductVariant::find($item['variant_id']);
                if (!$variant || $variant->stock < $item['quantity']) {
                    return response()->json(['message' => "Insufficient stock for product variant: {$product->name}"], 400);
                }
            } else {
                if ($product->stock < $item['quantity']) {
                    return response()->json(['message' => "Insufficient stock for product: {$product->name}"], 400);
                }
            }
        }

        DB::beginTransaction();
        Log::info('Step 9: Database transaction started.');

        try {
            Log::info('Step 10: Creating offline sale record...');
            $offlineSale = OfflineSale::create([
                'user_id'                   => $userId,
                'order_id'                  => $orderId,
                'sale_items'                => json_encode($saleItems),
                'total_amount'              => $totalAmount,
                'cash_wallet_deduction'     => $cashWalletUsed,
                'shopping_wallet_deduction' => $shoppingWalletUsed,
                'discount'                  => $discount,
                'amount_payable'            => $amountPayable,
                'payment_mode'              => $paymentMode,
                'status'                    => 'completed',
                'admin_role'                => $admin_role,
            ]);

            foreach ($saleItems as $item) {
                $product = Product::where('product_id', $item['product_id'])->first();
                if (!empty($item['variant_id'])) {
                    ProductVariant::find($item['variant_id'])->decrement('stock', $item['quantity']);
                } else {
                    $product->decrement('stock', $item['quantity']);
                }
            }

            if ($shoppingWalletUsed > 0) {
                ShopWalletIncome::create([
                    'user_id'       => $userId,
                    'membership_id' => null,
                    'amount'        => $shoppingWalletUsed,
                    'remark'        => "Store_purchase : $orderId",
                    'description'   => "Store_purchase",
                    'payment_mode'  => "wallet",
                    'type'          => "debit"
                ]);
            }

            if ($cashWalletUsed > 0) {
                CashWalletIncome::create([
                    'user_id'       => $userId,
                    'membership_id' => null,
                    'amount'        => $cashWalletUsed,
                    'remark'        => "Store_purchase : $orderId",
                    'description'   => "Store_purchase",
                    'payment_mode'  => "wallet",
                    'type'          => "debit"
                ]);
            }

            if ($cashWalletUsed > 0) {
                $userWallet->decrement('cash_wallet', $cashWalletUsed);
            }
            if ($shoppingWalletUsed > 0) {
                $userWallet->decrement('shopping_wallet', $shoppingWalletUsed);
            }

            DB::commit();
            Log::info('Step 14: Transaction committed successfully.');

            return response()->json([
                'message'  => 'Offline order placed successfully!',
                'sale_id'  => $offlineSale->id,
                'order_id' => $orderId
            ], 201);

        } catch (\Exception $e) {
            DB::rollBack();
            Log::error('DB Error: ' . $e->getMessage(), ['trace' => $e->getTraceAsString()]);
            return response()->json(['message' => 'Failed to place offline order due to a DB error.'], 500);
        }

    } catch (ValidationException $e) {
        return response()->json(['message' => 'Validation failed.', 'errors' => $e->errors()], 422);
    } catch (\Exception $e) {
        Log::error('Unexpected error: ' . $e->getMessage(), ['trace' => $e->getTraceAsString()]);
        return response()->json(['message' => 'Failed due to unexpected error.'], 500);
    }
}







  public function editOfflineSale($orderId)
    {
        Log::info("Attempting to edit Offline Sale for Order ID: $orderId");

        try {
            $offlineSale = OfflineSale::where('order_id', $orderId)->firstOrFail();

            $saleItems = json_decode($offlineSale->sale_items, true);
            $customerDetails = json_decode($offlineSale->customer_details, true) ?? [];

            foreach ($saleItems as &$item) {
                $product = Product::where('product_id', $item['product_id'])->first();
                $item['full_product_name'] = $product ? $product->title : ($item['product_title'] ?? 'N/A');

                if (isset($item['variant_id']) && $item['variant_id']) {
                    $variant = ProductVarient::find($item['variant_id']);
                    $item['full_variant_name'] = $variant ? $variant->variant_name : ($item['variant_name'] ?? 'N/A');
                } else {
                    $item['full_variant_name'] = 'N/A';
                }
            }
            unset($item); 

            $user = User::where('user_id', $offlineSale->user_id)->first();
            $address = null;
            if ($user) {
                $address = Address::where('userid', $user->user_id)
                                ->where('is_current', 1)
                                ->first();
            }

            if ($address) {
                $customerDetails['address'] = [
                    'house_no' => $address->house_no ?? '',
                    'road_name' => $address->road_name ?? '',
                    'landmark' => $address->landmark ?? '',
                    'district' => $address->district ?? '',
                    'state' => $address->state ?? '',
                    'pin' => $address->pin ?? '',
                ];
            } else {
                $customerDetails['address'] = [
                    'house_no' => '', 'road_name' => '', 'landmark' => '',
                    'district' => '', 'state' => '', 'pin' => '',
                ];
            }

            $contact = ContactUs::where('is_active', 1)->first();

            $data = [
                'offlineSale'               => $offlineSale,                                     
                'order_id'                  => $offlineSale->order_id,
                'name'                      => $user->name ?? 'N/A',
                'phone_number'              => $user->phone_number ?? 'N/A',
                'sale_id'                   => $offlineSale->id,
                'sale_date'                 => $offlineSale->created_at->format('Y-m-d H:i:s'),
                'customer'                  => $customerDetails,
                'items'                     => $saleItems,
                'total_amount'              => $offlineSale->total_amount,
                'cash_wallet_deduction'     => $offlineSale->cash_wallet_deduction,
                'shopping_wallet_deduction' => $offlineSale->shopping_wallet_deduction,
                'amount_payable'            => $offlineSale->amount_payable,
                'payment_mode'              => $offlineSale->payment_mode,
                'customer_name'             => $user->name ?? 'ADBAZZAR Customer',
                'company_name'              => $contact->company_name ?? 'ADBAZZAR',
                'company_address'           => $contact->company_address ?? 'N/A',
                'company_phone'             => $contact->phone_number ?? 'N/A',
                'company_email'             => $contact->email ?? 'N/A',
                'allProducts'               => Product::all(),                                    // Fetch all products for dynamic selection in the edit form
                'allVariants'               => ProductVarient::all(),                             // Fetch all variants if needed
            ];

            return view('admin.edit_offline_sale_invoice', $data);

        } catch (\Exception $e) {
            Log::error('Error loading Offline Sale for edit: ' . $e->getMessage(), [
                'order_id' => $orderId,
                'trace' => $e->getTraceAsString()
            ]);
            return redirect()->back()->with('error', 'Could not find the offline sale or an error occurred.');
        }
    }



       public function updateOfflineSale(Request $request, $orderId)
    {
        Log::info("Attempting to update Offline Sale for Order ID: $orderId");

        $request->validate([
            'total_amount'               => 'required|numeric|min:0',
            'amount_payable'             => 'required|numeric|min:0',
            'payment_mode'               => 'required|string|max:255',
            'sale_items'                 => 'required|array',
            'sale_items.*.product_id'    => 'required|integer',
            'sale_items.*.quantity'      => 'required|integer|min:1',
            'sale_items.*.sell_price'    => 'required|numeric|min:0',
            'customer_name'              => 'nullable|string|max:255',   
            'customer_phone_number'      => 'nullable|string|max:20',
            'customer_address_house_no'  => 'nullable|string|max:255',
            'customer_address_road_name' => 'nullable|string|max:255',
            'customer_address_landmark'  => 'nullable|string|max:255',
            'customer_address_district'  => 'nullable|string|max:255',
            'customer_address_state'     => 'nullable|string|max:255',
            'customer_address_pin'       => 'nullable|string|max:10',
        ]);

        DB::beginTransaction(); // Start a database transaction

        try {
            $offlineSale = OfflineSale::where('order_id', $orderId)->firstOrFail();

            // --- Update OfflineSale data ---
            $offlineSale->total_amount = $request->input('total_amount');
            $offlineSale->cash_wallet_deduction = $request->input('cash_wallet_deduction', 0); // Default to 0 if not provided
            $offlineSale->shopping_wallet_deduction = $request->input('shopping_wallet_deduction', 0); // Default to 0 if not provided
            $offlineSale->amount_payable = $request->input('amount_payable');
            $offlineSale->payment_mode = $request->input('payment_mode');

            // --- Handle sale_items update ---
            $updatedSaleItems = [];
            foreach ($request->input('sale_items') as $itemData) {
                // You might want to re-fetch product/variant details to ensure data integrity
                $product = Product::where('product_id', $itemData['product_id'])->first();
                $variant = null;
                if (isset($itemData['variant_id']) && $itemData['variant_id']) {
                    $variant = ProductVarient::find($itemData['variant_id']);
                }

                $updatedSaleItems[] = [
                    'product_id' => $itemData['product_id'],
                    'product_title' => $product ? $product->title : ($itemData['product_title'] ?? 'N/A'),
                    'variant_id' => $itemData['variant_id'] ?? null,
                    'variant_name' => $variant ? $variant->variant_name : ($itemData['variant_name'] ?? 'N/A'),
                    'quantity' => (int) $itemData['quantity'],
                    'sell_price' => (float) $itemData['sell_price'],
                    'amount' => (float) $itemData['quantity'] * (float) $itemData['sell_price'], // Recalculate amount
                ];
            }
            $offlineSale->sale_items = json_encode($updatedSaleItems);


            // --- Handle customer_details update ---
            $customerDetails = [
                'name' => $request->input('customer_name'),
                'phone_number' => $request->input('customer_phone_number'),
                'address' => [
                    'house_no' => $request->input('customer_address_house_no'),
                    'road_name' => $request->input('customer_address_road_name'),
                    'landmark' => $request->input('customer_address_landmark'),
                    'district' => $request->input('customer_address_district'),
                    'state' => $request->input('customer_address_state'),
                    'pin' => $request->input('customer_address_pin'),
                ],
            ];
            $offlineSale->customer_details = json_encode($customerDetails);

            $offlineSale->save(); // Save the changes to the database

            DB::commit(); // Commit the transaction

            return redirect()->route('offline_sales_invoice', ['orderId' => $offlineSale->order_id])
                             ->with('success', 'Offline Sale (Invoice) updated successfully!');

        } catch (\Exception $e) {
            DB::rollBack(); // Rollback in case of any error
            Log::error('Error updating Offline Sale: ' . $e->getMessage(), [
                'order_id' => $orderId,
                'trace' => $e->getTraceAsString(),
                'request_data' => $request->all() // Log request data for debugging
            ]);
            return redirect()->back()->withInput($request->all())->with('error', 'Failed to update invoice: ' . $e->getMessage());
        }
    }


// public function place_offlineorder(Request $request)
// {
//     Log::info('--- place_offlineorder called ---');

//     try {
//         Log::info('Step 1: Validating request inputs...');
//         $request->validate([
//             'user_id'              => 'required|exists:users,user_id',
//             'items'                => 'required|array|min:1',
//             'items.*.product_id'   => 'required|string',
//             'items.*.variant_id'   => 'nullable|exists:product_varients,id',
//             'items.*.sell_price'   => 'required|numeric|min:0',
//             'items.*.quantity'     => 'required|integer|min:1',
            
//             'items.*.amount'       => 'required|numeric|min:0',
//             'total_amount'         => 'required|numeric|min:0',
//             'cash_wallet_used'     => 'nullable|numeric|min:0',
//             'shopping_wallet_used' => 'nullable|numeric|min:0',
//             'discount'             => 'nullable|numeric|min:0',
//             'amount_payable'       => 'required|numeric|min:0',
//             'payment_mode'         => 'required|string|in:cash,card,upi',
//         ]);
//         Log::info('Step 2: Validation passed.');

//         $userId             = strtoupper($request->input('user_id'));
//         $cashWalletUsed     = (float) $request->input('cash_wallet_used', 0);
//         $shoppingWalletUsed = (float) $request->input('shopping_wallet_used', 0);
//         $totalAmount        = (float) $request->input('total_amount');
//         $amountPayable      = (float) $request->input('amount_payable');
//         $saleItems          = $request->input('items');
//         $paymentMode        = $request->input('payment_mode');

//         // Order ID
//         Log::info('Step 3: Generating order ID...');
//         $lastOfflineSale = OfflineSale::latest('id')->first();
//         $newOrderIdNumber = 101;
//         if ($lastOfflineSale && preg_match('/^store(\d+)$/', $lastOfflineSale->order_id, $matches)) {
//             $newOrderIdNumber = (int) $matches[1] + 1;
//         }
//         $orderId = 'store' . $newOrderIdNumber;

//         Log::info('Step 3: Parsed request inputs.', compact(
//             'userId', 'cashWalletUsed', 'shoppingWalletUsed',
//             'totalAmount', 'amountPayable', 'saleItems', 'orderId', 'paymentMode'
//         ));

//         Log::info("Step 4: Fetching user wallet with locking (user_id: $userId)...");
//         $userWallet = IncomeWallet::where('user_id', $userId)->lockForUpdate()->first();

//         if (($cashWalletUsed > 0 || $shoppingWalletUsed > 0) && !$userWallet) {
//             Log::warning("Wallet not found but required for deduction (user_id: $userId)");
//             return response()->json(['message' => 'User wallet not found. Cannot deduct wallet amount.'], 400);
//         }

//         Log::info('Step 5: User wallet check passed.');

//         if ($userWallet) {
//             Log::info('Step 7: Verifying wallet balances...');
//             if ($cashWalletUsed > $userWallet->cash_wallet) {
//                 Log::warning("Insufficient cash wallet balance for user_id: $userId");
//                 return response()->json(['message' => 'Insufficient cash wallet balance.'], 400);
//             }
//             if ($shoppingWalletUsed > $userWallet->shopping_wallet) {
//                 Log::warning("Insufficient shopping wallet balance for user_id: $userId");
//                 return response()->json(['message' => 'Insufficient shopping wallet balance.'], 400);
//             }
//         }

 
//         $discount = (float) $request->input('discount', 0);

//         $calculatedTotal = $cashWalletUsed + $shoppingWalletUsed + $amountPayable + $discount;
//         Log::info("Step 8: Validating amount consistency", [
//             'calculated_total' => $calculatedTotal,
//             'provided_total' => $totalAmount
//         ]);
//         if (abs($calculatedTotal - $totalAmount) > 0.01) {
//             return response()->json(['message' => 'Order amount calculation mismatch. Please try again.'], 400);
//         }

//         // Stock Check
//         Log::info('Step 8.5: Checking stock availability...');
//         foreach ($saleItems as $item) {
//             $product = Product::where('product_id', $item['product_id'])->first();
//             if (!$product) {
//                 return response()->json(['message' => "Product '{$item['product_id']}' not found."], 400);
//             }
//             if (!empty($item['variant_id'])) {
//                 $variant = ProductVariant::find($item['variant_id']);
//                 if (!$variant || $variant->stock < $item['quantity']) {
//                     return response()->json(['message' => "Insufficient stock for product variant: {$product->name}"], 400);
//                 }
//             } else {
//                 if ($product->stock < $item['quantity']) {
//                     return response()->json(['message' => "Insufficient stock for product: {$product->name}"], 400);
//                 }
//             }
//         }

//         DB::beginTransaction();
//         Log::info('Step 9: Database transaction started.');

//         try {
//             Log::info('Step 10: Creating offline sale record...');
//             $offlineSale = OfflineSale::create([
//                 'user_id'                   => $userId,
//                 'order_id'                  => $orderId,
//                 'sale_items'                => json_encode($saleItems),
//                 'total_amount'              => $totalAmount,
//                 'cash_wallet_deduction'     => $cashWalletUsed,
//                 'shopping_wallet_deduction' => $shoppingWalletUsed,
//                 'discount'                  => $discount,
//                 'amount_payable'            => $amountPayable,
//                 'payment_mode'              => $paymentMode,
//                 'status'                    => 'completed',
//             ]);

//             foreach ($saleItems as $item) {
//                 $product = Product::where('product_id', $item['product_id'])->first();
//                 if (!empty($item['variant_id'])) {
//                     $variant = ProductVariant::find($item['variant_id']);
//                     $variant->decrement('stock', $item['quantity']);
//                 } else {
//                     $product->decrement('stock', $item['quantity']);
//                 }
//             }

//             // Wallet income logs
//             if ($shoppingWalletUsed > 0) {
//                 ShopWalletIncome::create([
//                     'user_id'       => $userId,
//                     'membership_id' => null,
//                     'amount'        => $shoppingWalletUsed,
//                     'remark'        => "Store_purchase : " . $orderId,
//                     'description'   => "Store_purchase",
//                     'payment_mode'  => "wallet",
//                     'type'          => "debit"
//                 ]);
//             }

//             if ($cashWalletUsed > 0) {
//                 CashWalletIncome::create([
//                     'user_id'       => $userId,
//                     'membership_id' => null,
//                     'amount'        => $cashWalletUsed,
//                     'remark'        => "Store_purchase : " . $orderId,
//                     'description'   => "Store_purchase",
//                     'payment_mode'  => "wallet",
//                     'type'          => "debit"
//                 ]);
//             }

//             if ($cashWalletUsed > 0 && $userWallet) {
//                 $userWallet->decrement('cash_wallet', $cashWalletUsed);
//                 Log::info('Cash wallet updated. New balance:', ['cash_wallet' => $userWallet->cash_wallet]);
//             }

//             if ($shoppingWalletUsed > 0 && $userWallet) {
//                 $userWallet->decrement('shopping_wallet', $shoppingWalletUsed);
//                 Log::info('Shopping wallet updated. New balance:', ['shopping_wallet' => $userWallet->shopping_wallet]);
//             }

//             DB::commit();
//             Log::info('Step 14: Transaction committed successfully.');

//             return response()->json([
//                 'message'  => 'Offline order placed successfully!',
//                 'sale_id'  => $offlineSale->id,
//                 'order_id' => $orderId
//             ], 201);

//         } catch (\Exception $e) {
//             DB::rollBack();
//             Log::error('DB Error: ' . $e->getMessage(), ['trace' => $e->getTraceAsString()]);
//             return response()->json(['message' => 'Failed to place offline order due to a DB error.'], 500);
//         }

//     } catch (ValidationException $e) {
//         return response()->json(['message' => 'Validation failed.', 'errors' => $e->errors()], 422);
//     } catch (\Exception $e) {
//         Log::error('Unexpected error: ' . $e->getMessage(), ['trace' => $e->getTraceAsString()]);
//         return response()->json(['message' => 'Failed due to unexpected error.'], 500);
//     }
// }

    




    // public function generateInvoice($orderId)
    // {
    //     Log::info("Generating invoice for Order ID: $orderId");

    //     try {
    //         $offlineSale = OfflineSale::where('order_id', $orderId)
    //             ->firstOrFail();

    //         $saleItems = json_decode($offlineSale->sale_items, true);
    //         $customerDetails = json_decode($offlineSale->customer_details, true) ?? [];

    //         foreach ($saleItems as &$item) {
    //             $product = Product::where('product_id', $item['product_id'])->first();
    //             $item['full_product_name'] = $product ? $product->title : ($item['product_title'] ?? 'N/A');

    //             if (isset($item['variant_id']) && $item['variant_id']) {
    //                 $variant = ProductVarient::find($item['variant_id']);
    //                 $item['full_variant_name'] = $variant ? $variant->variant_name : ($item['variant_name'] ?? 'N/A');
    //             } else {
    //                 $item['full_variant_name'] = 'N/A';
    //             }
    //         }
    //         unset($item); // Unset the reference after the loop

    //         $user = User::where('user_id', $offlineSale->user_id)->first();

    //         $address = null;
    //         if ($user) {
    //             // Assuming 'userid' is the foreign key in the 'addresses' table
    //             $address = Address::where('userid', $user->user_id)
    //                 ->where('is_current', 1) // Assuming 'is_current' indicates the primary address
    //                 ->first();
    //         }

    //         // Populate customerDetails['address'] based on fetched address
    //         if ($address) {
    //             $customerDetails['address'] = [
    //                 'house_no' => $address->house_no ?? '',
    //                 'road_name' => $address->road_name ?? '',
    //                 'landmark' => $address->landmark ?? '',
    //                 'district' => $address->district ?? '',
    //                 'state' => $address->state ?? '',
    //                 'pin' => $address->pin ?? '',
    //             ];
    //         } else {
    //             // Provide empty strings if no address is found, to prevent errors in view
    //             $customerDetails['address'] = [
    //                 'house_no' => '',
    //                 'road_name' => '',
    //                 'landmark' => '',
    //                 'district' => '',
    //                 'state' => '',
    //                 'pin' => '',
    //             ];
    //         }

    //         $contact = ContactUs::where('is_active', 1)->first();

    //         $data = [
    //             'order_id'                  => $offlineSale->order_id,
    //             'name'                      => $user->name ?? 'N/A', // Use user's name if available
    //             'phone_number'              => $user->phone_number ?? 'N/A', // Use user's phone if available
    //             'sale_id'                   => $offlineSale->id,
    //             'sale_date'                 => $offlineSale->created_at->format('Y-m-d H:i:s'),
    //             'customer'                  => $customerDetails, // Pass the possibly updated customerDetails
    //             'items'                     => $saleItems,
    //             'total_amount'              => $offlineSale->total_amount,
    //             'cash_wallet_deduction'     => $offlineSale->cash_wallet_deduction,
    //             'shopping_wallet_deduction' => $offlineSale->shopping_wallet_deduction,
    //             'discount'                  => $offlineSale->discount,
    //             'amount_payable'            => $offlineSale->amount_payable,
    //             'payment_mode'              => $offlineSale->payment_mode, // Pass payment mode to invoice
    //             'customer_name'             => $user->name ?? 'ADBAZZAR Customer', // Redundant, but kept for consistency
    //             'company_name'              => $contact->company_name ?? 'ADBAZZAR',
    //             'company_address'           => $contact->company_address ?? 'N/A',
    //             'company_phone'             => $contact->phone_number ?? 'N/A',
    //             'company_email'             => $contact->email ?? 'N/A',
    //         ];

    //        return view('admin.offline_sale_invoice', $data);

    //     } catch (\Exception $e) {
    //         Log::error('Error generating invoice: ' . $e->getMessage(), [
    //             'order_id' => $orderId,
    //             'trace' => $e->getTraceAsString()
    //         ]);
    //         return response()->json(['message' => 'Failed to generate invoice. The order might not exist or there was an internal error.'], 500);
    //     }
    // }




}

