<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use App\Models\Order;
use App\Models\OrderDetails;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Facades\DB;

use App\Http\Controllers\Profile\ShiprocketController;


class OrderController extends Controller
{

    public function orders_accept($order_random_id)
    {

        Log::info('Accepting order shiprocket: ' . $order_random_id);

        $order = OrderDetails::with('user.currentAddress', 'product')->where('order_random_id', $order_random_id)->first();
        if (!$order) {
            Log::info('Order not found: ' . $order->order_random_id);
            return redirect()->back()->with('error', 'Order not found.');
        }
        $user    = $order->user;

        Log::info('User details: ' . json_encode($user));
        $address = $order->delivery_address;

        if (!$address) {
            Log::info('User address missing for order ID: ' . $order->id);
            return redirect()->back()->with('error', 'User address not found');
        }

        $productNames = json_decode($order->product_details_name, true);
        $quantities   = json_decode($order->quantity, true);
        $prices       = json_decode($order->price_details, true);
        $discounts    = json_decode($order->discount ?? '[]', true);
        $taxes        = json_decode($order->gst_price ?? '[]', true);
        $product_id   = json_decode($order->product_details_id ?? '[]', true);

        $orderItems = [];

        foreach ($productNames as $index => $name) {
            $orderItems[] = [
                'name'          => $name ?? 'Sample Product',
                'sku'           => $product_id[$index] ?? $name,
                'units'         => $quantities[$index] ?? 1,
                'selling_price' => $prices[$index] ?? 100,
                'discount'      => $discounts[$index] ?? 0,
                'tax'           => $taxes[$index] ?? 0,
                'hsn'           => '',
            ];
        }

        $data = [
            'order_id'              => $order->order_random_id,
            'order_date'            => now()->format('Y-m-d'),
            'pickup_location'       => 'Primary',
            'billing_customer_name' => $user->name ?? 'Guest',
            'billing_last_name'     => '',
            'billing_address'       => $order->house_no . ', ' . $order->road_name . ', ' . $order->landmark,
            'billing_city'          => $order->district ?? 'Unknown',
            'billing_pincode'       => $order->pin ?? '000000',
            'billing_state'         => $order->state ?? 'Unknown',
            'billing_country'       => 'India',
            'billing_email'         => $user->email ?? 'test@example.com',
            'billing_phone'         => $user->phone_number ?? '9999999999',
            'shipping_is_billing'   => true,
            'order_items'           => $orderItems,
            'payment_method'        => 'Prepaid',
            'shipping_charges'      => 0,
            'giftwrap_charges'      => 0,
            'transaction_charges'   => 0,
            'total_discount'        => 0,
            'sub_total'             => $order->amount ?? 100,
            'length'                => 5,
            'breadth'               => 5,
            'height'                => 5,
            'weight'                => 0.01,
        ];

        Log::info('Sending order to Shiprocket:', $data);

        try {
            DB::beginTransaction();

            $shipcontroller = new ShiprocketController();
            $response       = $shipcontroller->api('post', 'orders/create/adhoc', $data);

            Log::info('response create order from Shiprocket: ' . json_encode($response));

            if ($response->status == 'success') {
                $order->status              = 'accepted';
                $order->ship_rocket_status  = 'Placed';
                $order->shiprocket_order_id = $response->data['order_id'];
                $order->shipment_id         = $response->data['shipment_id'];
                $order->save();

                Log::info('Order successfully sent to Shiprocket. Order ID: ' . $order->id);
                DB::commit();

                return redirect()->back()->with('success', 'Order accepted & sent to Shiprocket!');
            } else {
                DB::rollback();
                Log::info('Shiprocket API failed. Response:', (array)$response);
                return redirect()->back()->with('error', 'Order failed to send to Shiprocket.');
            }
        } catch (\Exception $e) {
            DB::rollback();

            Log::info('Exception while sending order to Shiprocket: ' . $e->getMessage());
            return redirect()->back()->with('error', 'Something went wrong: ' . $e->getMessage());
        }
    }

    public function cancel_shiprocket_order(Request $request)
    {
        try {

            $orderId = $request->input('order_id');
            $order   = OrderDetails::where('id',  $orderId)->first();

            if (!$order) {
                Log::info('Order not found: ' . $order->order_random_id);
                return redirect()->back()->with('error', 'Order not found.');
            }

            $data = ['ids' => [$order->shiprocket_order_id]];

            $shipcontroller = new ShiprocketController();
            $response       = $shipcontroller->api('post', 'orders/cancel', $data);

            Log::info('response cancel order from Shiprocket: ' . json_encode($response));

            if ($response->status == 'success') {

                $order->ship_rocket_status = 'Cancelled';
                $order->save();

                Log::info('Order successfully sent to Shiprocket. Order ID: ' . $order->order_random_id);
                DB::commit();

                return redirect()->back()->with('success', 'Order Cancelled SuccessFully From Shiprocket!');
            } else {
                DB::rollback();
                Log::info('Shiprocket API failed. Response:', (array)$response);
                return redirect()->back()->with('error', 'Order failed to Cancel From Shiprocket.');
            }
            
        } catch (\Exception $e) {

            DB::rollback();
            Log::info('Exception while sending order to Shiprocket: ' . $e->getMessage());
            return redirect()->back()->with('error', 'Something went wrong: ' . $e->getMessage());
        }
    }


    public function decline_order(Request $request)
    {
        $request->validate([
            'order_id' => 'required|exists:order_details,id',
        ]);

        $order = OrderDetails::find($request->order_id);
        $order->status = 'cancelled';
        $order->save();

        return redirect()->route('recent_orders')->with('successmessage', 'Order declined successfully.');
    }
    public function ship_order(Request $request)
    {
        $request->validate([
            'order_id' => 'required|exists:order_details,id',

        ]);

        $order         = OrderDetails::find($request->order_id);
        $order->status = 'shipped';
        $order->save();

        return redirect()->route('accepted_orders')->with('successmessage', 'Order shipped successfully!');
    }

    public function confirm_delivery(Request $request)
    {
        $validated = $request->validate([
            'order_id' => 'required|exists:orders,id',
            'status' => 'required|string|in:delivered',
        ]);


        $order         = OrderDetails::find($request->order_id);
        $order->status = 'delivered';
        $order->save();

        return redirect()->route('delivered_orders')->with('successmessage', 'Order has been marked as delivered.');
    }


    public function cancelledOrders()
    {
        $cancelled_orders = OrderDetails::where('status', 'cancelled')->get();

        return view('admin.cancelled_order', compact('cancelled_orders'));
    }
    public function viewCancelOrder($order_random_id)
    {

        $cancelled_order = Order::with(['products.product.category', 'products.product.subcategory'])
            ->where('order_random_id', $order_random_id)
            ->where('status', 'cancelled')
            ->get();

        return view('admin.view_cancel_order', compact('cancelled_order'));
    }



 public function updateImeiSerial(Request $request)
    {
        $orderDetailId = $request->input('order_id'); 
        $itemIndex = $request->input('item_index'); 
        $imeiSerialNumber = $request->input('imei_serial_number');

        Log::info('Attempting to update IMEI/Serial Number.', [
            'order_detail_id' => $orderDetailId,
            'item_index' => $itemIndex,
            'imei_serial_number_input' => $imeiSerialNumber,
            'ip_address' => $request->ip(),
            'user_agent' => $request->header('User-Agent'),
        ]);

        try {
            // Find the specific OrderDetails record by its ID
            $orderDetails = OrderDetails::find($orderDetailId);

            if (!$orderDetails) {
                Log::warning('IMEI/Serial Number update failed: Order item not found.', [
                    'order_detail_id' => $orderDetailId
                ]);
                return response()->json(['success' => false, 'message' => 'Order item not found.'], 404);
            }

            // Decode the existing imei_serial_numbers JSON string
            $existingImeiSerialNumbers = json_decode($orderDetails->imei_serial_numbers, true) ?? [];

            Log::debug('Existing IMEI/Serial Numbers before update.', [
                'order_detail_id' => $orderDetailId,
                'current_imei_serial_numbers' => $existingImeiSerialNumbers,
            ]);

            // Update the specific item index with the new IMEI/Serial number
            // Ensure the itemIndex is treated as an integer for array access
            $existingImeiSerialNumbers[(int)$itemIndex] = $imeiSerialNumber;

            Log::info('Setting IMEI/Serial Number for item index.', [
                'order_detail_id' => $orderDetailId,
                'item_index' => $itemIndex,
                'new_imei_serial_number' => $imeiSerialNumber,
            ]);

            // Encode the updated array back to JSON and save
            $orderDetails->imei_serial_numbers = json_encode($existingImeiSerialNumbers);
            $orderDetails->save();

            Log::info('IMEI/Serial Number updated successfully.', [
                'order_detail_id' => $orderDetailId,
                'final_imei_serial_numbers' => $orderDetails->imei_serial_numbers,
            ]);

            return response()->json(['success' => true, 'message' => 'IMEI/Serial Number updated successfully.']);

        } catch (\Exception $e) {
            Log::error('Error updating IMEI/Serial Number: ' . $e->getMessage(), [
                'order_detail_id' => $orderDetailId,
                'item_index' => $itemIndex,
                'imei_serial_number_input' => $imeiSerialNumber,
                'exception' => $e->getTraceAsString(),
            ]);
            return response()->json(['success' => false, 'message' => 'Error updating IMEI/Serial Number: ' . $e->getMessage()], 500);
        }
    }




}
