<?php



namespace App\Http\Controllers\Admin;



use App\Http\Controllers\Controller;

use Illuminate\Http\Request;

use App\Models\Product;

use App\Models\User;

use App\Models\ProductVarient; 

use App\Models\Bill;

use App\Models\BillItem;

use Illuminate\Support\Facades\DB;



class PosController extends Controller

{

   



public function searchProducts(Request $request)

    {

        $query = $request->input('query');



        if (empty($query)) {

            return response()->json([]);

        }



        $products = Product::where('title', 'like', '%' . $query . '%')

                           ->orWhere('item_code', 'like', '%' . $query . '%')

                           ->orWhere('hsn_code', 'like', '%' . $query . '%')

                           ->with('variants')

                           ->get();



        $results = [];

        foreach ($products as $product) {

            $results[] = [

                'id'         => $product->id,

                'type'       => 'product',

                'name'       => $product->title,

                'item_code'  => $product->item_code,

                'hsn_code'   => $product->hsn_code ?? '',

                'mrp'        => $product->mrp,

                'sell_price' => $product->sell_price,

                'gst'        => $product->gst,

                'cgst'       => $product->cgst,

                'sgst'       => $product->sgst,

            ];



            foreach ($product->variants as $variant) {

                $results[] = [

                    'id'         => $variant->id,

                    'product_id' => $product->id,

                    'type'       => 'variant',

                    'name'       => $product->title . ' (' . $variant->variant_name . ')',

                    'item_code'  => $product->item_code,

                    'hsn_code'   => $product->hsn_code ?? '',

                    'mrp'        => $variant->mrp,

                    'sell_price' => $variant->sell_price,

                    'gst'        => $variant->gst,

                    'cgst'       => $variant->cgst,

                    'sgst'       => $variant->sgst,

                ];

            }

        }



        return response()->json($results);

    }

  

    public function getCustomerDetails(Request $request)

    {

        $identifier = $request->input('identifier');



        if (empty($identifier)) {

            return response()->json(null);

        }



        $user = User::where('user_id', $identifier)

                    ->orWhere('phone_number', $identifier)

                    ->first();



        if ($user) {

            

            $incomeWallet          = $user->incomeWallet;

            $cashWalletBalance     = $incomeWallet ? $incomeWallet->cash_wallet : 0;

            $shoppingWalletBalance = $incomeWallet ? $incomeWallet->shopping_wallet : 0;



            return response()->json([

                'name'                    => $user->name,

                'phone_number'            => $user->phone_number,

                'address'                 => $user->house_no . ', ' . $user->building_name . ', ' . $user->road_name . ', ' . $user->district . ', ' . $user->state . ' - ' . $user->pin,

                'user_id'                 => $user->user_id,

                'cash_wallet_balance'     => $cashWalletBalance,

                'shopping_wallet_balance' => $shoppingWalletBalance,

            ]);

        }



        return response()->json(null);

    }



  

    public function saveBill(Request $request)

    {

        try {

            $request->validate([

                'billItems'                    => 'required|array|min:1',

                'billItems.*.id'               => 'required|integer',

                'billItems.*.product_id'       => 'required|integer',       

                'billItems.*.type'             => 'required|string',

                'billItems.*.name'             => 'required|string',

                'billItems.*.item_code'        => 'required|string',

                'billItems.*.mrp'              => 'required|numeric',

                'billItems.*.sell_price'       => 'required|numeric',

                'billItems.*.quantity'         => 'required|integer|min:1',

                'billItems.*.gst'              => 'nullable|numeric',

                'billItems.*.cgst'             => 'nullable|numeric',

                'billItems.*.sgst'             => 'nullable|numeric',

                'subTotal'                     => 'required|numeric|min:0',

                'taxAmount'                    => 'required|numeric|min:0',

                'discountAmount'               => 'required|numeric|min:0',

                'additionalCharge'             => 'required|numeric|min:0',

                'totalAmount'                  => 'required|numeric|min:0',

                'amountPayable'                => 'required|numeric|min:0',   

                'receivedAmount'               => 'required|numeric|min:0',

                'cashWalletDeduction'          => 'nullable|numeric|min:0',   

                'shoppingWalletDeduction'      => 'nullable|numeric|min:0',   // New

                'customerDetails'              => 'nullable|array',           // New

                'customerDetails.user_id'      => 'nullable|string',          // New

                'customerDetails.name'         => 'nullable|string',          // New

                'customerDetails.phone_number' => 'nullable|string',          // New

                'paymentMode'                  => 'required|string',          // New

                'status'                       => 'required|string',          // 'completed' or 'held'

            ]);



            DB::beginTransaction();



            $customerDetails = $request->input('customerDetails');

            $userId = $customerDetails['user_id'] ?? null;



            // Generate a unique order_id for the offline sale

            $orderId = 'POS-' . date('YmdHis') . '-' . str_pad(mt_rand(1, 99999), 5, '0', STR_PAD_LEFT);



            $offlineSale = OfflineSale::create([

                'user_id' => $userId,

                'order_id' => $orderId,

                'sale_items' => $request->input('billItems'),

                'sub_total' => $request->input('subTotal'),

                'tax_amount' => $request->input('taxAmount'),

                'discount' => $request->input('discountAmount'),

                'additional_charge' => $request->input('additionalCharge'),

                'total_amount' => $request->input('totalAmount'), // Total before wallet deduction

                'cash_wallet_deduction' => $request->input('cashWalletDeduction', 0),

                'shopping_wallet_deduction' => $request->input('shoppingWalletDeduction', 0),

                'amount_payable' => $request->input('amountPayable'), // Final amount payable by customer

                'customer_details' => $customerDetails,

                'status' => $request->input('status'), // 'completed' or 'held'

                'payment_mode' => $request->input('paymentMode'),

                'description' => 'POS Sale', // You can customize this

            ]);



            // Deduct from wallets if used

            if ($userId) {

                $userWallet = IncomeWallet::where('user_id', $userId)->first();

                if ($userWallet) {

                    if ($request->input('cashWalletDeduction') > 0) {

                        $userWallet->cash_wallet -= $request->input('cashWalletDeduction');

                    }

                    if ($request->input('shoppingWalletDeduction') > 0) {

                        $userWallet->shopping_wallet -= $request->input('shoppingWalletDeduction');

                    }

                    $userWallet->save();

                }

            }



            DB::commit();



            return response()->json([

                'message' => 'Bill saved successfully!',

                'bill_number' => $offlineSale->order_id,

                'sale_id' => $offlineSale->id,

            ]);



        } catch (ValidationException $e) {

            DB::rollBack();

            return response()->json([

                'message' => 'Validation error',

                'errors' => $e->errors(),

            ], 422);

        } catch (\Exception $e) {

            DB::rollBack();

            \Log::error('POS Save Bill Error: ' . $e->getMessage() . ' in ' . $e->getFile() . ' on line ' . $e->getLine());

            return response()->json([

                'message' => 'An error occurred while saving the bill.',

                'error' => $e->getMessage()

            ], 500);

        }

    }



    /**

     * Get held bills.

     */

    public function getHeldBills(Request $request)

    {

        $heldBills = OfflineSale::where('status', OfflineSale::STATUS_HELD)

            ->orderByDesc('created_at')

            ->get();



        return response()->json($heldBills);

    }



    /**

     * Resume a held bill.

     */

    public function resumeHeldBill(Request $request, $id)

    {

        $heldBill = OfflineSale::where('id', $id)

                                ->where('status', OfflineSale::STATUS_HELD)

                                ->first();



        if (!$heldBill) {

            return response()->json(['message' => 'Held bill not found or already processed.'], 404);

        }



        return response()->json($heldBill);

    }



    /**

     * Delete a held bill.

     */

    public function deleteHeldBill(Request $request, $id)

    {

        $heldBill = OfflineSale::where('id', $id)

                                ->where('status', OfflineSale::STATUS_HELD)

                                ->first();



        if (!$heldBill) {

            return response()->json(['message' => 'Held bill not found.'], 404);

        }



        $heldBill->delete();



        return response()->json(['message' => 'Held bill deleted successfully.']);

    }



    /**

     * Print invoice (A4 or Thermal).

     * This is a simplified example; actual printing involves more complex HTML/CSS rendering

     * or a dedicated printing service/library.

     */

    public function printInvoice(Request $request, $id)

    {

        $offlineSale = OfflineSale::find($id);



        if (!$offlineSale) {

            abort(404);

        }



        $printType = $request->query('type', 'a4'); // 'a4' or 'thermal'



        // You'll need a dedicated view for each print type for proper formatting

        if ($printType === 'thermal') {

            return view('pos.invoices.thermal', compact('offlineSale'));

        }



        return view('pos.invoices.a4', compact('offlineSale'));

    }







}

