<?php
namespace App\Http\Controllers\Admin;
use App\Http\Controllers\Controller;
use App\Models\Product;
use Illuminate\Http\Request;
use App\Models\ProductPurchase;
use App\Models\MyPurchase;
use App\Models\MyPurchaseItem;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Str;
class PurchaseController extends Controller
{
public function invoice_a4_purchase($orderId)
{
    $myPurchase = MyPurchase::with(['purchaseParty', 'items'])->findOrFail($orderId);
    $user = auth()->user();
    $branchInfo = [
        'name'    => $user->name ?? 'AdBazzar',
        'address' => $user->building_name . ', ' . $user->road_name . ', ' . $user->district . ', ' . $user->state . ' - ' . $user->pin ?? 'Chennai, India',
        'phone'   => $user->phone_number ?? '+91 8531345678',
        'email'   => $user->email ?? 'adbazzar@gmail.com',
        'logo'    => !empty($user->logo) ? asset($user->logo) : null, // path to logo
    ];
    return view('admin.invoice_a4_purchase', compact('myPurchase', 'branchInfo'));
}
    public function invoice_thermal_purchase($orderId)
    {
        $myPurchase = MyPurchase::with(['purchaseParty', 'items.product'])->findOrFail($orderId);
        $user = auth()->user();
        $branchInfo = match ($user->role) {
            'sakthi' => [
                'name' => 'Sakthi Super Mart',
                'address' => '615/3 Pandian Street 
Chennimalai - 638051',
                'phone' => '+91 9500598595',
                'email' => 'admin1@gmail.com',
            ],
            'smart_world' => [
                'name' => 'SMARTWORLD',
                'address' => 'Chennai, India',
                'phone' => '+91 9443595314',
                'email' => 'admin2@gmail.com',
            ],
            'smart_world_gift' => [
                'name' => 'SMARTWORLD - Stationary & Gifts',
                'address' => 'Chennai, India',
                'phone' => '+91 9443595315',
                'email' => 'admin3@gmail.com',
            ],
            'yoga_naturals' => [
                'name' => 'Yoga Naturals',
                'address' => 'Chennai, India',
                'phone' => '+91 9443595316',
                'email' => 'admin4@gmail.com',
            ],
            default => [
                'name' => 'AdBazzar',
                'address' => 'Chennai, India',
                'phone' => '+91 8531345678',
                'email' => 'adbazzar@gmail.com',
            ],
        };
        return view('admin.invoice_thermal_purchase', compact('myPurchase', 'branchInfo'));
    }
    public function create_invoice()
    {
        return view('admin.create_invoice');
    }
    public function editPurchase($id)
    {
        $myPurchase = MyPurchase::with(['purchaseParty', 'items.product'])->findOrFail($id);
        $products = Product::all();
        $user = auth()->user();
        // Branch-wise info
        $branchInfo = match ($user->role) {
            'sakthi' => [
                'name' => 'Sakthi Super Mart',
                'address' => '615/3 Pandian Street Chennimalai - 638051',
                'phone' => '+91 9500598595',
                'email' => 'admin1@gmail.com',
            ],
            'smart_world' => [
                'name' => 'SMARTWORLD',
                'address' => 'Chennai, India',
                'phone' => '+91 9443595314',
                'email' => 'admin2@gmail.com',
            ],
            'smart_world_gift' => [
                'name' => 'SMARTWORLD - Stationary & Gifts',
                'address' => 'Chennai, India',
                'phone' => '+91 9443595315',
                'email' => 'admin3@gmail.com',
            ],
            'yoga_naturals' => [
                'name' => 'Yoga Naturals',
                'address' => 'Chennai, India',
                'phone' => '+91 9443595316',
                'email' => 'admin4@gmail.com',
            ],
            default => [
                'name' => 'AdBazzar',
                'address' => 'Chennai, India',
                'phone' => '+91 8531345678',
                'email' => 'adbazzar@gmail.com',
            ],
        };
        return view('admin.purchase_edit', compact('myPurchase', 'products', 'branchInfo'));
    }
    public function getAllProducts()
    {
        $products = Product::select('title', 'item_code', 'sell_price', 'purchase_price', 'stock')->get();
        return response()->json($products);
    }
   public function PurchasesearchProducts(Request $request)
{
    try {
        $query = $request->input('query');

        if (empty($query)) {
            return response()->json([], 200);
        }

        $user = auth()->user(); // 🔐 login user

        $productsQuery = Product::query();

        // 🔍 search condition (grouped)
        $productsQuery->where(function ($q) use ($query) {
            $q->where('title', 'LIKE', "%{$query}%")
              ->orWhere('item_code', 'LIKE', "%{$query}%")
              ->orWhere('product_id', 'LIKE', "%{$query}%")
              ->orWhere('hsn_code', 'LIKE', "%{$query}%")
              ->orWhere('barcode1', 'LIKE', "%{$query}%")
              ->orWhere('barcode2', 'LIKE', "%{$query}%");
        });

        // 🏬 Branch filter (admin illa na)
        if ($user->role !== 'admin') {
            $productsQuery->where('user_role', $user->role);
            // 👆 product table branch_name
            // 👆 user -> branch relation
        }

        $products = $productsQuery->select(
            'id',
            'product_id',
            'title as name',
            'item_code',
            'mrp',
            'purchase_price',
            'sell_price',
            'stock',
            'gst as gst_rate',
            'gst_price',
            'shop_wallet',
            'hsn_code',
            'barcode1',
            'barcode2'
        )->get();

        return response()->json($products);

    } catch (\Exception $e) {
        Log::error('Error in PurchasesearchProducts: ' . $e->getMessage(), [
            'trace' => $e->getTraceAsString()
        ]);

        return response()->json([
            'message' => 'An internal server error occurred during product search.',
            'error'   => $e->getMessage()
        ], 500);
    }
}

    // public function searchSuppliers(Request $request)
    // {
    //     $query = $request->input('query');
    //     $supplier = ProductPurchase::where('phone_number', $query)
    //         ->orWhere('customer_name', 'like', "%{$query}%")
    //         ->first();
    //     if ($supplier) {
    //         return response()->json($supplier);
    //     }
    //     return response()->json(['message' => 'Supplier not found.'], 404);
    // }
public function searchSuppliers(Request $request)
{
    $query = $request->input('query');
    // Ensure the query is not empty before searching
    if (empty($query)) {
        return response()->json(['message' => 'Query is required.'], 400);
    }
    // Search by exact phone number, or partial match on customer_name or agency_name
    $supplier = ProductPurchase::where('phone_number', $query)
        ->orWhere('customer_name', 'like', "%{$query}%")
        ->orWhere('agency_name', 'like', "%{$query}%") // ADDED agency_name search
        ->first();
    if ($supplier) {
        return response()->json($supplier);
    }
    return response()->json(['message' => 'Supplier not found.'], 404);
}
    public function storeSupplier(Request $request)
    {
        try {
            $user = auth()->user();
            $branch = null;
            Log::info('Step 4: Determining branch based on user role.', ['user_role' => $user->role]);
            if ($user->role === 'sakthi') {
                $branch = 'Sakthi Super Mart';
            } elseif ($user->role === 'smart_world') {
                $branch = 'SMARTWORLD';
            } elseif ($user->role === 'smart_world_gift') {
                $branch = 'SMARTWORLD - Stationary & Gifts';
            } elseif ($user->role === 'yoga_naturals') {
                $branch = 'yoga naturals';
            } else {
                Log::info('No specific branch determined for user role: ' . $user->role);
            }
            Log::info('Step 5: Branch determined as: ' . ($branch ?? 'N/A'));
            $rolePrefix = match ($user->role) {
                'sakthi' => 'Sakthi',
                'smart_world' => 'Smartworld',
                'smart_world_gift' => 'Smartworldgift',
                'yoga_naturals' => 'Yoganaturals',
                default => 'admin',
            };
            $validatedData = $request->validate([
                'customer_name' => 'required|string|max:255',
                'agency_name' => 'required|string|max:255',
                'phone_number' => 'nullable|string|max:20|unique:product_purchases,phone_number',
                // 'email' => 'nullable|email|max:255',
                'address' => 'nullable|string|max:255',
                'gstin' => 'nullable|string|max:15',
                'branch' => 'nullable|string|max:255',
            ]);
            $validatedData['branch'] = $validatedData['branch'] ?? $branch;
            $supplier = ProductPurchase::create([
                'customer_name' => $validatedData['customer_name'],
                 'agency_name' => $validatedData['agency_name'],
                'phone_number' => $validatedData['phone_number'] ?? null,
                // 'email' => $validatedData['email'] ?? null,
                'address' => $validatedData['address'] ?? null,
                'gstin' => $validatedData['gstin'] ?? null,
                'branch' => $validatedData['branch'] ?? null,
            ]);
            return response()->json([
                'success' => true,
                'message' => 'Supplier registered successfully.',
                'supplier' => $supplier
            ], 201);
        } catch (\Illuminate\Validation\ValidationException $e) {
            Log::error('Validation error for new supplier: ' . $e->getMessage(), ['errors' => $e->errors()]);
            return response()->json([
                'success' => false,
                'message' => 'Validation failed.',
                'errors' => $e->errors()
            ], 422);
        } catch (\Exception $e) {
            Log::error('Error registering new supplier: ' . $e->getMessage(), ['trace' => $e->getTraceAsString()]);
            return response()->json([
                'success' => false,
                'message' => 'Failed to register supplier due to an internal server error.'
            ], 500);
        }
    }
    public function purchase_store(Request $request)
    {
        Log::info('Incoming purchase request:', $request->all());
        Log::info('Step 1: Starting request validation.');
        $request->validate([
            'product_purchase_id' => 'required|exists:product_purchases,id',
            'invoice_items' => 'required|array',
            'invoice_items.*.product_id' => 'required|exists:products,id',
            'invoice_items.*.quantity' => 'required|integer|min:1',
            'invoice_items.*.purchase_price' => 'required|numeric|min:0',
            'invoice_items.*.sell_price' => 'required|numeric|min:0',
            'discount_amount' => 'nullable|numeric|min:0',
            'additional_charge' => 'nullable|numeric|min:0',
            'gst_total' => 'required|numeric|min:0',
            'sub_total_excluding_gst' => 'required|numeric|min:0',
            'sub_total_including_gst' => 'required|numeric|min:0',
            'total_amount' => 'required|numeric|min:0',
            'print_type' => 'required|in:a4,thermal',
        ]);
        Log::info('Step 2: Request validation successful.');
        Log::info('Step 3: Starting database transaction.');
        DB::beginTransaction();
        try {
            $user = auth()->user();
            $branch = null;
            Log::info('Step 4: Determining branch based on user role.', ['user_role' => $user->role]);
            if ($user->role === 'sakthi') {
                $branch = 'Sakthi Super Mart';
            } elseif ($user->role === 'smart_world') {
                $branch = 'SMARTWORLD';
            } elseif ($user->role === 'smart_world_gift') {
                $branch = 'SMARTWORLD - Stationary & Gifts';
            } elseif ($user->role === 'yoga_naturals') {
                $branch = 'yoga naturals';
            } else {
                Log::info('No specific branch determined for user role: ' . $user->role);
            }
            Log::info('Step 5: Branch determined as: ' . ($branch ?? 'N/A'));
            $rolePrefix = match ($user->role) {
                'sakthi' => 'Sakthi',
                'smart_world' => 'Smartworld',
                'smart_world_gift' => 'Smartworldgift',
                'yoga_naturals' => 'Yoganaturals',
                default => 'admin',
            };
            // $invoiceNumber = 'PI-' . date('Ymd') . '-' . Str::upper(Str::random(6));
            $invoiceNumber = 'PI-' . $rolePrefix . '-' . date('Ymd') . '-' . Str::upper(Str::random(6));
            Log::info('Step 6: Generated invoice number: ' . $invoiceNumber);
            Log::info('Step 7: Creating MyPurchase record with data:', [
                'product_purchase_id' => $request->product_purchase_id,
                'invoice_number' => $invoiceNumber,
                'purchase_date' => now(),
                'discount_amount' => $request->discount_amount ?? 0,
                'additional_charge' => $request->additional_charge ?? 0,
                'gst_total' => $request->gst_total,
                'sub_total_excluding_gst' => $request->sub_total_excluding_gst,
                'sub_total_including_gst' => $request->sub_total_including_gst,
                'total_amount' => $request->total_amount,
                'branch' => $branch,
                'print_type' => $request->print_type,
            ]);
            $purchase = MyPurchase::create([
                'product_purchase_id' => $request->product_purchase_id,
                'invoice_number' => $invoiceNumber,
                'purchase_date' => now(),
                'discount_amount' => $request->discount_amount ?? 0,
                'discount_note' => $request->discount_note,
                'additional_charge' => $request->additional_charge ?? 0,
                'additional_charge_note' => $request->additional_charge_note,
                'gst_total' => $request->gst_total,
                'sub_total_excluding_gst' => $request->sub_total_excluding_gst,
                'sub_total_including_gst' => $request->sub_total_including_gst,
                'total_amount' => $request->total_amount,
                'branch' => $branch,
                'print_type' => $request->print_type,
            ]);
            Log::info('Step 8: MyPurchase record created successfully with ID: ' . $purchase->id);
            Log::info('Step 9: Processing invoice items. Total items: ' . count($request->invoice_items));
            foreach ($request->invoice_items as $index => $itemData) {
                Log::info("Step 9.{$index}: Fetching product for ID: {$itemData['product_id']} to get GST details.");
                $product = Product::find($itemData['product_id']);
                $gst_percentage = 0;
                $gst_price = 0;
                $cgst = 0;
                $sgst = 0;
                if ($product) {
                    // Assuming your 'products' table has a 'gst' column for the GST percentage
                    $gst_percentage = $product->gst ?? 0;
                    $gst_price = $product->gst_price ?? 0; // Use the 'gst' column from the product
                    $cgst = $gst_price / 2;
                    $sgst = $gst_price / 2;
                    Log::info("Step 9.{$index}: Product found. GST Rate: {$gst_price}%, CGST: {$cgst}%, SGST: {$sgst}%.");
                    Log::info("Step 9.{$index}: Creating MyPurchaseItem for product ID: {$itemData['product_id']} with calculated GST values.");
                    MyPurchaseItem::create([
                        'my_purchase_id' => $purchase->id,
                        'product_id'     => $itemData['product_id'],
                        'quantity'       => $itemData['quantity'],
                        'purchase_price' => $itemData['purchase_price'],
                        'sell_price'     => $itemData['sell_price'],
                        'gst_percentage' => $gst_percentage,
                        'gst_price'      => $gst_price,
                        'cgst'           => $cgst,
                        'sgst'           => $sgst,
                    ]);
                    Log::info("Step 9.{$index}: MyPurchaseItem created for product ID: {$itemData['product_id']}.");
                    Log::info("Step 9.{$index}: Updating product stock and prices for ID: {$itemData['product_id']}.");
                    $product->stock += $itemData['quantity'];
                    $product->purchase_price = $itemData['purchase_price'];  
                    $product->sell_price = $itemData['sell_price'];     
                    $product->save();
                    Log::info("Step 9.{$index}: Product stock and prices updated for ID: {$itemData['product_id']}.");
                } else {
                    Log::warning("Step 9.{$index}: Product with ID {$itemData['product_id']} not found during item processing. Skipping stock/price update and GST calculation for this item.");
                    MyPurchaseItem::create([
                        'my_purchase_id' => $purchase->id,
                        'product_id' => $itemData['product_id'],
                        'quantity' => $itemData['quantity'],
                        'purchase_price' => $itemData['purchase_price'],
                        'sell_price' => $itemData['sell_price'],
                        'gst_percentage' => 0, 
                        'cgst' => 0,
                        'sgst' => 0,
                    ]);
                }
            }
            Log::info('Step 10: All invoice items processed.');
            Log::info('Step 11: Committing database transaction.');
            DB::commit();
            Log::info('Step 12: Database transaction committed successfully.');
            if ($request->print_type === 'a4') {
                $printUrl = route('invoice_a4_purchase', ['orderId' => $purchase->id]);
            } elseif ($request->print_type === 'thermal') {
                $printUrl = route('invoice_thermal_purchase', ['orderId' => $purchase->id]);
            } else {
                abort(404, 'Invalid print type.');
            }
            Log::info('Step 13: Generated print URL: ' . $printUrl);
            Log::info('Step 14: Returning success response. Invoice ID: ' . $purchase->id);
            return response()->json([
                'success' => true,
                'message' => 'Purchase invoice created successfully.',
                'invoice_id' => $purchase->id,
                'print_url' => $printUrl
            ], 201);
        } catch (\Exception $e) {
            Log::error('Step 15: An error occurred. Rolling back database transaction.', ['error_message' => $e->getMessage(), 'trace' => $e->getTraceAsString()]);
            DB::rollBack();
            Log::info('Step 16: Database transaction rolled back.');
            Log::error('Step 17: Returning error response.');
            return response()->json(['success' => false, 'message' => 'Failed to create purchase invoice.', 'error' => $e->getMessage()], 500);
        }
    }
    public function purchase_parties()
    {
        $user = auth()->user();
        $branch = null;
        if ($user->role === 'sakthi') {
            $branch = 'Sakthi Super Mart';
        } elseif ($user->role === 'smart_world') {
            $branch = 'SMARTWORLD';
        } elseif ($user->role === 'smart_world_gift') {
            $branch = 'SMARTWORLD - Stationary & Gifts';
        } elseif ($user->role === 'yoga_naturals') {
            $branch = 'yoga naturals';
        }
        $purchases = ProductPurchase::query()
            ->when($user->role !== 'admin', function ($query) use ($branch) {
                $query->where('branch', $branch);
            })
            ->with('supplier') 
            ->latest()
            ->get();
        return view('admin.purchase_parties', compact('purchases'));
    }
    public function printPurchaseInvoice($id, $type)
    {
        $purchase = ProductPurchase::with(['supplier', 'items.product'])->findOrFail($id);
        if ($type === 'a4') {
            return view('purchase_invoice_a4', compact('purchase'));
        } elseif ($type === 'thermal') {
            return view('purchase_invoice_thermal', compact('purchase'));
        }
        abort(404, 'Invalid print type.');
    }
    public function updatePurchase(Request $request, $id)
    {
        Log::info('Incoming update purchase request for ID ' . $id, $request->all());
        Log::info('Step 1: Starting request validation.');
        // Validate the incoming request data
        $request->validate([
            'product_purchase_id' => 'required|exists:product_purchases,id',
            'invoice_items' => 'required|array',
            'invoice_items.*.id' => 'sometimes|nullable|exists:my_purchase_items,id', // 'id' is optional for new items
            'invoice_items.*.product_id' => 'required|exists:products,id',
            'invoice_items.*.quantity' => 'required|integer|min:1',
            'invoice_items.*.purchase_price' => 'required|numeric|min:0',
            'invoice_items.*.sell_price' => 'required|numeric|min:0',
            // 'original_quantity' is not needed in the request if we recalculate stock, but is helpful for logging
            'discount_amount' => 'nullable|numeric|min:0',
            'additional_charge' => 'nullable|numeric|min:0',
            'gst_total' => 'required|numeric|min:0',
            'sub_total_excluding_gst' => 'required|numeric|min:0',
            'sub_total_including_gst' => 'required|numeric|min:0',
            'total_amount' => 'required|numeric|min:0',
            'print_type' => 'required|in:a4,thermal',
        ]);
        Log::info('Step 2: Request validation successful.');
        Log::info('Step 3: Starting database transaction.');
        DB::beginTransaction();
        try {
            $purchase = MyPurchase::with('items')->findOrFail($id);
            $existingItemIds = $purchase->items->pluck('id')->toArray();
            $updatedItemIds = [];
            // Step 4: Revert old stock before updating
            Log::info('Reverting old stock before processing updates.');
            foreach ($purchase->items as $item) {
                $product = Product::find($item->product_id);
                if ($product) {
                    $product->stock -= $item->quantity;
                    $product->save();
                    Log::info("Reverted {$item->quantity} from stock for product {$product->id}. New stock: {$product->stock}");
                }
            }
            // Step 5: Update the main purchase record
            Log::info('Updating MyPurchase record with data:', $request->only([
                'discount_amount',
                'discount_note',
                'additional_charge',
                'additional_charge_note',
                'gst_total',
                'sub_total_excluding_gst',
                'sub_total_including_gst',
                'total_amount',
                'print_type'
            ]));
            $purchase->update([
                'product_purchase_id' => $request->product_purchase_id,
                'discount_amount' => $request->discount_amount ?? 0,
                'discount_note' => $request->discount_note,
                'additional_charge' => $request->additional_charge ?? 0,
                'additional_charge_note' => $request->additional_charge_note,
                'gst_total' => $request->gst_total,
                'sub_total_excluding_gst' => $request->sub_total_excluding_gst,
                'sub_total_including_gst' => $request->sub_total_including_gst,
                'total_amount' => $request->total_amount,
                'print_type' => $request->print_type,
            ]);
            Log::info('MyPurchase record updated successfully.');
            // Step 6: Process new and updated items
            Log::info('Processing invoice items. Total items: ' . count($request->invoice_items));
            foreach ($request->invoice_items as $index => $itemData) {
                $product = Product::find($itemData['product_id']);
                if (!$product) {
                    Log::warning("Product with ID {$itemData['product_id']} not found. Skipping.");
                    continue;
                }
                // Get GST details from the product
                $gst_percentage = $product->gst ?? 0;
                $gst_price = ($itemData['purchase_price'] * $gst_percentage) / 100;
                $cgst = $gst_price / 2;
                $sgst = $gst_price / 2;
                // Update or create the MyPurchaseItem
                if (isset($itemData['id']) && in_array($itemData['id'], $existingItemIds)) {
                    // This is an existing item, update it
                    MyPurchaseItem::where('id', $itemData['id'])->update([
                        'product_id' => $itemData['product_id'],
                        'quantity' => $itemData['quantity'],
                        'purchase_price' => $itemData['purchase_price'],
                        'sell_price' => $itemData['sell_price'],
                        'gst_percentage' => $gst_percentage,
                        // 'gst_price' => $gst_price,
                        'cgst' => $cgst,
                        'sgst' => $sgst,
                    ]);
                    $updatedItemIds[] = $itemData['id'];
                    Log::info("Updated existing MyPurchaseItem with ID {$itemData['id']}.");
                } else {
                    // This is a new item, create it
                    $newItem = MyPurchaseItem::create([
                        'my_purchase_id' => $purchase->id,
                        'product_id' => $itemData['product_id'],
                        'quantity' => $itemData['quantity'],
                        'purchase_price' => $itemData['purchase_price'],
                        'sell_price' => $itemData['sell_price'],
                        'gst_percentage' => $gst_percentage,
                        // 'gst_price' => $gst_price,
                        'cgst' => $cgst,
                        'sgst' => $sgst,
                    ]);
                    $updatedItemIds[] = $newItem->id;
                    Log::info("Created new MyPurchaseItem with ID {$newItem->id}.");
                }
                // Step 7: Update product stock and prices
                $product->stock += $itemData['quantity'];
                $product->purchase_price = $itemData['purchase_price'];
                $product->sell_price = $itemData['sell_price'];
                $product->save();
                Log::info("Updated product stock for ID {$product->id}. New stock: {$product->stock}.");
            }
            Log::info('All invoice items processed.');
            // Step 8: Delete any items that were removed from the invoice
            $itemsToDelete = array_diff($existingItemIds, $updatedItemIds);
            if (!empty($itemsToDelete)) {
                MyPurchaseItem::destroy($itemsToDelete);
                Log::info('Deleted old MyPurchaseItems with IDs: ' . implode(', ', $itemsToDelete));
            }
            Log::info('Step 9: Committing database transaction.');
            DB::commit();
            Log::info('Database transaction committed successfully.');
            $printUrl = route('invoice_a4_purchase', ['orderId' => $purchase->id]);
            if ($request->print_type === 'thermal') {
                $printUrl = route('invoice_thermal_purchase', ['orderId' => $purchase->id]);
            }
            Log::info('Step 10: Returning success response. Invoice ID: ' . $purchase->id);
            return response()->json([
                'success' => true,
                'message' => 'Purchase invoice updated successfully.',
                'invoice_id' => $purchase->id,
                'print_url' => $printUrl
            ], 200);
        } catch (\Exception $e) {
            Log::error('An error occurred. Rolling back database transaction.', ['error_message' => $e->getMessage(), 'trace' => $e->getTraceAsString()]);
            DB::rollBack();
            return response()->json([
                'success' => false,
                'message' => 'Failed to update purchase invoice.',
                'error' => $e->getMessage()
            ], 500);
        }
    }
}
