<?php

namespace App\Http\Controllers\Bank;

use App\Http\Controllers\Controller;
use App\Models\BankDetails;
use App\Models\ContactUs;
use App\Models\IncomeWallet;
use App\Models\CashWalletIncome;
use App\Models\shopWalletIncome;
use App\Models\WithdrawRequest;
use Illuminate\Support\Facades\Validator;
use Illuminate\Http\Request;
use App\Models\Bank;
use Carbon\Carbon;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\File;

class BankController extends Controller
{
    public function add_bank(Request $request)
    {
        $user = Auth::user();

        $validator = Validator::make($request->all(), [
            'account_holder_name' => 'required|string|max:255',
            'ifsc_code' => 'required|string|max:11',
            'account_number' => 'required|string|max:20',
            'branch' => 'required|string|max:255',
            'bank_name' => 'required|exists:banks,name',

        ]);

        if ($validator->fails()) {
            return response()->json([
                'status' => 'error',
                'message' => 'Validation errors occurred.',
                'data' => $validator->errors(),
                'code' => 422,
            ], 422);
        }

        BankDetails::where('user_id', $user->user_id)->update(['is_active' => 0]);

        $bankDetail = BankDetails::create([
            'user_id' => $user->user_id,
            'account_holder_name' => $request->account_holder_name,
            'ifsc_code' => $request->ifsc_code,
            'account_number' => $request->account_number,
            'branch' => $request->branch,
            'bank_name' => $request->bank_name,
            'type' => 'user',
            'is_active' => 1,

        ]);

        return response()->json([
            'success' => true,
            'message' => 'Bank details added successfully',
            'data' => $bankDetail
        ], 201);
    }
    public function update_bank(Request $request, $id)
    {
        $validator = Validator::make($request->all(), [
            'account_holder_name' => 'required|string|max:255',
            'bank_name' => 'required|string|max:255',
            'branch' => 'required|string|max:255',
            'ifsc_code' => 'required|string|max:255',
            'account_number' => 'required|string|max:255',
            'upi_id' => 'nullable|string|max:255',
            'image' => 'nullable|image|mimes:jpeg,png,jpg,gif|max:2048',
        ]);

        if ($validator->fails()) {
            return redirect()->back()->with('error', $validator->errors()->first());
        }

        $bank = BankDetails::findOrFail($id);

        $bank->account_holder_name = $request->account_holder_name;
        $bank->bank_name = $request->bank_name;
        $bank->branch = $request->branch;
        $bank->ifsc_code = $request->ifsc_code;
        $bank->account_number = $request->account_number;
        $bank->upi_id = $request->upi_id;

        // Image path variable
        $imagePath = null;

        // Handle image upload
        if ($request->hasFile('image')) {
            // Delete old image if it exists
            if ($bank->image && File::exists(public_path($bank->image))) {
                File::delete(public_path($bank->image));
            }

            $file = $request->file('image');
            $fileName = time() . '_' . $file->getClientOriginalName();
            $destinationPath = public_path('assets/images/bank_qr');
            $file->move($destinationPath, $fileName);
            $imagePath = 'assets/images/bank_qr/' . $fileName;

            $bank->image = $imagePath;
        }

        $bank->save();

        return redirect()->route('add_bank')->with('success', 'Bank details added successfully');

    }
    public function get_userbank_details(Request $request)
    {
        $user = Auth::user();
        $bankDetails = BankDetails::where('user_id', $user->user_id)->get();

        return response()->json([
            'success' => true,
            'message' => 'User bank details retrieved successfully',
            'data' => $bankDetails,
        ], 200);
    }
 public function get_activebank_details(Request $request)
{
    $user = Auth::user();

    $bankDetail = BankDetails::where('is_active', 0)
        ->latest()
        ->first();

    if (!$bankDetail) {
        return response()->json([
            'success' => false,
            'message' => 'No active bank details found',
            'data' => null,
        ], 404);
    }

    $bankDetail->image = url($bankDetail->image);

    return response()->json([
        'success' => true,
        'message' => 'Latest active bank detail retrieved successfully',
        'data' => $bankDetail,
    ], 200);
}



    public function help_center()
    {
        $help_center = ContactUs::latest()->first();

        if (!$help_center) {
            return response()->json([
                'success' => false,
                'message' => 'No help_center found',
                'data' => null,
            ], 404);
        }

        return response()->json([
            'success' => true,
            'message' => 'Latest help_center retrieved successfully',
            'data' => $help_center,
        ], 200);
    }


    public function get_user_wallet(Request $request)
    {
        try {
            $user = Auth::user();
            $wallet = IncomeWallet::where('user_id', $user->user_id)->first();

            if (!$wallet) {
                return response()->json([
                    'status' => 'success',
                    'message' => 'Wallet not found.',
                    'data' => [],
                    'code' => 200,
                ], 200);
            }

            return response()->json([
                'status' => 'success',
                'message' => 'Wallet details fetched successfully.',
                'data' => [
                    'user_id' => $user->user_id,
                    'name' => $user->name,
                    'cash_wallet' => number_format($wallet->cash_wallet, 2),
                    'shopping_wallet' => number_format($wallet->shopping_wallet, 2),
                    'withdraw_wallet' => number_format($wallet->withdraw_wallet, 2),
                    'referral_income' => number_format($wallet->referral_income, 2),
                    'is_active' => $wallet->is_active,
                ],
                'code' => 200,
            ], 200);

        } catch (\Exception $e) {
            return response()->json([
                'status' => 'error',
                'message' => 'Failed to fetch wallet details.',
                'error' => $e->getMessage(),
                'code' => 500,
            ], 500);
        }
    }

   

    public function get_wallet_histories()
    {
        $user = Auth::user();

        $cashWalletHistory = CashWalletIncome::where('user_id', $user->user_id)
            ->orderBy('created_at', 'desc')
            ->get()
            ->map(function ($item) {
                return [
                    'id' => $item->id,
                    'user_id' => $item->user_id,
                    'membership_id' => $item->membership_id,
                    'amount' => $item->amount,
                    'remark' => $item->remark,
                    'is_active' => $item->is_active,
                    'type' => $item->type,
                    'description' => $item->description,
                    'created_at' => Carbon::parse($item->created_at)->format('d-m-Y h:i A'),
                    'updated_at' => Carbon::parse($item->created_at)->format('d-m-Y h:i A'),
                ];
            });

        $shoppingWalletHistory = ShopWalletIncome::where('user_id', $user->user_id)
            ->orderBy('created_at', 'desc')
            ->get()
            ->map(function ($item) {
                return [
                    'id' => $item->id,
                    'user_id' => $item->user_id,
                    'membership_id' => $item->membership_id,
                    'amount' => $item->amount,
                    'remark' => $item->remark,
                    'is_active' => $item->is_active,
                    'type' => $item->type,
                    'description' => $item->description,
                    'created_at' => Carbon::parse($item->created_at)->format('d-m-Y h:i A'),
                    'updated_at' => Carbon::parse($item->created_at)->format('d-m-Y h:i A'),
                ];
            });

        return response()->json([
            'status' => true,
            'message' => 'Wallet histories retrieved successfully',
            'data' => [
                'cash_wallet' => $cashWalletHistory,
                'shopping_wallet' => $shoppingWalletHistory
            ],
            'code' => 200
        ]);
    }


    public function add_bank_details(Request $request)
    {
        try {
            $user = Auth::user();
            $validator = Validator::make($request->all(), [
                'account_holder_name' => 'required|string|max:255',
                'ifsc_code'           => 'required|string|max:11',
                'branch'              => 'required|string|max:255',
                'bank_name'           => 'required|string|max:255',
                'upi_id'              => 'required|string|max:255',
                'image'               => 'required|image|mimes:jpeg,png,jpg,gif|max:2048'
            ]);

            if ($validator->fails()) {
                return redirect()->back()->with('error', $validator->errors()->first());
            }

            BankDetails::where('type', 'admin')->update(['is_active' => 0]);
            $imagePath = null;
            if ($request->hasFile('image')) {
                $file = $request->file('image');
                $fileName = time() . '_' . $file->getClientOriginalName();
                $destinationPath = public_path('assets/images/bank_qr');
                $file->move($destinationPath, $fileName);
                $imagePath = 'assets/images/bank_qr/' . $fileName;
            }
            $bankDetail = new BankDetails([
                'user_id'             => $user->user_id,
                'account_holder_name' => $request->account_holder_name,
                'ifsc_code'           => $request->ifsc_code,
                'account_number'      => $request->account_number,
                'branch'              => $request->branch,
                'bank_name'           => $request->bank_name,
                'type'                => 'admin',
                'is_active'           => 1,
                'upi_id'              => $request->upi_id,
                'image'               => $imagePath
            ]);

            $bankDetail->save();

            return redirect()->route('add_bank')->with('success', 'Bank details added successfully');


        } catch (\Exception $e) {
            return redirect()->back()->with('error', 'Something went wrong: ' . $e->getMessage());
        }
    }
    public function get_all_banks()
    {
        $banks = Bank::all();

        return response()->json([
            'status' => true,
            'message' => 'Bank list fetched successfully',
            'data' => $banks
        ]);
    }

    // public function get_shoppingwallet_history()
    // {
    //     $user = Auth::user();

    //     $transactions = shopWalletIncome::where('user_id', $user->user_id)
    //         ->orderBy('created_at', 'asc')
    //         ->get();

    //     $wallet = IncomeWallet::where('user_id', $user->user_id)->first();
    //     if (!$wallet) {
    //         return response()->json([
    //             'status' => false,
    //             'message' => 'Wallet not found',
    //             'data' => [],
    //             'code' => 404
    //         ]);
    //     }

    //     $data = [];

    //     foreach ($transactions as $transaction) {

    //         $adjustedBalance = $transaction->type === 'credit'
    // ? $wallet->shopping_wallet + $transaction->amount
    // : $wallet->shopping_wallet - $transaction->amount;

    //         $data[] = [
    //             'id'            => $transaction->id,
    //             'user_id'       => $transaction->user_id,
    //             'membership_id' => $transaction->membership_id,
    //             'amount'        => $transaction->amount,
    //             'remark'        => $transaction->remark,
    //             'is_active'     => $transaction->is_active,
    //             'type'          => $transaction->type,
    //             'description'   => json_decode($transaction->description, true),
    //             'created_at'    => \Carbon\Carbon::parse($transaction->created_at)
    //                 ->timezone('Asia/Kolkata')
    //                 ->format('d-m-Y h:i A'),
    //             'updated_at' => \Carbon\Carbon::parse($transaction->updated_at)
    //                 ->timezone('Asia/Kolkata')
    //                 ->format('d-m-Y h:i A'),
    //             'balance' => $adjustedBalance
    //         ];
    //     }

    //     return response()->json([
    //         'status' => true,
    //         'message' => 'Shopping wallet history retrieved successfully',
    //         'data' => $data,
    //         'code' => 200
    //     ]);
    // }



    public function get_shoppingwallet_history()
{
    $user = Auth::user();

    // Fetch all shopping wallet transactions ordered by oldest first
    $transactions = shopWalletIncome::where('user_id', $user->user_id)
        ->orderBy('created_at', 'asc')
        ->get();

    // Get user's current wallet info
    $wallet = IncomeWallet::where('user_id', $user->user_id)->first();
    if (!$wallet) {
        return response()->json([
            'status' => false,
            'message' => 'Wallet not found',
            'data' => [],
            'code' => 404
        ]);
    }

    $data = [];

    // Start from current shopping wallet balance
    $balance = $wallet->shopping_wallet;

    // Reverse loop to get actual progressive deduction
    foreach ($transactions->reverse() as $transaction) {
        // Save current balance before changing it
        $data[] = [
            'id'            => $transaction->id,
            'user_id'       => $transaction->user_id,
            'membership_id' => $transaction->membership_id,
            'amount'        => $transaction->amount,
            'remark'        => $transaction->remark,
            'is_active'     => $transaction->is_active,
            'type'          => $transaction->type,
            'description'   => json_decode($transaction->description, true),
            'created_at'    => Carbon::parse($transaction->created_at)->timezone('Asia/Kolkata')->format('d-m-Y h:i A'),
            'updated_at'    => Carbon::parse($transaction->updated_at)->timezone('Asia/Kolkata')->format('d-m-Y h:i A'),
            'balance'       => $balance
        ];

        // Subtract or add amount based on transaction type for next balance
        if ($transaction->type === 'credit') {
            $balance -= $transaction->amount;
        } else {
            $balance += $transaction->amount;
        }
    }

    // Since we reversed the transaction for balance calculation, reverse the data again to maintain order
    $data = array_reverse($data);

    return response()->json([
        'status' => true,
        'message' => 'Shopping wallet history retrieved successfully',
        'data' => $data,
        'code' => 200
    ]);
}


    // public function get_cashwallet_history()
    // {
    //     $user = Auth::user();

    //     $incomeWallet = IncomeWallet::where('user_id', $user->user_id)->first();

    //     if (!$incomeWallet) {
    //         return response()->json([
    //             'status' => false,
    //             'message' => 'Income wallet not found',
    //             'data' => [],
    //             'code' => 404
    //         ]);
    //     }

    //     $transactions = CashWalletIncome::where('user_id', $user->user_id)
    //         ->orderBy('created_at', 'asc')
    //         ->get();

    //     $data = [];

    //     foreach ($transactions as $transaction) {
    //         $data[] = [
    //             'id'            => $transaction->id,
    //             'user_id'       => $transaction->user_id,
    //             'membership_id' => $transaction->membership_id,
    //             'amount'        => $transaction->amount,
    //             'remark'        => $transaction->remark,
    //             'is_active'     => $transaction->is_active,
    //             'type'          => $transaction->type,
    //             'description'   => $transaction->description,
    //             'created_at'    => \Carbon\Carbon::parse($transaction->created_at)
    //                 ->timezone('Asia/Kolkata')
    //                 ->format('d-m-Y h:i A'),
    //             'updated_at' => \Carbon\Carbon::parse($transaction->updated_at)
    //                 ->timezone('Asia/Kolkata')
    //                 ->format('d-m-Y h:i A'),
    //             'balance' => $incomeWallet->cash_wallet
    //         ];
    //     }

    //     return response()->json([
    //         'status'  => true,
    //         'message' => 'Cash wallet history retrieved successfully',
    //         'data'    => $data,
    //         'code'    => 200
    //     ]);
    // }




    public function get_cashwallet_history()
{
    $user = Auth::user();

    // Fetch user's current wallet
    $incomeWallet = IncomeWallet::where('user_id', $user->user_id)->first();

    if (!$incomeWallet) {
        return response()->json([
            'status' => false,
            'message' => 'Income wallet not found',
            'data' => [],
            'code' => 404
        ]);
    }

    // Get all transactions ordered by oldest first
    $transactions = CashWalletIncome::where('user_id', $user->user_id)
        ->orderBy('created_at', 'asc')
        ->get();

    $data = [];

    // Start from current cash_wallet balance
    $balance = $incomeWallet->cash_wallet;

    // Reverse loop for progressive deduction
    foreach ($transactions->reverse() as $transaction) {
        // Store current balance
        $data[] = [
            'id'            => $transaction->id,
            'user_id'       => $transaction->user_id,
            'membership_id' => $transaction->membership_id,
            'amount'        => $transaction->amount,
            'remark'        => $transaction->remark,
            'is_active'     => $transaction->is_active,
            'type'          => $transaction->type,
            'description'   => json_decode($transaction->description, true),
            'created_at'    => Carbon::parse($transaction->created_at)->timezone('Asia/Kolkata')->format('d-m-Y h:i A'),
            'updated_at'    => Carbon::parse($transaction->updated_at)->timezone('Asia/Kolkata')->format('d-m-Y h:i A'),
            'balance'       => $balance
        ];

        // Adjust balance for next loop
        if ($transaction->type === 'credit') {
            $balance -= $transaction->amount;
        } else {
            $balance += $transaction->amount;
        }
    }

    // Reverse data to show oldest first
    $data = array_reverse($data);

    return response()->json([
        'status'  => true,
        'message' => 'Cash wallet history retrieved successfully',
        'data'    => $data,
        'code'    => 200
    ]);
}

   public function delete_bank($id)
{
    $bank = BankDetails::findOrFail($id);
  if ($bank->image) {
    $imagePath = public_path($bank->image);
    if (file_exists($imagePath) && is_writable($imagePath)) {
        unlink($imagePath);
    }
}

    $bank->delete();

    return back()->with('success', 'Bank detail deleted successfully!');
}
public function toggle_bank_status($id)
{
    $bank = BankDetails::findOrFail($id);
    $bank->is_active = !$bank->is_active;
    $bank->save();

    return back()->with('success', 'Status updated successfully!');
}


}




