<?php

namespace App\Http\Controllers\Employee;

use App\Http\Controllers\Controller;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Validator;
use App\Models\Employee;
use Illuminate\Support\Facades\Hash;
use Illuminate\Support\Facades\Crypt;
use Illuminate\Support\Facades\Mail;
use Illuminate\Support\Facades\Log;
use Carbon\Carbon;
use SendinBlue\Client\Configuration;
use SendinBlue\Client\Api\TransactionalEmailsApi;
use GuzzleHttp\Client;

class EmployeeForgetController extends Controller
{
    public function handle(Request $request)
    {
        $action = $request->input('action');
        if ($action === 'send_otp') {
            return $this->sendOtp($request);
        } elseif ($action === 'verify_otp') {
            return $this->verifyOtp($request);
        }
        return redirect()->back()->with('errormessage', 'Invalid action.');
    }

 public function sendOtp(Request $request)
{
    $request->validate([
        'email' => 'required|email',
    ]);

    $employee = Employee::where('email', $request->email)->first();

    if (!$employee) {
        Log::info("OTP request failed - Employee not found for email: {$request->email}");
        return response()->json(['status' => 'error', 'message' => 'Employee not found.'], 404);
    }

    // $otp          = rand(100000, 999999);
   $otp = 123456;

    $encryptedOtp = Crypt::encryptString($otp);
    $expiryTime   = Carbon::now()->addMinutes(5);

    $employee->otp            = $encryptedOtp;
    $employee->otp_expires_at = $expiryTime;
    $employee->save();

    try {
        Log::info("Sending OTP to: {$employee->email}");
        $sendMailResponse = $this->send_mail_api($employee->email, $otp);

        if ($sendMailResponse['status'] === 'success') {
            Log::info("OTP sent successfully to {$employee->email}");
            return response()->json([
                'status'  => 'success',
                'message' => 'OTP sent successfully.',
                'email'   => $employee->email
            ]);
        } else {
            Log::info("OTP send failed for {$employee->email}: " . $sendMailResponse['error']);
            return response()->json([
                'status'  => 'error',
                'message' => 'Failed to send OTP. Please try again later.'
            ], 500);
        }
    } catch (\Exception $e) {
        Log::info('Exception while sending OTP', ['email' => $employee->email, 'error' => $e->getMessage()]);
        return response()->json([
            'status'  => 'error',
            'message' => 'An error occurred. Please try again later.'
        ], 500);
    }
}

public function send_mail_api($email, $otp)
{
    $apiKey      = env('BREVO_API_KEY');
    $config      = Configuration::getDefaultConfiguration()->setApiKey('api-key', $apiKey);
    $apiInstance = new TransactionalEmailsApi(new Client(), $config);

    $sendSmtpEmail = new \SendinBlue\Client\Model\SendSmtpEmail([
        'subject'     => 'Your Employee OTP Code',
        'htmlContent' => '
<div style="font-family: Arial, sans-serif; padding: 20px; background-color: #f9f9f9; color: #333;">
    <div style="max-width: 600px; margin: auto; background-color: #ffffff; border-radius: 8px; padding: 30px; box-shadow: 0 0 10px rgba(0,0,0,0.05);">
        <h2 style="color: #2c3e50; text-align: center;">Your One-Time Password (OTP)</h2>
        <p>Hello,</p>
        <p>Use the following OTP to complete your password reset process. This code is valid for 5 minutes:</p>
        <div style="text-align: center; margin: 20px 0;">
            <span style="font-size: 28px; font-weight: bold; background: #f0f0f0; padding: 10px 20px; border-radius: 6px; display: inline-block; letter-spacing: 2px; color: #34495e;">' . $otp . '</span>
        </div>
        <p style="margin-top: 30px;">If you did not request this OTP, please ignore this email or contact support.</p>
        <p>Thank you,<br><strong>Adbazzar Team</strong></p>
    </div>
</div>',
        'sender'      => ['name' => 'Adbazzar', 'email' => 'smartworld2025@gmail.com'],
        'to'          => [['email' => $email]],
    ]);

    try {
        $apiInstance->sendTransacEmail($sendSmtpEmail);
        return ['status' => 'success'];
    } catch (\Exception $e) {
        Log::info('Failed to send email via Brevo', ['email' => $email, 'error' => $e->getMessage()]);
        return ['status' => 'error', 'error' => $e->getMessage()];
    }
}

public function verifyOtp(Request $request)
{
    $request->validate([
        'email'    => 'required|email',
        'send_otp'      => 'required|string|min:6|max:6',
    ]);

    $employee = Employee::where('email', $request->email)
        ->where('otp_expires_at', '>', Carbon::now())
        ->first();

    if (!$employee) {
        return response()->json(['status' => 'error', 'message' => 'The OTP is invalid or has expired.'], 400);
    }

    try {
        if (Crypt::decryptString($employee->otp) !== $request->send_otp) {
            return response()->json(['status' => 'error', 'message' => 'The OTP is incorrect.'], 400);
        }

         return redirect()->route('employee.create_password', [
            'email' => $employee->email,
        ])->with('success', 'OTP verified successfully.');

    } catch (\Exception $e) {
        Log::error('Error verifying OTP', [
            'email' => $request->email,
            'error' => $e->getMessage(),
        ]);
        return back()->withErrors(['send_otp' => 'Something went wrong! Please try again.']);
    }
}
 public function create_password($email)
    {
        return view('employee.create_password', compact('email'));
    }
    public function reset_password_update(Request $request)
    {
    
        $validated = $request->validate([
            'email'                 => 'required|email',
            'password'              => 'required|string|min:8|confirmed',
            'password_confirmation' => 'required|string|min:8'
        ]);
    
        $Employee = Employee::where('email', $request->email)->first();
    
        if (!$Employee) {
            return redirect()->back()->withErrors(['email' => 'Employee not found.']);
        }
    
        $Employee->password = Hash::make($request->password);
        $Employee->save();
    
    
        return redirect()->route('employee.login')->with('success', 'Password has been successfully reset.');
    }

    
    // public function forget_update(Request $request)
    // {
        
          

    //         $validator = Validator::make($request->all(), 
    //         [
    //             'password'              => 'required',
    //             'password_confirmation' => 'required|same:password',
    //         ]);

    //         if ($validator->fails()) {
    //             return redirect()->back()->with('errormessage', $validator->errors());
    //         }
           
    //         $forgot_password = Employee::where('email',$request->email)->where('forgot_token', $request->forgot_token)->first();
            
    //         if (!$forgot_password) {
    //             return redirect()->back()->with('errormessage', 'SomeThing Went Wrong! Try Again.');
    //         }

           
    //         $forgot_password->password = Hash::make($request->password);
    //         $forgot_password->forgot_token = "";
    //         $forgot_password->save();

    //     return redirect()->route('admin.sucess_password')->with('success', 'Password has been successfully reset.');
           
    //     }
public function forgot_password()
    {
        return view('employee.forgot_password');
    }

}
