<?php
namespace App\Http\Controllers\Employee;
use App\Http\Controllers\Controller;
use App\Models\User;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Facades\Hash;
use Illuminate\Support\Facades\Crypt;
use App\Mail\OtpMail;
use Carbon\Carbon;
use Illuminate\Support\Facades\Mail;
use Illuminate\Support\Facades\Validator;
use App\Models\Employee;
use App\Models\Branch;
class EmployeeLoginController extends Controller
{
    public function employeedashboard()
    {
        return view('employee.dashboard');
    }
    public function employee_delete($id)
{
    $employee = Employee::findOrFail($id);
    $employee->delete();
    return redirect()->back()->with('success', 'Employee deleted successfully!');
}
public function edit_branch()
{
   $admins = User::whereNotIn('role', ['user', 'admin'])->get();
    return view('admin.edit_branch', compact('admins'));
}
    public function employee_list()
    {
        $employees = Employee::all();
        return view('admin.employee_list', compact('employees'));
    }
    public function view_employee($id)
    {
        $employee = Employee::findOrFail($id);
        return view('admin.view_employee', compact('employee'));
    }
    public function profile()
    {
        $user = Auth::guard('employee')->user();
        Log::info($user);
        return view('employee.profile', compact('user'));
    }
    public function reset_password()
    {
        return view('employee.reset_password');
    }
    public function reset_password_handle(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'password' => ['required', 'string', 'min:6'],
            'new_pwd' => ['required', 'string', 'min:6', 'confirmed', 'regex:/[!@#$%^&*(),.?":{}|<>]/'],
            'new_pwd_confirmation' => ['required', 'string', 'min:6'],
        ]);
        if ($validator->fails()) {
            return redirect()->back()
                ->with('error', $validator->errors()->first());
        }
        $user = Auth::guard('employee')->user();
        if (!Hash::check($request->input('password'), $user->password)) {
            return redirect()->back()
                ->withErrors(['password' => 'The old password is incorrect.'])
                ->withInput();
        }
        $user->password = Hash::make($request->input('new_pwd'));
        $user->save();
        return redirect()->route('employee.dashboard')
            ->with('success', 'Password has been updated successfully.');
    }
    public function profile_update(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'name' => 'required|string|max:255',
            'email' => 'required|email|max:255',
            'phone_number' => 'required|regex:/^[0-9]{10}$/',
        ]);
        if ($validator->fails()) {
            return redirect()->back()->with('error', $validator->errors()->first());
        }
        $user = Auth::guard('employee')->user();
        // Update user fields
        $user->name = $request->name;
        $user->email = $request->email;
        $user->phone = $request->phone_number;
        $user->save();
        return redirect()->route('employee.profile')->with('success', 'Profile updated successfully!');
    }
    public function employee_logout(Request $request)
    {
        Auth::guard('employee')->logout();
        $request->session()->invalidate();
        $request->session()->regenerateToken();
        return redirect()->route('employee.login');
    }
    public function employee_login_sendotp(Request $request)
    {
        try {
            $request->validate([
                'email' => 'required|email',
                'password' => 'required',
            ]);
            $employee = Employee::where('email', $request->input('email'))->first();
            if (!$employee || !Hash::check($request->input('password'), $employee->password)) {
                return response()->json(['status' => 'error', 'message' => 'Invalid email or password']);
            }
            // ✅ Fetch branch
            $branch = Branch::where('branch_name', $employee->branch)->first();
            if (!$branch) {
                return response()->json(['status' => 'error', 'message' => 'Branch not found for this employee']);
            }
            // ✅ OTP generation
            $otp = '123456'; // for testing; replace with rand(100000, 999999) in production
            $employee->otp = Crypt::encryptString($otp);
            $employee->otp_expires_at = now()->addMinutes(1);
            $employee->save();
            return response()->json([
                'status' => 'success',
                'message' => 'OTP sent successfully',
                'branch' => $branch->branch_name // optional, to confirm in UI
            ]);
        } catch (\Exception $e) {
            return response()->json(['status' => 'error', 'message' => 'An error occurred while sending OTP. Please try again.']);
        }
    }
    public function employee_login_verifyotp(Request $request)
    {
        try {
            $request->validate([
                'email' => 'required|email',
                'otp' => 'required|numeric',
                'password' => 'required',
            ]);
            $employee = Employee::where('email', $request->input('email'))->first();
            if (!$employee || !Hash::check($request->input('password'), $employee->password)) {
                return response()->json(['status' => 'error', 'message' => 'Invalid credentials']);
            }
            // ✅ Ensure branch exists
            $branch = Branch::where('branch_name', $employee->branch)->first();
            if (!$branch) {
                return response()->json(['status' => 'error', 'message' => 'Branch not found for this employee']);
            }
            // ✅ OTP expiry check
            if (now()->greaterThan($employee->otp_expires_at)) {
                return response()->json(['status' => 'error', 'message' => 'OTP has expired']);
            }
            $decryptedOtp = Crypt::decryptString($employee->otp);
            if ($request->input('otp') != $decryptedOtp) {
                return response()->json(['status' => 'error', 'message' => 'Invalid OTP']);
            }
            // ✅ Clear OTP fields
            $employee->otp = null;
            $employee->otp_expires_at = null;
            $employee->save();
            // ✅ Authenticate
            // Auth::login($employee);
            Auth::guard('employee')->login($employee);
            // ✅ Redirect based on branch (or role)
            $redirectUrl = route('employee.dashboard');
            return response()->json(['status' => 'success', 'redirect_url' => $redirectUrl]);
        } catch (\Exception $e) {
            return response()->json(['status' => 'error', 'message' => 'An error occurred while verifying OTP. Please try again.']);
        }
    }
// public function store_branch_admin(Request $request)
// {
//     try {
//         $validator = Validator::make($request->all(), [
//             'name'          => 'required|string|max:255',
//             'user_id'       => 'required|string|unique:users,user_id',
//              'password'       => 'required|string|unique:users,user_id',
//             'email'         => 'required|email|unique:users,email',
//             'phone_number'  => 'required|string|max:20',
//             'building_name' => 'nullable|string|max:255',
//             'logo'          => 'nullable|image|mimes:jpg,jpeg,png|max:2048',
//         ]);
//         if ($validator->fails()) {
//             return response()->json([
//                 'success' => false,
//                 'errors'  => $validator->errors(),
//             ], 422);
//         }
//         $logoPath = null;
//         if ($request->hasFile('logo')) {
//             $fileName = time() . '_' . $request->file('logo')->getClientOriginalName();
//             $destinationPath = public_path('assets/images/branch_logos');
//             $request->file('logo')->move($destinationPath, $fileName);
//             $logoPath = 'assets/images/branch_logos/' . $fileName;
//         }
//         $user = User::create([
//             'name'          => $request->name,
//             'user_id'       => $request->user_id,
//             'email'         => $request->email,
//             'phone_number'  => $request->phone_number,
//             'building_name' => $request->building_name,
//             'logo'          => $logoPath,
//             'role'          => 'branch_admin',
//             'password'      => Hash::make('password123'), // Default password
//         ]);
//         return response()->json([
//             'success' => true,
//             'message' => 'Branch admin created successfully.',
//             'data'    => $user,
//         ]);
//     } catch (\Exception $e) {
//         Log::error('Store Branch Admin Error: ' . $e->getMessage());
//         return response()->json([
//             'success' => false,
//             'message' => 'Something went wrong while creating the branch admin.',
//         ], 500);
//     }
// }
// public function update_branch_admin(Request $request, $id)
// {
//     try {
//         $user = User::findOrFail($id);
//         $validator = Validator::make($request->all(), [
//             'name'          => 'required|string|max:255',
//             'email'         => 'required|email',
//             'phone_number'  => 'required|string|max:20',
//             'building_name' => 'nullable|string|max:255',
//             'logo'          => 'nullable|image|mimes:jpg,jpeg,png|max:2048',
//         ]);
//         if ($validator->fails()) {
//             return response()->json([
//                 'success' => false,
//                 'errors'  => $validator->errors(),
//             ], 422);
//         }
//         // Handle logo upload
//         if ($request->hasFile('logo')) {
//             $fileName = time() . '_' . $request->file('logo')->getClientOriginalName();
//             $destinationPath = public_path('assets/images/branch_logos');
//             $request->file('logo')->move($destinationPath, $fileName);
//             $logoPath = 'assets/images/branch_logos/' . $fileName;
//             // Delete old logo if exists
//             if ($user->logo && file_exists(public_path($user->logo))) {
//                 unlink(public_path($user->logo));
//             }
//             $user->logo = $logoPath;
//         }
//         $user->update([
//             'name'          => $request->name,
//             'email'         => $request->email,
//             'phone_number'  => $request->phone_number,
//             'building_name' => $request->building_name,
//             'logo'          => $user->logo,
//         ]);
//         return response()->json([
//             'success' => true,
//             'message' => 'Branch admin updated successfully.',
//         ]);
//     } catch (\Exception $e) {
//         \Log::error('Update Branch Admin Error: ' . $e->getMessage());
//         return response()->json([
//             'success' => false,
//             'message' => 'Something went wrong while updating.',
//         ], 500);
//     }
// }
 public function store_branch_admin(Request $request)
    {
        try {
            $validator = Validator::make($request->all(), [
                'name'          => 'required|string|max:255',
                'user_id'       => 'required|string|unique:users,user_id',
                'email'         => 'required|email|unique:users,email',
                'password'      => 'required|string|min:6',
                'phone_number'  => 'required|string|max:20',
                'building_name' => 'nullable|string|max:255',
                'logo'          => 'nullable|image|mimes:jpg,jpeg,png|max:2048',
            ]);
            if ($validator->fails()) {
                return response()->json([
                    'success' => false,
                    'errors'  => $validator->errors(),
                ], 422);
            }
            $logoPath = null;
            if ($request->hasFile('logo')) {
                $fileName = time() . '_' . $request->file('logo')->getClientOriginalName();
                $destinationPath = public_path('assets/images/branch_logos');
                $request->file('logo')->move($destinationPath, $fileName);
                $logoPath = 'assets/images/branch_logos/' . $fileName;
            }
            $user = User::create([
                'name'          => $request->name,
                'user_id'       => $request->user_id,
                'email'         => $request->email,
                'phone_number'  => $request->phone_number,
                'building_name' => $request->building_name,
                'logo'          => $logoPath,
                'role'          => 'branch_admin',
                'password'      => Hash::make($request->password),
                 'password_hint'=> $request->password,
            ]);
            return response()->json([
                'success' => true,
                'message' => 'Branch admin created successfully.',
                'data'    => $user,
            ]);
        } catch (\Exception $e) {
            Log::error('Store Branch Admin Error: ' . $e->getMessage());
            return response()->json([
                'success' => false,
                'message' => 'Something went wrong while creating the branch admin.',
            ], 500);
        }
    }
public function update_branch_admin(Request $request, $id)
{
    \Log::info("🔄 Branch Admin Update Request Received", [
        'branch_admin_id' => $id,
        'input' => $request->all()
    ]);
    try {
        $user = User::findOrFail($id);
        \Log::info("👤 Branch Admin Found", ['id' => $id,'name'=>$user->name]);
        // FIXED VALIDATION ✔
        $validator = Validator::make($request->all(), [
            'name'          => 'required|string|max:255',
            'email'         => 'required|email|unique:users,email,' . $id,
            'phone_number'  => 'required|string|max:20',
            'building_name' => 'nullable|string|max:255',
            'password'      => 'nullable|string|min:6',
            
            'logo'          => 'nullable', // <-- REAL FIX
        ]);
        if ($validator->fails()) {
            \Log::warning("⚠ Validation Failed", ['id' => $id, 'errors' => $validator->errors()]);
            return response()->json([
                'success' => false,
                'errors'  => $validator->errors(),
            ], 422);
        }
        // Upload logo only if selected
        if ($request->hasFile('logo')) {
            \Log::info("📁 Logo Upload Started",['id'=>$id]);
            $fileName = time().'_'.$request->logo->getClientOriginalName();
            $path = 'assets/images/branch_logos/'.$fileName;
            $request->logo->move(public_path('assets/images/branch_logos'),$fileName);
            if ($user->logo && file_exists(public_path($user->logo))) {
                unlink(public_path($user->logo));
                \Log::info("🗑 Old Logo Deleted",['old_logo'=>$user->logo]);
            }
            $user->logo = $path;
            \Log::info("✔ New Logo Saved",['new_logo'=>$path]);
        }
        // Update other fields
        $user->name          = $request->name;
        $user->email         = $request->email;
        $user->phone_number  = $request->phone_number;
        $user->building_name = $request->building_name;
        $user->password_hint = $request->password;
        // Update password only if new entered
        if ($request->filled('password')) {
            $user->password = Hash::make($request->password);
            \Log::info("🔐 Password Updated",['id'=>$id]);
        }
        $user->save();
        \Log::info("🎉 Branch Admin Updated Successfully",['id'=>$id]);
        return response()->json([
            'success'=>true,
            'message'=>'Branch admin updated successfully.'
        ]);
    } catch (\Exception $e) {
        \Log::error("❌ Update Error: ".$e->getMessage(),['id'=>$id]);
        return response()->json([
            'success'=>false,
            'message'=>'Something went wrong.'
        ],500);
    }
}
    // public function update_branch_admin(Request $request, $id)
    // {
    //     try {
    //         $user = User::findOrFail($id);
    //         $validator = Validator::make($request->all(), [
    //             'name'          => 'required|string|max:255',
    //             'email'         => 'required|email|unique:users,email,' . $id,
    //             'phone_number'  => 'required|string|max:20',
    //             'building_name' => 'nullable|string|max:255',
    //             'password'      => 'nullable|string|min:6',
    //             'logo'          => 'nullable|image|mimes:jpg,jpeg,png|max:2048',
    //         ]);
    //         if ($validator->fails()) {
    //             return response()->json([
    //                 'success' => false,
    //                 'errors'  => $validator->errors(),
    //             ], 422);
    //         }
    //         if ($request->hasFile('logo')) {
    //             $fileName = time() . '_' . $request->file('logo')->getClientOriginalName();
    //             $destinationPath = public_path('assets/images/branch_logos');
    //             $request->file('logo')->move($destinationPath, $fileName);
    //             $logoPath = 'assets/images/branch_logos/' . $fileName;
    //             if ($user->logo && file_exists(public_path($user->logo))) {
    //                 unlink(public_path($user->logo));
    //             }
    //             $user->logo = $logoPath;
    //         }
    //         $user->name = $request->name;
    //         $user->email = $request->email;
    //         $user->phone_number = $request->phone_number;
    //          $user->password_hint = $request->password;
    //         $user->building_name = $request->building_name;
    //         if (!empty($request->password)) {
    //             $user->password = Hash::make($request->password);
    //         }
    //         $user->save();
    //         return response()->json([
    //             'success' => true,
    //             'message' => 'Branch admin updated successfully.',
    //         ]);
    //     } catch (\Exception $e) {
    //         \Log::error('Update Branch Admin Error: ' . $e->getMessage());
    //         return response()->json([
    //             'success' => false,
    //             'message' => 'Something went wrong while updating.',
    //         ], 500);
    //     }
    // }
}
