<?php
namespace App\Http\Controllers\Employee;
use App\Http\Controllers\Controller;
use App\Models\User;
use App\Models\Product;
use App\Models\ProductVarient;
use App\Models\OfflineSale;
use Illuminate\Support\Str;
use Illuminate\Support\Facades\DB;
use Illuminate\Validation\ValidationException;
use Illuminate\Support\Facades\Log;
use App\Models\IncomeWallet;
use App\Models\shopWalletIncome;
use App\Models\CashWalletIncome;
use Illuminate\Http\Request;
use App\Models\Category;
use App\Models\OrderDetails;
use App\Models\Branch;
use Carbon\Carbon;
class EmployeePosController extends Controller
{
    public function employee_login()
    {
        return view('employee.login');
    }
public function registerNewCustomer(Request $request)
    {
        try {
            $validated = $request->validate([
                'phone_number' => 'required|string|unique:users,phone_number',
                'name' => 'required|string|max:255',
                'email' => 'nullable|email|max:255|unique:users,email',
            ]);
            DB::beginTransaction();
            // Generate sequential user_id like AD2025, AD2026, etc.
            $latestUser = User::where('user_id', 'LIKE', 'AD%')
                ->orderByDesc('user_id')
                ->first();
            if ($latestUser && preg_match('/AD(\d+)/', $latestUser->user_id, $matches)) {
                $nextNumber = intval($matches[1]) + 1;
            } else {
                $nextNumber = 2025;
            }
            do {
                $user_id = 'AD' . $nextNumber;
                $nextNumber++;
            } while (User::where('user_id', $user_id)->exists());
            $user = User::create([
                'user_id' => $user_id,
                'name' => $validated['name'],
                'phone_number' => $validated['phone_number'],
                'email' => $validated['email'] ?? null,
                'password' => bcrypt(Str::random(10)),
                'user_type' => 'customer',
                'is_active' => true,
            ]);
            // Create wallet entry
            IncomeWallet::create([
                'user_id' => $user->user_id,
                'cash_wallet' => 0.00,
                'shopping_wallet' => 0.00,
                'withdraw_wallet' => 0.00,
                'referral_income' => 0.00,
                'cash_wallet_deduction' => 0.00,
                'shopping_wallet_deduction' => 0.00,
                'is_active' => true,
            ]);
            DB::commit();
            return response()->json([
                'message' => 'Customer registered successfully!',
                'user_id' => $user->user_id,
                'name' => $user->name,
                'phone_number' => $user->phone_number,
                'email' => $user->email,
                'address' => 'N/A',
                'cash_wallet_balance' => 0.00,
                'shopping_wallet_balance' => 0.00,
            ], 201);
        } catch (ValidationException $e) {
            DB::rollBack();
            return response()->json(['errors' => $e->errors()], 422);
        } catch (\Exception $e) {
            DB::rollBack();
            Log::error('New Customer Registration Error', [
                'message' => $e->getMessage(),
                'file' => $e->getFile(),
                'line' => $e->getLine(),
                'trace' => $e->getTraceAsString(),
            ]);
            return response()->json([
                'message' => 'An error occurred during customer registration.',
                'error' => $e->getMessage(),
            ], 500);
        }
    }
    public function employeeoffline_sales_history()
    {
        $user = auth('employee')->user();
        Log::info($user);
        $branch = $user->branch;
        Log::info($branch);
        $offlineSales = OfflineSale::with('user')
            ->where('branch', $branch)
            ->whereDate('created_at', Carbon::today())
            ->latest()
            ->get();
        return view('employee.offline_sales_history', compact('offlineSales', 'branch'));
    }
    //         public function employeeoffline_sales_history()
// {
//     $user = auth()->user();
//     // Determine branch based on user role
//     $branch = null;
//     if ($user->role === 'sakthi') {
//         $branch = 'Sakthi Super Mart';
//     } elseif ($user->role === 'smart_world') {
//         $branch = 'SMARTWORLD';
//     } elseif ($user->role === 'smart_world_gift') {
//         $branch = 'SMARTWORLD - Stationary & Gifts';
//     } elseif ($user->role === 'yoga_naturals') {
//         $branch = 'yoga naturals';
//     }
//     // Fetch offline sales filtered by branch if applicable
//     $offlineSales = OfflineSale::with('user')
//         ->when($branch, function ($query) use ($branch) {
//             return $query->where('branch', $branch);
//         })
//         ->latest()
//         ->get();
//     return view('employee.offline_sales_history', compact('offlineSales'));
// }
// public function employeeoffline_sales_history()
// {
//     $user = auth('employee')->user(); // ✅ Correct guard
//     Auth::guard('employee')->login($employee);
//     // Determine branch based on user role
//     $branch = null;
//     if ($user->role === 'sakthi') {
//         $branch = 'Sakthi Super Mart';
//     } elseif ($user->role === 'smart_world') {
//         $branch = 'SMARTWORLD';
//     } elseif ($user->role === 'smart_world_gift') {
//         $branch = 'SMARTWORLD - Stationary & Gifts';
//     } elseif ($user->role === 'yoga_naturals') {
//         $branch = 'yoga naturals';
//     }
//     // Fetch offline sales filtered by branch
//     $offlineSales = OfflineSale::with('user')
//         ->when($branch, function ($query) use ($branch) {
//             return $query->where('branch', $branch);
//         })
//         ->latest()
//         ->get();
//     return view('employee.offline_sales_history', compact('offlineSales'));
// }
    public function employeepos_billing()
    {
        return view('employee.pos_billing');
    }
    public function employeesearchCustomer(Request $request)
    {
        $identifier = $request->input('identifier');
        $trimmedIdentifier = trim($identifier);
        if (empty($trimmedIdentifier)) {
            return response()->json(null);
        }
        $user = User::where('user_id', $trimmedIdentifier)
            ->orWhere('phone_number', $trimmedIdentifier)
            ->with(['incomeWallet', 'currentAddress'])
            ->first();
        if ($user) {
            $customerDetails = [
                'name' => $user->name,
                'user_id' => $user->user_id,
                'phone_number' => $user->phone_number,
                'email' => $user->email,
                'address' => 'N/A',
                'cash_wallet_balance' => $user->incomeWallet ? (float) $user->incomeWallet->cash_wallet : 0.00,
                'shopping_wallet_balance' => $user->incomeWallet ? (float) $user->incomeWallet->shopping_wallet : 0.00,
            ];
            if ($user->currentAddress) {
                $addressParts = array_filter([
                    $user->currentAddress->house_no ?? null,
                    $user->currentAddress->building_name ?? null,
                    $user->currentAddress->road_name ?? null,
                    $user->currentAddress->landmark ?? null,
                    $user->currentAddress->district ?? null,
                    $user->currentAddress->state ?? null,
                    $user->currentAddress->pin ?? null
                ]);
                $customerDetails['address'] = trim(implode(', ', $addressParts));
            }
            return response()->json($customerDetails);
        }
        return response()->json(null);
    }
    public function employeesearchProducts(Request $request)
    {
        $query = $request->input('query');
        if (empty($query)) {
            return response()->json([]);
        }
        $products = Product::where('title', 'LIKE', '%' . $query . '%')
            ->orWhere('item_code', 'LIKE', '%' . $query . '%')
            ->orWhere('product_id', 'LIKE', '%' . $query . '%')
            ->orWhere('hsn_code', 'LIKE', '%' . $query . '%')
            ->orWhere('barcode1', 'LIKE', '%' . $query . '%')
            ->orWhere('barcode2', 'LIKE', '%' . $query . '%')
            ->select(
                'id',
                'product_id',
                'title as name',
                'item_code',
                'mrp',
                'sell_price',
                'purchase_price',
                'stock',
                'gst',
                'cgst',
                'sgst',
                'gst_price',
                'hsn_code',
                'barcode1',
                'barcode2',
                'shop_wallet'
            )
            ->get()
            ->map(function ($product) {
                return (object) [
                    'type' => 'product',
                    'item_identifier' => (string) $product->product_id,
                    'name' => $product->name,
                    'item_code' => $product->item_code,
                    'mrp' => (float) ($product->mrp ?? 0),
                    'sell_price' => (float) ($product->sell_price ?? 0),
                    'purchase_price' => (float) ($product->purchase_price ?? 0),
                    'gst' => (float) ($product->gst ?? 0),
                    'cgst' => (float) ($product->cgst ?? 0),
                    'sgst' => (float) ($product->sgst ?? 0),
                    'gst_price' => (float) ($product->gst_price ?? 0),
                    'stock' => (int) ($product->stock ?? 0),
                    'hsn_code' => $product->hsn_code ?? $product->item_code,
                    'barcode1' => $product->barcode1,
                    'barcode2' => $product->barcode2,
                    'shopping_wallet_price' => (float) ($product->shop_wallet ?? 0),
                ];
            });
        $variants = ProductVarient::where('variant_name', 'LIKE', '%' . $query . '%')
            ->orWhereHas('product', function ($q) use ($query) {
                $q->where('title', 'LIKE', '%' . $query . '%')
                    ->orWhere('item_code', 'LIKE', '%' . $query . '%')
                    ->orWhere('product_id', 'LIKE', '%' . $query . '%')
                    ->orWhere('hsn_code', 'LIKE', '%' . $query . '%')
                    ->orWhere('barcode1', 'LIKE', '%' . $query . '%')
                    ->orWhere('barcode2', 'LIKE', '%' . $query . '%');
            })
            ->with('product:id,product_id,title,item_code,stock,mrp,sell_price,purchase_price,gst,cgst,sgst,gst_price,hsn_code,barcode1,barcode2,shop_wallet')
            ->get()
            ->map(function ($variant) {
                if (!$variant->product)
                    return null;
                return (object) [
                    'type' => 'variant',
                    'item_identifier' => (string) $variant->variant_id,
                    'product_id_main' => (string) $variant->product->product_id,
                    'name' => $variant->product->title . ' (' . $variant->variant_name . ')',
                    'item_code' => $variant->product->item_code . '-' . $variant->variant_id,
                    'mrp' => (float) ($variant->mrp ?? $variant->product->mrp ?? 0),
                    'sell_price' => (float) ($variant->sell_price ?? $variant->product->sell_price ?? 0),
                    'purchase_price' => (float) ($variant->purchase_price ?? $variant->product->purchase_price ?? 0),
                    'gst' => (float) ($variant->gst ?? $variant->product->gst ?? 0),
                    'cgst' => (float) ($variant->cgst ?? $variant->product->cgst ?? 0),
                    'sgst' => (float) ($variant->sgst ?? $variant->product->sgst ?? 0),
                    'gst_price' => (float) ($variant->gst_price ?? $variant->product->gst_price ?? 0),
                    'stock' => (int) ($variant->stock ?? $variant->product->stock ?? 0),
                    'hsn_code' => $variant->product->hsn_code ?? $variant->product->item_code,
                    'barcode1' => $variant->product->barcode1 ?? null,
                    'barcode2' => $variant->product->barcode2 ?? null,
                    'shopping_wallet_price' => (float) ($variant->shop_wallet ?? $variant->product->shop_wallet ?? 0),
                ];
            })->filter();
        return response()->json($products->merge($variants)->values());
    }
    public function employeesaveBill(Request $request)
    {
        try {
            $validated = $request->validate([
                'order_id' => 'nullable|string',
                'user_id' => 'nullable|string|exists:users,user_id',
                'customer_details' => 'nullable|array',
                'bill_items' => 'required|array',
                'bill_items.*.item_identifier' => 'required|string',
                'bill_items.*.product_id_main' => 'nullable|string',
                'bill_items.*.type' => 'required|in:product,variant',
                'bill_items.*.name' => 'required|string',
                'bill_items.*.item_code' => 'required|string',
                'bill_items.*.mrp' => 'required|numeric|min:0',
                'bill_items.*.sell_price' => 'required|numeric|min:0',
                'bill_items.*.purchase_price' => 'required|numeric|min:0',
                'bill_items.*.quantity' => 'required|integer|min:1',
                'bill_items.*.gst' => 'required|numeric|min:0',
                'bill_items.*.cgst' => 'required|numeric|min:0',
                'bill_items.*.sgst' => 'required|numeric|min:0',
                'bill_items.*.gst_price' => 'required|numeric|min:0',
                'bill_items.*.shopping_wallet_price' => 'required|numeric|min:0',
                'sub_total' => 'required|numeric|min:0',
                'discount_amount' => 'required|numeric|min:0',
                'additional_charge' => 'required|numeric|min:0',
                'total_amount' => 'required|numeric|min:0',
                'use_cash_wallet' => 'boolean',
                'use_shopping_wallet' => 'boolean',
                'cash_wallet_deduction' => 'required|numeric|min:0',
                'shopping_wallet_deduction' => 'required|numeric|min:0',
                'amount_payable' => 'required|numeric|min:0',
                'payment_mode' => 'required|string|in:cash,card,upi',
                'payment_received' => 'required|numeric|min:0',
                'change_amount' => 'nullable|numeric',
                'status' => 'required|string|in:completed,held',
                'description' => 'nullable|string|max:255',
            ]);
            DB::beginTransaction();
            $user = auth()->user();
            // Determine branch
            $branch = match ($user->role) {
                'sakthi' => 'Sakthi Super Mart',
                'smart_world' => 'SMARTWORLD',
                'smart_world_gift' => 'SMARTWORLD - Stationary & Gifts',
                'yoga_naturals' => 'yoga naturals',
                default => 'Unknown Branch',
            };
            $orderId = $validated['order_id'];
            $offlineSale = null;
            if ($orderId) {
                $offlineSale = OfflineSale::where('order_id', $orderId)->first();
            }
            if (!$offlineSale) {
                $branchPrefix = match ($user->role) {
                    'sakthi' => 'SS',
                    'smart_world' => 'smart',
                    'smart_world_gift' => 'gift',
                    'yoga_naturals' => 'yoga',
                    default => 'store',
                };
                $lastOrder = OfflineSale::where('order_id', 'like', "{$branchPrefix}%")
                    ->latest('id')
                    ->first();
                $newOrderIdNumber = 101;
                if ($lastOrder && preg_match("/^{$branchPrefix}(\d+)$/", $lastOrder->order_id, $matches)) {
                    $newOrderIdNumber = (int) $matches[1] + 1;
                }
                $orderId = $branchPrefix . $newOrderIdNumber;
                $offlineSale = new OfflineSale();
            }
            // Tax Calculations
            $total_cgst = 0;
            $total_sgst = 0;
            $total_tax = 0;
            $sub_total = 0;
            foreach ($validated['bill_items'] as $item) {
                $item_total = $item['sell_price'] * $item['quantity'];
                $item_cgst = $item['cgst'] * $item['quantity'];
                $item_sgst = $item['sgst'] * $item['quantity'];
                $item_tax = $item['gst_price'] * $item['quantity'];
                $total_cgst += $item_cgst;
                $total_sgst += $item_sgst;
                $total_tax += $item_tax;
                $sub_total += $item_total;
            }
            // Save Offline Sale
            $offlineSale->user_id = $validated['user_id'] ?? null;
            $offlineSale->order_id = $orderId;
            $offlineSale->sale_items = $validated['bill_items'];
            $offlineSale->total_amount = $validated['total_amount'];
            $offlineSale->cash_wallet_deduction = $validated['cash_wallet_deduction'];
            $offlineSale->shopping_wallet_deduction = $validated['shopping_wallet_deduction'];
            $offlineSale->discount = $validated['discount_amount'];
            $offlineSale->additional_charge = $validated['additional_charge'];
            $offlineSale->amount_payable = $validated['amount_payable'];
            $offlineSale->customer_details = $validated['customer_details'];
            $offlineSale->status = $validated['status'];
            $offlineSale->payment_mode = $validated['payment_mode'];
            $offlineSale->payment_received = $validated['payment_received'];
            $offlineSale->change_amount = $validated['change_amount'];
            $offlineSale->description = $validated['description'] ?? null;
            $offlineSale->branch = $branch;
            $offlineSale->total_tax = $total_tax;
            $offlineSale->tax_value = $total_tax;
            $offlineSale->cgst = $total_cgst;
            $offlineSale->sgst = $total_sgst;
            $offlineSale->sub_total = $sub_total;
            $offlineSale->save();
            // Complete Order Logic
            if ($validated['status'] === 'completed') {
                // Deduct wallet
                if ($offlineSale->user_id) {
                    $user = User::where('user_id', $offlineSale->user_id)->with('incomeWallet')->first();
                    if ($user && $user->incomeWallet) {
                        $incomeWallet = $user->incomeWallet;
                        // Wallet Deduction
                        if ($validated['cash_wallet_deduction'] > 0) {
                            if ($incomeWallet->cash_wallet < $validated['cash_wallet_deduction']) {
                                throw new \Exception('Insufficient cash wallet balance for transaction.');
                            }
                            $incomeWallet->cash_wallet -= $validated['cash_wallet_deduction'];
                        }
                        if ($validated['shopping_wallet_deduction'] > 0) {
                            if ($incomeWallet->shopping_wallet < $validated['shopping_wallet_deduction']) {
                                throw new \Exception('Insufficient shopping wallet balance for transaction.');
                            }
                            $incomeWallet->shopping_wallet -= $validated['shopping_wallet_deduction'];
                        }
                        $incomeWallet->save();
                        // 💾 Save Wallet History
                        if ($validated['shopping_wallet_deduction'] > 0) {
                            ShopWalletIncome::create([
                                'user_id' => $offlineSale->user_id,
                                'membership_id' => null,
                                'amount' => $validated['shopping_wallet_deduction'],
                                'remark' => "POS Purchase : {$offlineSale->order_id}",
                                'description' => "POS Purchase at {$branch}",
                                'payment_mode' => $validated['payment_mode'],
                                'type' => "debit"
                            ]);
                        }
                        if ($validated['cash_wallet_deduction'] > 0) {
                            CashWalletIncome::create([
                                'user_id' => $offlineSale->user_id,
                                'membership_id' => null,
                                'amount' => $validated['cash_wallet_deduction'],
                                'remark' => "POS Purchase : {$offlineSale->order_id}",
                                'description' => "POS Purchase at {$branch}",
                                'payment_mode' => $validated['payment_mode'],
                                'type' => "debit"
                            ]);
                        }
                    } else {
                        if ($validated['cash_wallet_deduction'] > 0 || $validated['shopping_wallet_deduction'] > 0) {
                            Log::warning("User or wallet not found for deduction (user_id: {$offlineSale->user_id})");
                            throw new \Exception('User or wallet not found for wallet deductions.');
                        }
                    }
                }
                // Deduct stock
                foreach ($validated['bill_items'] as $item) {
                    if ($item['type'] === 'product') {
                        $product = Product::where('product_id', $item['item_identifier'])->first();
                        if ($product) {
                            if ($product->stock < $item['quantity']) {
                                throw new \Exception("Insufficient stock for product: {$product->title}");
                            }
                            $product->stock = max(0, $product->stock - $item['quantity']);
                            $product->save();
                        }
                    } else if ($item['type'] === 'variant') {
                        $variant = ProductVarient::where('variant_id', $item['item_identifier'])->first();
                        if ($variant) {
                            if ($variant->stock < $item['quantity']) {
                                throw new \Exception("Insufficient stock for variant: {$variant->variant_name}");
                            }
                            $variant->stock = max(0, $variant->stock - $item['quantity']);
                            $variant->save();
                        }
                        $mainProduct = Product::where('product_id', $item['product_id_main'])->first();
                        if ($mainProduct) {
                            $mainProduct->stock = max(0, $mainProduct->stock - $item['quantity']);
                            $mainProduct->save();
                        }
                    }
                }
            }
            DB::commit();
            return response()->json([
                'message' => 'Bill saved successfully!',
                'order_id' => $offlineSale->order_id,
                'status' => $offlineSale->status
            ]);
        } catch (ValidationException $e) {
            DB::rollBack();
            return response()->json(['errors' => $e->errors()], 422);
        } catch (\Exception $e) {
            DB::rollBack();
            Log::error("POS Bill Save Error: " . $e->getMessage() . " in " . $e->getFile() . " on line " . $e->getLine());
            return response()->json(['message' => $e->getMessage() ?? 'An error occurred while saving the bill. Please try again.'], 500);
        }
    }
    public function employeeholdBill(Request $request)
    {
        $request->merge(['status' => 'held']);
        return $this->employeesaveBill($request);
    }
    public function employeegetHeldBills()
    {
        $user = auth()->user();
        // Determine branch name same way as in employeesaveBill
        $branch = match ($user->role) {
            'sakthi' => 'Sakthi Super Mart',
            'smart_world' => 'SMARTWORLD',
            'smart_world_gift' => 'SMARTWORLD - Stationary & Gifts',
            'yoga_naturals' => 'yoga naturals',
            default => 'Unknown Branch',
        };
        $heldBills = OfflineSale::where('status', 'held')
            ->where('branch', $branch)
            ->orderBy('updated_at', 'desc')
            ->get();
        return response()->json($heldBills);
    }
    public function employeeresumeHeldBill(string $orderId)
    {
        $heldBill = OfflineSale::where('order_id', $orderId)->where('status', 'held')->first();
        if (!$heldBill) {
            return response()->json(['message' => 'Held bill not found or is already completed/deleted.'], 404);
        }
        $customerDetails = null;
        if ($heldBill->user_id) {
            $user = User::where('user_id', $heldBill->user_id)->with('incomeWallet', 'currentAddress')->first();
            if ($user) {
                $customerDetails = [
                    'name' => $user->name,
                    'user_id' => $user->user_id,
                    'phone_number' => $user->phone_number,
                    'email' => $user->email,
                    'address' => 'N/A',
                    'cash_wallet_balance' => $user->incomeWallet ? (float) $user->incomeWallet->cash_wallet : 0.00,
                    'shopping_wallet_balance' => $user->incomeWallet ? (float) $user->incomeWallet->shopping_wallet : 0.00,
                ];
                if ($user->currentAddress) {
                    $addressParts = array_filter([
                        $user->currentAddress->house_no ?? null,
                        $user->currentAddress->building_name ?? null,
                        $user->currentAddress->road_name ?? null,
                        $user->currentAddress->landmark ?? null,
                        $user->currentAddress->district ?? null,
                        $user->currentAddress->state ?? null,
                        $user->currentAddress->pin ?? null
                    ]);
                    $customerDetails['address'] = trim(implode(', ', $addressParts));
                }
            }
        }
        $billItems = collect($heldBill->sale_items)->map(function ($item) {
            $item['item_identifier'] = (string) $item['item_identifier'];
            if (isset($item['product_id_main'])) {
                $item['product_id_main'] = (string) $item['product_id_main'];
            }
            $item['sell_price'] = (float) ($item['sell_price'] ?? 0);
            $item['purchase_price'] = (float) ($item['purchase_price'] ?? 0);
            $item['mrp'] = (float) ($item['mrp'] ?? 0);
            $item['quantity'] = (int) ($item['quantity'] ?? 1);
            $item['gst'] = (float) ($item['gst'] ?? 0);
            $item['cgst'] = (float) ($item['cgst'] ?? 0);
            $item['sgst'] = (float) ($item['sgst'] ?? 0);
            $item['gst_price'] = (float) ($item['gst_price'] ?? 0);
            $item['shopping_wallet_price'] = (float) ($item['shopping_wallet_price'] ?? 0);  // Ensure this is loaded
            return $item;
        })->toArray();
        return response()->json([
            'message' => 'Held bill data retrieved successfully.',
            'bill_data' => [
                'order_id' => $heldBill->order_id,
                'customerIdentifier' => $heldBill->user_id ?? ($heldBill->customer_details['phone_number'] ?? ''),
                'customerDetails' => $customerDetails ?? $heldBill->customer_details,
                'billItems' => $billItems,
                'discountAmount' => (float) $heldBill->discount,
                'additionalCharge' => (float) $heldBill->additional_charge,
                'useCashWallet' => (bool) ($heldBill->cash_wallet_deduction > 0),
                'useShoppingWallet' => (bool) ($heldBill->shopping_wallet_deduction > 0),                           // This will now represent if *additional* shopping wallet was used
                'cashWalletDeduction' => (float) $heldBill->cash_wallet_deduction,
                'shoppingWalletDeduction' => (float) $heldBill->shopping_wallet_deduction,                                // Total deduction
                'paymentMode' => $heldBill->payment_mode,
                'payment_received' => (float) $heldBill->payment_received,
                'description' => $heldBill->description,
            ]
        ]);
    }
    public function employeedeleteHeldBill(string $orderId)
    {
        $deleted = OfflineSale::where('order_id', $orderId)->where('status', 'held')->delete();
        if ($deleted) {
            return response()->json(['message' => 'Held bill deleted successfully.']);
        }
        return response()->json(['message' => 'Held bill not found or could not be deleted.'], 404);
    }
    public function employeeprintA4Invoice(string $orderId)
    {
        $offlineSale = OfflineSale::where('order_id', $orderId)->firstOrFail();
        $user = auth()->user(); 
      
        // $branchInfo = match ($user->role) {
        //     'sakthi' => [
        //         'name' => 'Sakthi Super Mart',
        //         'address' => '615/3 Pandian Street 
        //                      Chennimalai - 638051 ',
        //         'phone' => '+91 9500598595',
        //         'email' => 'admin1@gmail.com',
        //     ],
        //     'smart_world' => [
        //         'name' => 'SMARTWORLD',
        //         'address' => 'Chennai, India',
        //         'phone' => '+91 9443595314',
        //         'email' => 'admin2@gmail.com',
        //     ],
        //     'smart_world_gift' => [
        //         'name' => 'SMARTWORLD - Stationary & Gifts',
        //         'address' => 'Chennai, India',
        //         'phone' => '+91 9443595315',
        //         'email' => 'admin3@gmail.com',
        //     ],
        //     'yoga_naturals' => [
        //         'name' => 'Yoga Naturals',
        //         'address' => 'Chennai, India',
        //         'phone' => '+91 9443595316',
        //         'email' => 'admin4@gmail.com',
        //     ],
        //     default => [
        //         'name' => 'AdBazzar',
        //         'address' => 'Chennai, India',
        //         'phone' => '+91 8531345678',
        //         'email' => 'adbazzar@gmail.com',
        //     ],
        // };
          $branchInfo = [
            'name' => $user->name ?? 'AdBazzar',
            'address' => $user->building_name ?? 'Chennai, India',  
            'phone' => $user->phone_number ?? '+91 8531345678',
            'email' => $user->email ?? 'adbazzar@gmail.com',
            'logo' => !empty($user->logo) ? asset($user->logo) : null,
        ];
        return view('employee.invoice_a4', compact('offlineSale', 'branchInfo'));
    }
//     public function employeeprintThermalInvoice(string $orderId)
//     {
//         $offlineSale = OfflineSale::where('order_id', $orderId)->firstOrFail();
//         $user = auth()->user(); 
//        $branchInfo = [
//     'name'   => $user->name,
//     'address'=> $user->building_name ?? '',       
//     'phone'  => $user->phone_number,
//     'email'  => $user->email,
// ];
//         return view('employee.invoice_thermal', compact('offlineSale', 'branchInfo'));
//     }
public function employeeprintThermalInvoice(string $orderId)
{
    $offlineSale = OfflineSale::where('order_id', $orderId)->firstOrFail();

    $employee = auth()->user(); 
    $branchName = $employee->branch;

    $branchUser = User::where('name', $branchName)->first();

    $branchInfo = [
        'name'    => $branchUser->name ?? '',
        'address' => $branchUser->building_name ?? '',
        'phone'   => $branchUser->phone_number ?? '',
        'email'   => $branchUser->email ?? '',
    ];

    return view('employee.invoice_thermal', compact('offlineSale', 'branchInfo'));
}

    public function employeeeditInvoice($id)
    {
        $offlineSale = OfflineSale::where('order_id', $id)->firstOrFail();
        $user = auth()->user();
         $branchInfo = [
    'name'   => $user->name,
    'address'=> $user->building_name ?? '',       
    'phone'  => $user->phone_number,
    'email'  => $user->email,
];
        // $branchInfo = match ($user->role) {
        //     'sakthi' => [
        //         'name' => 'Sakthi Super Mart',
        //         'address' => 'Chennai, India',
        //         'phone' => '+91 9500598595',
        //         'email' => 'admin1@gmail.com',
        //     ],
        //     'smart_world' => [
        //         'name' => 'SMARTWORLD',
        //         'address' => 'Chennai, India',
        //         'phone' => '+91 9443595314',
        //         'email' => 'admin2@gmail.com',
        //     ],
        //     'smart_world_gift' => [
        //         'name' => 'SMARTWORLD - Stationary & Gifts',
        //         'address' => 'Chennai, India',
        //         'phone' => '+91 9443595315',
        //         'email' => 'admin3@gmail.com',
        //     ],
        //     'yoga_naturals' => [
        //         'name' => 'Yoga Naturals',
        //         'address' => 'Chennai, India',
        //         'phone' => '+91 9443595316',
        //         'email' => 'admin4@gmail.com',
        //     ],
        //     default => [
        //         'name' => 'AdBazzar',
        //         'address' => 'Chennai, India',
        //         'phone' => '+91 8531345678',
        //         'email' => 'adbazzar@gmail.com',
        //     ],
        // };
        return view('employee.edit_invoice', compact('offlineSale', 'branchInfo'));
    }
    public function employeeupdateInvoice(Request $request, $order_id)
    {
        Log::info('Update Invoice Request:', $request->all());
        DB::beginTransaction();
        try {
            $offlineSale = OfflineSale::where('order_id', $order_id)->firstOrFail();
            // 1. Get old wallet deductions
            $oldCashWalletDeduction = $offlineSale->cash_wallet_deduction;
            $oldShoppingWalletDeduction = $offlineSale->shopping_wallet_deduction;
            $userId = $offlineSale->user_id;
            // Restore old wallet amounts if a user was associated
            if ($userId) {
                $userWallet = IncomeWallet::where('user_id', $userId)->first();
                if ($userWallet) {
                    // Add old deductions back to the wallet
                    $userWallet->cash_wallet += $oldCashWalletDeduction;
                    $userWallet->shopping_wallet += $oldShoppingWalletDeduction;
                    $userWallet->save();
                    Log::info("Restored old wallet amounts for user: {$userId}. Cash: {$oldCashWalletDeduction}, Shopping: {$oldShoppingWalletDeduction}");
                }
            }
            // --- Existing stock restoration and deduction logic ---
            $oldItems = is_array($offlineSale->sale_items) ? $offlineSale->sale_items : [];
            $newItems = $request->json('bill_items');
            // Step 1: Restore stock from old items
            foreach ($oldItems as $oldItem) {
                $itemIdentifier = $oldItem['item_identifier'] ?? null;
                $quantity = $oldItem['quantity'] ?? 0;
                $variantId = $oldItem['variant_id'] ?? null;
                if (!$itemIdentifier || !$quantity)
                    continue;
                $product = Product::where('item_code', $itemIdentifier)->first();
                if (!$product)
                    continue;
                if ($variantId) {
                    $variant = ProductVarient::where('variant_id', $variantId)
                        ->where('product_id', $product->product_id)
                        ->first();
                    if ($variant) {
                        $variant->stock += $quantity;
                        $variant->save();
                        Log::info("Restored stock for variant: {$variant->variant_name}, qty: {$quantity}");
                    }
                } else {
                    $product->stock += $quantity;
                    $product->save();
                    Log::info("Restored stock for product: {$product->title}, qty: {$quantity}");
                }
            }
            // Step 2: Validate and Deduct stock for new items
            foreach ($newItems as $item) {
                $itemIdentifier = $item['item_identifier'] ?? null;
                $quantity = $item['quantity'] ?? 0;
                $variantId = $item['variant_id'] ?? null;
                if (!$itemIdentifier || !$quantity)
                    continue;
                $product = Product::where('item_code', $itemIdentifier)->first();
                if (!$product)
                    continue;
                if ($variantId) {
                    $variant = ProductVarient::where('variant_id', $variantId)
                        ->where('product_id', $product->product_id)
                        ->first();
                    if (!$variant) {
                        return response()->json(['status' => 'error', 'message' => 'Product variant not found.'], 404);
                    }
                    if ($variant->stock < $quantity) {
                        return response()->json([
                            'status' => 'error',
                            'message' => 'Insufficient stock for variant: ' . $variant->variant_name,
                            'code' => 400
                        ], 400);
                    }
                    $variant->stock -= $quantity;
                    $variant->save();
                    Log::info("Deducted stock for variant: {$variant->variant_name}, qty: {$quantity}");
                } else {
                    if ($product->stock < $quantity) {
                        return response()->json([
                            'status' => 'error',
                            'message' => 'Insufficient stock for product: ' . $product->title,
                            'code' => 400
                        ], 400);
                    }
                    $product->stock -= $quantity;
                    $product->save();
                    Log::info("Deducted stock for product: {$product->title}, qty: {$quantity}");
                }
            }
            // --- New wallet deduction logic ---
            // Deduct from wallets if they are used in the new request
            $newCashWalletDeduction = $request->json('cash_wallet_deduction', 0);
            $newShoppingWalletDeduction = $request->json('shopping_wallet_deduction', 0);
            if ($userId) {
                $userWallet = IncomeWallet::where('user_id', $userId)->first();
                if ($userWallet) {
                    if ($newCashWalletDeduction > 0) {
                        $userWallet->cash_wallet -= $newCashWalletDeduction;
                    }
                    if ($newShoppingWalletDeduction > 0) {
                        $userWallet->shopping_wallet -= $newShoppingWalletDeduction;
                    }
                    $userWallet->save();
                    Log::info("Deducted new wallet amounts for user: {$userId}. Cash: {$newCashWalletDeduction}, Shopping: {$newShoppingWalletDeduction}");
                }
            }
            // Step 3: Update invoice
            $offlineSale->update([
                'customer_details' => $request->json('customer_details'),
                'sale_items' => $newItems,
                'tax_value' => $request->json('gst_total'),
                'sub_total' => $request->json('sub_total'),
                'discount' => $request->json('discount_amount'),
                'additional_charge' => $request->json('additional_charge'),
                'cash_wallet_deduction' => $newCashWalletDeduction,
                'shopping_wallet_deduction' => $newShoppingWalletDeduction,
                'amount_payable' => $request->json('amount_payable'),
                'payment_mode' => $request->json('payment_mode'),
                'pos_user_id' => $request->json('sold_by'),
            ]);
            DB::commit();
            return response()->json([
                'status' => 'success',
                'message' => 'Invoice updated and stock updated correctly',
                'order_id' => $offlineSale->order_id
            ], 200);
        } catch (\Exception $e) {
            DB::rollBack();
            Log::error("Invoice update failed: " . $e->getMessage());
            return response()->json([
                'status' => 'error',
                'message' => 'Failed to update invoice.',
                'error' => $e->getMessage(),
                'code' => 500
            ], 500);
        }
    }
    public function employeedeleteInvoice($id)
    {
        $invoice = OfflineSale::where('order_id', $id)->first();
        if (!$invoice) {
            return response()->json(['message' => 'Invoice not found.'], 404);
        }
        DB::beginTransaction();
        try {
            // The `sale_items` column is already an array, so remove the json_decode call.
            $saleItems = $invoice->sale_items;
            foreach ($saleItems as $item) {
                $product = Product::where('item_code', $item['item_code'])->first();
                if ($product) {
                    $product->stock += $item['quantity'];
                    $product->save();
                }
            }
            $invoice->delete();
            DB::commit();
            return response()->json(['message' => 'Invoice and associated stock updated successfully.'], 200);
        } catch (\Exception $e) {
            DB::rollBack();
            // You can log the error for more detailed debugging
            // \Log::error("Failed to delete invoice: " . $e->getMessage());
            return response()->json(['message' => 'Failed to delete invoice and update stock: ' . $e->getMessage()], 500);
        }
    }
    public function products()
    {
        $user = auth('employee')->user();
        $branch = $user->branch;
        $products = Product::with(['variants', 'category'])
            ->orderBy('product_id', 'asc')
            ->where('branch_name', $branch)
            ->get();
        return view('employee.product', compact('products'));
    }
    public function view_product($id)
    {
        $product = Product::with(['variants'])->findOrFail($id);
        $categories = Category::all();
        $branches = Branch::all();
        return view('employee.view_product', compact('product', 'categories', 'branches'));
    }
    public function updateStock(Request $request)
    {
        $request->validate([
            'id' => 'required|integer|exists:products,id',
            'stock' => 'required|integer|min:0',
        ]);
        $product = Product::find($request->id);
        $product->stock = $request->stock;
        $product->save();
        return redirect()->back()->with('success', 'Stock updated successfully!');
    }
}
