<?php

namespace App\Http\Controllers\Withdraw;

use App\Http\Controllers\Controller;
use App\Models\CashWalletIncome;
use Illuminate\Http\Request;
use App\Models\BankDetails;
use App\Models\IncomeWallet;
use App\Models\WithdrawRequest;
use Illuminate\Support\Facades\Validator;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Log;

class WithdrawalController extends Controller
{
    
    public function withdraw_request(Request $request)
{
    if (!auth()->guard('sanctum')->check()) {
        return response()->json([
            'status' => 'error',
            'message' => 'Please login!',
            'data' => [],
            'code' => 401,
        ]);
    }
    $user = auth()->guard('sanctum')->user();

    $validator = Validator::make($request->all(), [
        'type'        => 'required|in:bank,gpay',
        'amount'      => 'required|numeric|min:300',
        'bank_id'     => 'nullable|exists:bank_details,id,user_id,' . $user->user_id,
        'gpay_number' => 'nullable|digits:10',
    ]);

    if ($validator->fails()) {
        return response()->json([
            'status'  => 'error',
            'message' => 'Validation errors occurred.',
            'data'    => $validator->errors(),
            'code'    => 422,
        ], 422);
    }

    $userId       = $user->user_id;
    $amount       = $request->amount;
    $incomeWallet = IncomeWallet::where('user_id', $userId)->first();

    if (!$incomeWallet) {
        return response()->json([
            'success' => false,
            'message' => 'Income wallet not found',
        ], 404);
    }

    if ($incomeWallet->cash_wallet < $amount) {
        return response()->json([
            'success' => false,
            'message' => 'Insufficient balance in cash wallet',
        ], 400);
    }

    DB::beginTransaction();
    try {
        $incomeWallet->cash_wallet -= $amount;
        $incomeWallet->withdraw_wallet += $amount;
        $incomeWallet->save();

        $withdrawRequest              = new WithdrawRequest();
        $withdrawRequest->user_id     = $userId;
        $withdrawRequest->bank_id     = $request->bank_id;
        $withdrawRequest->amount      = $request->amount;
        $withdrawRequest->type        = $request->type;
        $withdrawRequest->gpay_number = $request->gpay_number;
        $withdrawRequest->status      = 'requested';
        $withdrawRequest->is_active   = '0';
        $withdrawRequest->save();

        CashWalletIncome::create([
            'user_id'      => $userId,
            'amount'       => $amount,
            'type'         => 'debit', 
            'remark'       => 'Withdrawal request',
            'description'  => 'Withdrawn from cash wallet',
            'is_active'    => 1,
        ]);

        DB::commit();

        return response()->json([
            'success' => true,
            'message' => 'Withdrawal request submitted successfully',
            'data'    => $withdrawRequest,
        ], 201);

    } catch (\Exception $e) {
        DB::rollBack();
        return response()->json([
            'success' => false,
            'message' => 'Error processing withdrawal request',
            'error'   => $e->getMessage(),
        ], 500);
    }
}

    
    
    public function accept_withdraw(Request $request)
    {
        try {
            Log::info('Accept Withdraw Request initiated.', ['accept_id' => $request->accept_id]);
            $withdraw = WithdrawRequest::find($request->accept_id);
    
            if (!$withdraw) {
                return redirect()->route('withdraw_request')->with('errormessage', "Withdraw request not found!");
            }
            Log::info('Withdraw request found.', ['withdraw_id' => $withdraw->id, 'user_id' => $withdraw->user_id]);
            $withdraw->status = "accepted";
            $withdraw->is_active = 1;
            $withdraw->save();
    
            return redirect()->route('accepted_withdraw')->with('successmessage', "Withdraw request accepted successfully!");
        } catch (\Exception $e) {
            Log::error('Error accepting withdraw request', ['error' => $e->getMessage()]);
            return redirect()->route('withdraw_request')->with('errormessage', "Something went wrong! Please try again.");
        }
    }

    
    // public function reject_withdraw(Request $request)
    // {
    //     try {
    //         $withdraw = WithdrawRequest::find($request->reject_id);
    
    //         if (!$withdraw) {
    //             return redirect()->back()->with('error', 'Withdrawal request not found!');
    //         }
    //         $incomeWallet = IncomeWallet::where('user_id', $withdraw->user_id)->first();
    //         if (!$incomeWallet) {
    //             return redirect()->back()->with('error', 'Income wallet not found!');
    //         }
    //         DB::beginTransaction();
    //         $incomeWallet->cash_wallet     += $withdraw->amount;
    //         $incomeWallet->withdraw_wallet -= $withdraw->amount;
    //         $incomeWallet->save();
    
    //         $withdraw->status    = "rejected";
    //         $withdraw->is_active = 0;
    //         $withdraw->save();
    
    //         DB::commit();
    
    //         return redirect()->back()->with('success', 'Withdrawal request rejected successfully!');
    
    //     } catch (\Exception $e) {
    //         DB::rollBack();
    //         return redirect()->back()->with('error', 'Something went wrong! Please try again.');
    //     }
    // }
    
    public function withdraw_history()
{
    if (!auth()->guard('sanctum')->check()) {
        return response()->json([
            'status' => 'error',
            'message' => 'Please login!',
            'data' => [],
            'code' => 401,
        ]);
    }

    $user = auth()->guard('sanctum')->user(); 
    $userId = $user->user_id; 

    $withdrawals = WithdrawRequest::where('user_id', $userId)
        ->with(['bankDetails']) 
        ->orderBy('created_at', 'desc')
        ->get();

    return response()->json([
        'success' => true,
        'message' => 'Withdrawal history fetched successfully',
        'data' => $withdrawals
    ]);
}

public function reject_withdraw(Request $request)
{
    try {
        $withdraw = WithdrawRequest::find($request->reject_id);

        if (!$withdraw) {
            return redirect()->back()->with('error', 'Withdrawal request not found!');
        }

        $incomeWallet = IncomeWallet::where('user_id', $withdraw->user_id)->first();
        if (!$incomeWallet) {
            return redirect()->back()->with('error', 'Income wallet not found!');
        }

        DB::beginTransaction();

        $incomeWallet->cash_wallet     += $withdraw->amount;
        $incomeWallet->withdraw_wallet -= $withdraw->amount;
        $incomeWallet->save();

        $withdraw->status    = "rejected";
        $withdraw->is_active = 0;
        $withdraw->save();

        CashWalletIncome::create([
            'user_id'     => $withdraw->user_id,
            'amount'      => $withdraw->amount,
            'type'        => 'credit',
            'remark'      => 'Withdrawal refund',
            'description' => 'withdrawal rejection',
            'is_active'   => 1,
        ]);

        DB::commit();

        return redirect()->route('rejected_withdraw')->with('successmessage', "Withdraw request accepted successfully!");

    } catch (\Exception $e) {
        DB::rollBack();
        return redirect()->back()->with('error', 'Something went wrong! Please try again.');
    }
}

}
