<!DOCTYPE html>
<html lang="en" x-data="purchaseInvoice()" @keydown.window="handleGlobalKeys($event)">
<head>
    <meta charset="UTF-8" />
    <meta name="viewport" content="width=device-width, initial-scale=1.0" />
    <title>Purchase Invoice</title>
    <link href="https://cdn.jsdelivr.net/npm/bootstrap@5.3.0/dist/css/bootstrap.min.css" rel="stylesheet" />
    <link href="https://cdn.jsdelivr.net/npm/bootstrap-icons@1.10.5/font/bootstrap-icons.css" rel="stylesheet" />
    <meta name="csrf-token" content="{{ csrf_token() }}">
    <style>
        .purchase-search-results {
            max-height: 200px;
            overflow-y: auto;
            z-index: 1000;
            position: absolute;
            width: calc(100% - 2rem);
            /* Adjust based on padding/margin of parent */
            background-color: white;
            border: 1px solid #dee2e6;
            border-top: none;
            left: 1rem;
            /* Adjust to align with input */
            right: 1rem;
            /* Adjust to align with input */
        }
        .table-scroll-container {
            max-height: calc(100vh - 450px);
            /* Adjust based on header/footer height */
            overflow-y: auto;
        }
        .table thead th {
            position: sticky;
            top: 0;
            background-color: #f8f9fa;
            z-index: 10;
        }
        .supplier-details-compact p {
            margin-bottom: 0;
            line-height: 1.2;
        }
    </style>
</head>
<body>
    <div class="container-fluid bg-light border-bottom py-2 px-3 d-flex justify-content-between align-items-center">
        <button class="btn btn-outline-secondary btn-sm" @click="exitPurchaseInvoice">
            <i class="bi bi-arrow-left"></i> <strong>Exit Purchase Invoice</strong> [ESC]
        </button>
        <h6 class="m-0">Purchase Invoice</h6>
        <div class="d-flex gap-2">
            <button class="btn btn-outline-secondary btn-sm" @click="showHeldInvoicesModal">
                <strong>Held Invoices</strong> [F8]
            </button>
            <button class="btn btn-outline-secondary btn-sm" @click="openSettings">
                <strong>Settings</strong> [CTRL+S]
            </button>
        </div>
    </div>
    <div class="container-fluid p-3">
        {{-- <div class="d-flex flex-wrap align-items-center gap-3 px-2 py-2 bg-light border rounded">
            <div class="card-header bg-primary text-white py-2 px-3 rounded-top">
                <h6 class="mb-0">Supplier Details</h6>
            </div>
            <div class="flex-grow-1">
                <input type="text" class="form-control" x-model.debounce.1000ms="currentInvoice().supplierIdentifier"
                    @input.debounce.1000ms="fetchSupplierDetails()" placeholder="Type supplier Phonenumber and wait...">
            </div>
            <template x-if="currentInvoice().supplierDetails">
                <div class="d-flex flex-wrap align-items-center gap-3 supplier-details-compact">
                    <div>
                        <strong x-text="currentInvoice().supplierDetails.name"></strong>
                        <small class="text-muted">(ID: <span
                                x-text="currentInvoice().supplierDetails.id"></span>)</small>
                    </div>
                    <div>
                        <strong x-text="currentInvoice().supplierDetails.customer_name"></strong>
                    </div>
                    <div class="text-muted small">
                        <i class="bi bi-telephone"></i>
                        <span x-text="currentInvoice().supplierDetails.phone_number"></span>
                    </div>
                    <div class="text-muted small">
                        <i class="bi bi-geo-alt"></i>
                        <span x-text="currentInvoice().supplierDetails.address"></span>
                    </div>
                </div>
            </template>
            <div class="text-muted small fst-italic"
                x-show="currentInvoice().supplierIdentifier && !currentInvoice().supplierDetails && !currentInvoice().supplierDetailsLoading">
                No supplier found.
            </div>
            <div class="text-info small fst-italic" x-show="currentInvoice().supplierDetailsLoading">
                Searching...
            </div>
        </div> --}}
        <div class="d-flex flex-wrap align-items-center gap-3 px-2 py-2 bg-light border rounded">
            <div class="card-header bg-primary text-white py-2 px-3 rounded-top">
                <h6 class="mb-0">Supplier Details</h6>
            </div>
            <div class="flex-grow-1">
                <input type="text" class="form-control" x-model.debounce.1000ms="currentInvoice().supplierIdentifier"
                    @input.debounce.1000ms="fetchSupplierDetails()" placeholder="Type supplier Phonenumber, Name or Agency Name...">
            </div>
            <template x-if="currentInvoice().supplierDetails">
                <div class="d-flex flex-wrap align-items-center gap-3 supplier-details-compact">
                    <div>
                        <strong x-text="currentInvoice().supplierDetails.customer_name"></strong>
                        <span x-show="currentInvoice().supplierDetails.agency_name">
                            (<em x-text="currentInvoice().supplierDetails.agency_name"></em>)
                        </span>
                        <small class="text-muted">(ID: <span
                                x-text="currentInvoice().supplierDetails.id"></span>)</small>
                    </div>
                    <div class="text-muted small">
                        <i class="bi bi-telephone"></i>
                        <span x-text="currentInvoice().supplierDetails.phone_number"></span>
                    </div>
                    <div class="text-muted small">
                        <i class="bi bi-geo-alt"></i>
                        <span x-text="currentInvoice().supplierDetails.address"></span>
                    </div>
                </div>
            </template>
            <div class="text-muted small fst-italic"
                x-show="currentInvoice().supplierIdentifier && !currentInvoice().supplierDetails && !currentInvoice().supplierDetailsLoading">
                No supplier found. Opening new supplier registration if 10-digit number or name was entered.
            </div>
            <div class="text-info small fst-italic" x-show="currentInvoice().supplierDetailsLoading">
                Searching...
            </div>
        </div>
        <ul class="nav nav-tabs mb-1">
            <template x-for="invoice in invoices" :key="invoice.id">
                <li class="nav-item">
                    <a href="#" class="nav-link" :class="{ active: invoice.id === activeInvoiceId }"
                        @click.prevent="activateInvoice(invoice.id)">
                        <span x-text="invoice.name"></span>
                        <i class="bi bi-x-lg ms-2" x-show="invoices.length > 1"
                            @click.stop="closeInvoice(invoice.id)"></i>
                    </a>
                </li>
            </template>
            <li class="nav-item">
                <a href="#" class="nav-link" @click.prevent="addNewInvoice"><i class="bi bi-plus"></i> New Invoice
                    [CTRL+B]</a>
            </li>
        </ul>
        <div class="row">
            <div class="col-md-8 border-end position-relative">
                <div class="d-flex gap-2 mb-3">
                    <button class="btn btn-outline-secondary btn-sm" @click="openPurchasePriceChangeModal">Change
                        Purchase Price
                        [CTRL+P]</button>
                    <button class="btn btn-outline-secondary btn-sm" @click="openQuantityChangeModal">Change QTY
                        [CTRL+Q]</button>
                    <button class="btn btn-outline-secondary btn-sm" @click="deleteSelectedItem()">Delete Item
                        [DEL]</button>
                </div>
                <div class="input-group mb-3">
                    <span class="input-group-text"><i class="bi bi-search"></i></span>
                    <input x-ref="searchInput" class="form-control" type="text" placeholder="Search item/code"
                        x-model.debounce.300ms="currentInvoice().searchQuery" @input.debounce.300ms="searchProducts"
                        @keydown.enter.prevent="handleSearchEnter" @keydown.arrow-down.prevent="selectNextResult"
                        @keydown.arrow-up.prevent="selectPrevResult" @keydown.escape.stop="clearSearchResults">
                    <span class="input-group-text">[F1]</span>
                </div>
                <ul class="list-group purchase-search-results" 
    x-show="(currentInvoice().searchResults || []).length">
    <template x-for="(prod, idx) in currentInvoice().searchResults || []" 
              :key="prod.item_identifier || idx">
        <li class="list-group-item list-group-item-action"
            :class="{ active: idx === currentInvoice().selectedSearchResultIndex }"
            @click="addItemToInvoice(prod); clearSearchResults();"
            @mouseover="currentInvoice().selectedSearchResultIndex = idx">
            <span x-text="prod?.name || 'Unnamed'"></span> - 
            <small x-text="prod?.item_code || ''"></small> 
            (P.Price: ₹ <span x-text="Number(prod?.purchase_price || 0).toFixed(2)"></span> | 
            Stock: <span x-text="prod?.stock || 0"></span>)
        </li>
    </template>
</ul>
                <div class="table">
                    <table class="table table-bordered text-center">
                        <thead class="table-light">
                            <tr>
                                <th>#</th>
                                <th>Item</th>
                                <th>Code</th>
                                <th>MRP</th>
                                <th>P.Price</th>
                                <th>S.Price</th>
                                <th>S.wallet</th>
                                <th>QTY</th>
                                <th>GST</th>
                                <th>Amount</th>
                                <th>Actions</th>
                            </tr>
                        </thead>
                        <tbody>
                            <template x-if="!currentInvoice().invoiceItems.length">
                                <tr>
                                    <td colspan="10" class="py-5 text-muted"><i class="bi bi-box-seam fs-1"></i>
                                        <p>Add items to the purchase invoice...</p>
                                    </td>
                                </tr>
                            </template>
                            <template x-for="(item, idx) in currentInvoice().invoiceItems" :key="idx">
                                <tr :class="{ 'table-primary': currentInvoice().selectedItemIndex === idx }"
                                    @click="selectItem(idx)">
                                    <td x-text="idx+1"></td>
                                    <td x-text="item.title"></td>
                                    <td><span x-text="item.item_code"></span>
                                        <small class="text-muted">(Stock:<span x-text="item.stock"></span>)</small>
                                    </td>
                                    <td>
                                        <input type="number" class="form-control form-control-sm text-center"
                                            style="width:100px;margin:auto;" x-model.number="item.mrp"
                                            @change="calculateInvoiceTotals()" min="0" step="1">
                                    </td>
                                    <td>
                                        <input type="number" class="form-control form-control-sm text-center"
                                            style="width:100px;margin:auto;" x-model.number="item.purchase_price"
                                            @change="calculateInvoiceTotals()" min="0" step="1">
                                    </td>
                                    <td>
                                        <input type="number" class="form-control form-control-sm text-center"
                                            style="width:100px;margin:auto;" x-model.number="item.sell_price"
                                            @change="calculateInvoiceTotals()" min="0" step="1">
                                    </td>
                                    <td>
                                        <input type="number" class="form-control form-control-sm text-center"
                                            style="width:100px;margin:auto;"
                                            x-model.number="item.shopping_wallet_price"
                                            @change="calculateInvoiceTotals()" min="0" step="1">
                                    </td>
                                    <td>
                                        <input type="number" class="form-control form-control-sm text-center"
                                            style="width:80px;margin:auto;" x-model.number="item.quantity"
                                            @change="calculateInvoiceTotals()" min="1">
                                    </td>
                                    <td>
                                        <span
                                            x-text="item.gst_rate % 1 === 0 ? parseInt(item.gst_rate) + '%' : parseFloat(item.gst_rate).toFixed(2) + '%'"></span>
                                    </td>
                                    <td x-text="((Number(item.purchase_price) || 0) * item.quantity).toFixed(2)"></td>
                                    <td>
                                        <div class="d-flex justify-content-spacearound">
                                        <button type="button" class="btn btn-info btn-sm me-2"
                                            @click.stop="openBarcodeModal(item)">
                                            <i class="bi bi-upc-scan"></i>
                                        </button>
                                        <button type="button" class="btn btn-danger btn-sm"
                                            @click="deleteSelectedItem()">
                                            <i class="bi bi-x-lg"></i>
                                        </button>
                                        </div>
                                    </td>
                                </tr>
                            </template>
                        </tbody>
                    </table>
                </div>
            </div>
            <div class="col-md-4">
                <div class="d-flex gap-2 mb-2">
                    <button class="btn btn-outline-secondary btn-sm flex-fill" @click="openDiscountModal">Add Discount
                        [F2]</button>
                    <button class="btn btn-outline-secondary btn-sm flex-fill" @click="openAdditionalChargeModal">Add
                        Charge [F3]</button>
                </div>
                <div class="card p-3 mb-2">
                    <h6>Invoice Summary</h6>
                    <div class="d-flex justify-content-between">
                        <span>GST Total</span>
                        <span>₹ <span x-text="currentInvoice().gstTotal.toFixed(2)"></span></span>
                    </div>
                    <div class="d-flex justify-content-between">
                        <span>Sub Total (Excl. GST)</span>
                        <span>₹ <span x-text="currentInvoice().subTotalExcludingGST.toFixed(2)"></span></span>
                    </div>
                    <div class="d-flex justify-content-between">
                        <span>Sub Total (Incl. GST)</span>
                        <span>₹ <span x-text="currentInvoice().subTotalIncludingGST.toFixed(2)"></span></span>
                    </div>
                    <template x-if="currentInvoice().discountAmount > 0">
                        <div class="d-flex justify-content-between">
                            <span>Discount</span>
                            <span>₹ <span x-text="currentInvoice().discountAmount.toFixed(2)"></span></span>
                        </div>
                    </template>
                    <template x-if="currentInvoice().additionalCharge > 0">
                        <div class="d-flex justify-content-between">
                            <span>Charge</span>
                            <span>₹ <span x-text="currentInvoice().additionalCharge.toFixed(2)"></span></span>
                        </div>
                    </template>
                    <hr>
                    <div class="d-flex justify-content-between fw-bold text-success">
                        <span>Grand Total</span>
                        <span>₹ <span x-text="currentInvoice().totalAmount.toFixed(2)"></span></span>
                    </div>
                </div>
                <div class="d-flex gap-2">
                    <button class="btn btn-primary flex-fill" @click="saveInvoiceAndPrint">Save & Print [F6]</button>
                    <button class="btn btn-outline-secondary flex-fill" @click="saveHeldInvoice">Hold Invoice
                        [F7]</button>
                </div>
            </div>
        </div>
    </div>
    <div class="modal fade" id="purchasePriceChangeModal" tabindex="-1">
        <div class="modal-dialog modal-sm modal-dialog-centered">
            <div class="modal-content">
                <div class="modal-header">
                    <h5>Change Purchase Price</h5><button class="btn-close" data-bs-dismiss="modal"></button>
                </div>
                <div class="modal-body"><input type="number" class="form-control"
                        x-model.number="tempPurchasePrice" step="1"></div>
                <div class="modal-footer"><button class="btn btn-primary"
                        @click="confirmChangePurchasePrice">Apply</button><button class="btn btn-outline-secondary"
                        data-bs-dismiss="modal">Cancel</button></div>
            </div>
        </div>
    </div>
    <div class="modal fade" id="qtyChangeModal" tabindex="-1">
        <div class="modal-dialog modal-sm modal-dialog-centered">
            <div class="modal-content">
                <div class="modal-header">
                    <h5>Change Quantity</h5><button class="btn-close" data-bs-dismiss="modal"></button>
                </div>
                <div class="modal-body"><input type="number" class="form-control" x-model.number="tempQty"
                        min="1"></div>
                <div class="modal-footer"><button class="btn btn-primary"
                        @click="confirmChangeQty">Apply</button><button class="btn btn-outline-secondary"
                        data-bs-dismiss="modal">Cancel</button></div>
            </div>
        </div>
    </div>
    <div class="modal fade" id="discountModal" tabindex="-1">
        <div class="modal-dialog modal-sm modal-dialog-centered">
            <div class="modal-content">
                <div class="modal-header">
                    <h5>Add Discount</h5><button class="btn-close" data-bs-dismiss="modal"></button>
                </div>
                <div class="modal-body"><input type="number" class="form-control mb-2"
                        x-model.number="tempDiscount" placeholder="Amount" step="1"><input
                        class="form-control" x-model="tempDiscountNote" placeholder="Reason (opt)"></div>
                <div class="modal-footer"><button class="btn btn-primary"
                        @click="confirmDiscount">Apply</button><button class="btn btn-outline-secondary"
                        data-bs-dismiss="modal">Cancel</button></div>
            </div>
        </div>
    </div>
    <div class="modal fade" id="chargeModal" tabindex="-1">
        <div class="modal-dialog modal-sm modal-dialog-centered">
            <div class="modal-content">
                <div class="modal-header">
                    <h5>Add Charge</h5><button class="btn-close" data-bs-dismiss="modal"></button>
                </div>
                <div class="modal-body"><input type="number" class="form-control mb-2" x-model.number="tempCharge"
                        placeholder="Amount" step="1"><input class="form-control" x-model="tempChargeNote"
                        placeholder="Reason (opt)"></div>
                <div class="modal-footer"><button class="btn btn-primary"
                        @click="confirmCharge">Apply</button><button class="btn btn-outline-secondary"
                        data-bs-dismiss="modal">Cancel</button></div>
            </div>
        </div>
    </div>
    <div class="modal fade" id="heldInvoicesModal" tabindex="-1">
        <div class="modal-dialog modal-lg modal-dialog-centered">
            <div class="modal-content">
                <div class="modal-header">
                    <h5>Held Purchase Invoices</h5><button class="btn-close" data-bs-dismiss="modal"></button>
                </div>
                <div class="modal-body">
                    <ul class="list-group">
                        <template x-for="hi in heldInvoices" :key="hi.invoice_id">
                            <li class="list-group-item d-flex justify-content-between align-items-center"><span><strong
                                        x-text="hi.invoice_id"></strong> – ₹<span
                                        x-text="hi.total_amount.toFixed(2)"></span></span>
                                <div><button class="btn btn-secondary btn-sm me-2"
                                        @click="resumeInvoice(hi.invoice_id)">Resume</button><button
                                        class="btn btn-danger btn-sm"
                                        @click="deleteHeldInvoice(hi.invoice_id)">Delete</button></div>
                            </li>
                        </template>
                        <template x-if="!heldInvoices.length">
                            <li class="list-group-item text-center text-muted">No held invoices found.</li>
                        </template>
                    </ul>
                </div>
            </div>
        </div>
    </div>
    <div class="modal fade" id="settingsModal" tabindex="-1">
        <div class="modal-dialog modal-sm modal-dialog-centered">
            <div class="modal-content">
                <div class="modal-header">
                    <h5>Select Print Type</h5>
                    <button class="btn-close" data-bs-dismiss="modal"></button>
                </div>
                <div class="modal-body">
                    <div class="form-check">
                        <input class="form-check-input" type="radio" x-model="printType" value="a4"
                            id="print_a4">
                        <label class="form-check-label" for="print_a4">A4 Invoice</label>
                    </div>
                    <div class="form-check">
                        <input class="form-check-input" type="radio" x-model="printType" value="thermal"
                            id="print_thermal">
                        <label class="form-check-label" for="print_thermal">Thermal (3 inch)</label>
                    </div>
                </div>
                <div class="modal-footer">
                    <button class="btn btn-primary" @click="saveSettings">Save</button>
                    <button class="btn btn-outline-secondary" data-bs-dismiss="modal">Close</button>
                </div>
            </div>
        </div>
    </div>
    {{-- <div class="modal fade" id="newSupplierModal" tabindex="-1">
        <div class="modal-dialog modal-sm modal-dialog-centered">
            <div class="modal-content">
                <div class="modal-header">
                    <h5>Register New Supplier</h5>
                    <button class="btn-close" data-bs-dismiss="modal"></button>
                </div>
                <div class="modal-body">
                    <div class="mb-3">
                        <label for="new_supplier_phone" class="form-label">Phone Number</label>
                        <input type="text" class="form-control" id="new_supplier_phone"
                            x-model="newSupplier.phone_number" disabled>
                    </div>
                    <div class="mb-3">
                        <label for="new_supplier_name" class="form-label">Customer Name <span
                                class="text-danger">*</span></label>
                        <input type="text" class="form-control" id="new_supplier_name"
                            x-model="newSupplier.customer_name">
                    </div>
                     <div class="mb-3">
                        <label for="new_supplier_name" class="form-label">Agency Name <span
                                class="text-danger">*</span></label>
                        <input type="text" class="form-control" id="new_agency_name"
                            x-model="newSupplier.customer_name">
                    </div>
                    <div class="mb-3">
                        <label for="new_supplier_email" class="form-label">GST IN</label>
                        <input type="email" class="form-control" id="new_supplier_gst"
                            x-model="newSupplier.email">
                    </div>
                    <div class="mb-3">
                        <label for="new_supplier_address" class="form-label">Address (Optional)</label>
                        <input type="text" class="form-control" id="new_supplier_address"
                            x-model="newSupplier.address">
                    </div>
                </div>
                <div class="modal-footer">
                    <button class="btn btn-primary" @click="confirmNewSupplier">Register</button>
                    <button class="btn btn-outline-secondary" data-bs-dismiss="modal">Cancel</button>
                </div>
            </div>
        </div>
    </div> --}}
    <div class="modal fade" id="newSupplierModal" tabindex="-1">
    <div class="modal-dialog modal-sm modal-dialog-centered">
        <div class="modal-content">
            <div class="modal-header">
                <h5>Register New Supplier</h5>
                <button class="btn-close" data-bs-dismiss="modal"></button>
            </div>
            <div class="modal-body">
                <div class="mb-3">
                    <label for="new_supplier_phone" class="form-label">Phone Number</label>
                    <input type="text" class="form-control" id="new_supplier_phone"
                        x-model="newSupplier.phone_number">
                </div>
                <div class="mb-3">
                    <label for="new_supplier_name" class="form-label">Customer Name <span class="text-danger">*</span></label>
                    <input type="text" class="form-control" id="new_supplier_name"
                        x-model="newSupplier.customer_name">
                </div>
                <div class="mb-3">
                    <label for="new_agency_name" class="form-label">Agency Name <span class="text-danger">*</span></label>
                    <input type="text" class="form-control" id="new_agency_name"
                        x-model="newSupplier.agency_name">
                </div>
                <div class="mb-3">
                    <label for="new_supplier_gst" class="form-label">GST IN</label>
                    <input type="text" class="form-control" id="new_supplier_gst"
                        x-model="newSupplier.gstin">
                </div>
                <div class="mb-3">
                    <label for="new_supplier_address" class="form-label">Address (Optional)</label>
                    <input type="text" class="form-control" id="new_supplier_address"
                        x-model="newSupplier.address">
                </div>
            </div>
            <div class="modal-footer">
                <button class="btn btn-primary" @click="confirmNewSupplier">Register</button>
                <button class="btn btn-outline-secondary" data-bs-dismiss="modal">Cancel</button>
            </div>
        </div>
    </div>
</div>
    <div class="modal fade" id="barcodeModal" tabindex="-1">
        <div class="modal-dialog modal-dialog-centered">
            <div class="modal-content">
                <div class="modal-header">
                    <h5 class="modal-title">Generate Barcode Labels</h5>
                    <button type="button" class="btn-close" data-bs-dismiss="modal"></button>
                </div>
                <template x-if="barcodeModalItem">
                    <div class="modal-body text-center">
                        <p>
                            <strong x-text="barcodeModalItem.title"></strong>
                            <span class="badge bg-secondary" x-text="barcodeModalItem.item_code"></span>
                        </p>
                        <div class="my-3">
                            <svg id="purchaseBarcodeModalSvg"></svg>
                        </div>
                    </div>
                </template>
                <div class="modal-footer d-flex justify-content-center">
                    <button type="button" class="btn btn-primary" @click="printBarcode()">
                        <i class="bi bi-printer"></i> Print Labels
                    </button>
                </div>
            </div>
        </div>
    </div>
</body>
</html>
<script src="https://cdn.jsdelivr.net/npm/bootstrap@5.3.0/dist/js/bootstrap.bundle.min.js"></script>
<script defer src="https://cdn.jsdelivr.net/npm/alpinejs@3/dist/cdn.min.js"></script>
<!-- JsBarcode CDN -->
<script src="https://cdn.jsdelivr.net/npm/jsbarcode@3.11.6/dist/JsBarcode.all.min.js"></script>
<script src="https://cdnjs.cloudflare.com/ajax/libs/jspdf/2.5.1/jspdf.umd.min.js"></script>
<script>
    document.addEventListener('alpine:init', () => {
        Alpine.data('purchaseInvoice', () => ({
            activeInvoiceId: 1,
            invoices: [{
                id: 1,
                name: 'Invoice 1',
                supplierIdentifier: '',
                supplierDetails: null, // Will hold customer_name, gstin, phone_number, address, branch
                supplierDetailsLoading: false,
                searchQuery: '',
                searchResults: [],
                selectedSearchResultIndex: -1,
                invoiceItems: [],
                selectedItemIndex: -1,
                discountAmount: 0,
                discountNote: '',
                additionalCharge: 0,
                additionalChargeNote: '',
                gstTotal: 0, // Calculated GST total
                subTotalExcludingGST: 0, // Sum of (purchase_price * quantity)
                subTotalIncludingGST: 0, // Sum of (purchase_price * quantity + gst_amount)
                totalAmount: 0, // Grand total after all calculations
            }],
            tempPurchasePrice: 0,
            tempQty: 0,
            tempDiscount: 0,
            tempDiscountNote: '',
            tempCharge: 0,
            tempChargeNote: '',
            heldInvoices: [], 
            printType: localStorage.getItem('purchasePrintType') || 'a4', 
            newSupplier: {
                customer_name: '',
                gstin: '', 
                phone_number: '',
                address: '',
                branch: '', 
            },
            barcodeModal: null, 
            barcodeModalItem: null, 
            numberOfLabelsToPrint: 1,
            init() {
                this.loadHeldInvoices();
                this.calculateInvoiceTotals(); // Initial calculation
                // Initialize modals
                this.purchasePriceChangeModal = new bootstrap.Modal(document.getElementById(
                    'purchasePriceChangeModal'));
                this.qtyChangeModal = new bootstrap.Modal(document.getElementById(
                    'qtyChangeModal'));
                this.discountModal = new bootstrap.Modal(document.getElementById('discountModal'));
                this.chargeModal = new bootstrap.Modal(document.getElementById('chargeModal'));
                this.heldInvoicesModal = new bootstrap.Modal(document.getElementById(
                    'heldInvoicesModal'));
                this.settingsModal = new bootstrap.Modal(document.getElementById('settingsModal'));
                this.newSupplierModal = new bootstrap.Modal(document.getElementById(
                    'newSupplierModal'));
                this.barcodeModal = new bootstrap.Modal(document.getElementById('barcodeModal'));
            },
            currentInvoice() {
                return this.invoices.find(invoice => invoice.id === this.activeInvoiceId);
            },
            addNewInvoice() {
                const newId = this.invoices.length ? Math.max(...this.invoices.map(b => b.id)) + 1 :
                    1;
                this.invoices.push({
                    id: newId,
                    name: `Invoice ${newId}`,
                    supplierIdentifier: '',
                    supplierDetails: null,
                    supplierDetailsLoading: false,
                    searchQuery: '',
                    searchResults: [],
                    selectedSearchResultIndex: -1,
                    invoiceItems: [],
                    selectedItemIndex: -1,
                    discountAmount: 0,
                    discountNote: '',
                    additionalCharge: 0,
                    additionalChargeNote: '',
                    gstTotal: 0,
                    subTotalExcludingGST: 0,
                    subTotalIncludingGST: 0,
                    totalAmount: 0,
                });
                this.activeInvoiceId = newId;
            },
            activateInvoice(id) {
                this.activeInvoiceId = id;
                this.$refs.searchInput.focus();
                this.calculateInvoiceTotals();
            },
            closeInvoice(id) {
                if (this.invoices.length === 1) {
                    alert('Cannot close the last invoice.');
                    return;
                }
                this.invoices = this.invoices.filter(invoice => invoice.id !== id);
                if (this.activeInvoiceId === id) {
                    this.activeInvoiceId = this.invoices[0].id;
                }
                this.calculateInvoiceTotals();
            },
            exitPurchaseInvoice() {
                if (confirm('Are you sure you want to exit? Any unsaved invoices will be lost.')) {
                    window.location.href = '/admin/dashboard'; // Adjust as per your route
                }
            },
            handleGlobalKeys(event) {
                if (event.key === 'F1') {
                    event.preventDefault();
                    this.$refs.searchInput.focus();
                }
                if (event.key === 'F2') {
                    event.preventDefault();
                    this.openDiscountModal();
                }
                if (event.key === 'F3') {
                    event.preventDefault();
                    this.openAdditionalChargeModal();
                }
                if (event.key === 'F6') {
                    event.preventDefault();
                    this.saveInvoiceAndPrint();
                }
                if (event.key === 'F7') {
                    event.preventDefault();
                    this.saveHeldInvoice();
                }
                if (event.key === 'F8') {
                    event.preventDefault();
                    this.showHeldInvoicesModal();
                }
                if (event.key === 'Escape') {
                    event.preventDefault();
                    this.exitPurchaseInvoice();
                }
                if (event.ctrlKey && event.key === 'b') {
                    event.preventDefault();
                    this.addNewInvoice();
                }
                if (event.ctrlKey && event.key === 'p') {
                    event.preventDefault();
                    this.openPurchasePriceChangeModal();
                }
                if (event.ctrlKey && event.key === 'q') {
                    event.preventDefault();
                    this.openQuantityChangeModal();
                }
                if (event.key === 'Delete') {
                    event.preventDefault();
                    this.deleteSelectedItem();
                }
                if (event.ctrlKey && event.key === 's') {
                    event.preventDefault();
                    this.openSettings();
                }
            },
            // async fetchSupplierDetails() {
            //     const current = this.currentInvoice();
            //     const identifier = current.supplierIdentifier ? String(current.supplierIdentifier).trim() : '';
            //     if (!identifier) {
            //         current.supplierDetails = null;
            //         return;
            //     }
            //     current.supplierDetailsLoading = true;
            //     try {
            //         const response = await fetch(
            //             `{{ route('suppliers_search') }}?query=${encodeURIComponent(identifier)}`
            //         );
            //         if (!response.ok) {
            //             const errorData = await response.json();
            //             console.warn('Supplier search failed:', response.status, errorData
            //                 .message);
            //             if (response.status === 404) {
            //                 current.supplierDetails = null;
            //                 if (/^\d{10}$/.test(identifier)) {
            //                     this.newSupplier.phone_number = identifier;
            //                 } else {
            //                     this.newSupplier.customer_name = identifier;
            //                 }
            //                 this.newSupplierModal.show();
            //             } else {
            //                 throw new Error(errorData.message ||
            //                     `HTTP error! status: ${response.status}`);
            //             }
            //         } else {
            //             const data = await response.json();
            //             if (data && data.id) {
            //                 current.supplierDetails = {
            //                     id: data.id,
            //                     customer_name: data.customer_name,
            //                     agency_name: data.agency_name, 
            //                     gstin: data.gstin,
            //                     phone_number: data.phone_number,
            //                     address: data.address,
            //                     branch: data.branch,
            //                 };
            //                 current.supplierIdentifier = data.phone_number || data
            //                     .customer_name || data.agency_name;
            //             } else {
            //                 current.supplierDetails = null;
            //                 if (/^\d{10}$/.test(identifier)) {
            //                     this.newSupplier.phone_number = identifier;
            //                 } else {
            //                     this.newSupplier.customer_name = identifier;
            //                 }
            //                 this.newSupplierModal.show();
            //             }
            //         }
            //     } catch (error) {
            //         console.error('Error fetching supplier details:', error);
            //         current.supplierDetails = null;
            //     } finally {
            //         current.supplierDetailsLoading = false;
            //         this.calculateInvoiceTotals();
            //     }
            // },
            async fetchSupplierDetails() {
    const current = this.currentInvoice();
    const identifier = current.supplierIdentifier ? String(current.supplierIdentifier).trim() : '';
    // ✅ If identifier is empty or too short, don't trigger search
    if (!identifier || identifier.length < 4) {
        current.supplierDetails = null;
        return;
    }
    // ✅ For phone numbers, wait until full 10 digits are typed
    const isPhoneSearch = /^\d+$/.test(identifier);
    if (isPhoneSearch && identifier.length < 10) {
        current.supplierDetails = null;
        return;
    }
    current.supplierDetailsLoading = true;
    try {
        const response = await fetch(
            `{{ route('suppliers_search') }}?query=${encodeURIComponent(identifier)}`
        );
        if (!response.ok) {
            let errorData = {};
            try {
                errorData = await response.json();
            } catch {}
            console.warn('Supplier search failed:', response.status, errorData.message);
            // ✅ Only show modal if it's a *confirmed search*, not during partial input
            if (response.status === 404) {
                current.supplierDetails = null;
                if (isPhoneSearch && identifier.length === 10) {
                    this.newSupplier.phone_number = identifier;
                    this.newSupplier.customer_name = '';
                    this.newSupplierModal.show();
                } else if (!isPhoneSearch) {
                    this.newSupplier.customer_name = identifier;
                    this.newSupplier.phone_number = '';
                    this.newSupplierModal.show();
                }
            } else {
                throw new Error(errorData.message || `HTTP error! status: ${response.status}`);
            }
        } else {
            const data = await response.json();
            if (data && data.id) {
                current.supplierDetails = {
                    id: data.id,
                    customer_name: data.customer_name,
                    agency_name: data.agency_name,
                    gstin: data.gstin,
                    phone_number: data.phone_number,
                    address: data.address,
                    branch: data.branch,
                };
                current.supplierIdentifier =
                    data.phone_number || data.customer_name || data.agency_name;
            } else {
                current.supplierDetails = null;
                if (isPhoneSearch && identifier.length === 10) {
                    this.newSupplier.phone_number = identifier;
                    this.newSupplier.customer_name = '';
                    this.newSupplierModal.show();
                } else if (!isPhoneSearch) {
                    this.newSupplier.customer_name = identifier;
                    this.newSupplier.phone_number = '';
                    this.newSupplierModal.show();
                }
            }
        }
    } catch (error) {
        console.error('Error fetching supplier details:', error);
        current.supplierDetails = null;
    } finally {
        current.supplierDetailsLoading = false;
        this.calculateInvoiceTotals();
    }
},
            async confirmNewSupplier() {
            const current = this.currentInvoice();
            if (!this.newSupplier.customer_name) {
                alert('Supplier Name is required.');
                return;
            }
            try {
                const response = await fetch('{{ route('suppliers_store') }}', {
                    method: 'POST',
                    headers: {
                        'Content-Type': 'application/json',
                        'X-CSRF-TOKEN': document.querySelector('meta[name="csrf-token"]').content
                    },
                    body: JSON.stringify(this.newSupplier)
                });
                const responseData = await response.json();
                if (!response.ok) {
                    let errorMessage = responseData.message || `HTTP error! status: ${response.status}`;
                    if (response.status === 422 && responseData.errors) {
                        errorMessage += '\n' + Object.values(responseData.errors).flat().join('\n');
                    }
                    alert('Failed to register supplier: ' + errorMessage);
                    return;
                }
                if (responseData.success) {
                    current.supplierDetails = {
                        id: responseData.supplier.id,
                        customer_name: responseData.supplier.customer_name,
                        agency_name: responseData.supplier.agency_name,
                        gstin: responseData.supplier.gstin,
                        phone_number: responseData.supplier.phone_number,
                        address: responseData.supplier.address,
                        branch: responseData.supplier.branch,
                    };
                    current.supplierIdentifier = responseData.supplier.phone_number || responseData.supplier.customer_name;
                    // Close modal properly
                    const modalEl = document.getElementById('newSupplierModal');
                    const modalInstance = bootstrap.Modal.getInstance(modalEl);
                    modalInstance.hide();
                    alert('New supplier registered successfully!');
                } else {
                    alert('Failed to register supplier: ' + (responseData.message || 'Unknown error'));
                }
            } catch (error) {
                console.error('Error registering new supplier:', error);
                alert('An error occurred while registering the new supplier. Check console for details.');
            } finally {
                this.newSupplier = {
                    customer_name: '',
                    agency_name: '',
                    gstin: '',
                    phone_number: '',
                    address: '',
                    branch: ''
                };
            }
        },
        async searchProducts() {
                const current = this.currentInvoice();
                const q = current.searchQuery.trim();
                if (!q) {
                    current.searchResults = [];
                    current.selectedSearchResultIndex = -1;
                    return;
                }
                try {
                    const response = await fetch(
                        `{{ route('products_search') }}?query=${encodeURIComponent(q)}`);
                    if (!response.ok) {
                        const errorData = await response.json().catch(() => ({
                            message: 'Server error or invalid JSON response'
                        }));
                        console.error('Error searching products (HTTP):', response.status,
                            errorData.message || 'Unknown error');
                        current.searchResults = [];
                        return;
                    }
                    const data = await response.json();
                    current.searchResults = data;
                    current.selectedSearchResultIndex = data.length > 0 ? 0 : -1;
                    const lowerQuery = q.toLowerCase();
                    const exactMatch = current.searchResults.find(prod =>
                        lowerQuery === prod.item_code?.toLowerCase() ||
                        lowerQuery === prod.name?.toLowerCase() ||
                        lowerQuery === prod.hsn_code?.toLowerCase() ||
                        lowerQuery === prod.barcode1?.toLowerCase() ||
                        lowerQuery === prod.barcode2?.toLowerCase()
                    );
                    if (exactMatch) {
                        this.addItemToInvoice(exactMatch);
                        this.clearSearchResults();
                    }
                } catch (error) {
                    console.error('Error searching products:', error);
                    current.searchResults = [];
                }
            },
            handleSearchEnter() {
                const current = this.currentInvoice();
                if (current.selectedSearchResultIndex !== -1 && current.searchResults.length > 0) {
                    this.addItemToInvoice(current.searchResults[current.selectedSearchResultIndex]);
                } else if (current.searchResults.length === 1) {
                    this.addItemToInvoice(current.searchResults[0]);
                }
                this.clearSearchResults();
            },
            selectNextResult() {
                const current = this.currentInvoice();
                if (current.searchResults.length > 0) {
                    current.selectedSearchResultIndex = (current.selectedSearchResultIndex + 1) %
                        current.searchResults.length;
                }
            },
            selectPrevResult() {
                const current = this.currentInvoice();
                if (current.searchResults.length > 0) {
                    current.selectedSearchResultIndex = (current.selectedSearchResultIndex - 1 +
                        current.searchResults.length) % current.searchResults.length;
                }
            },
            clearSearchResults() {
                const current = this.currentInvoice();
                current.searchQuery = '';
                current.searchResults = [];
                current.selectedSearchResultIndex = -1;
            },
            addItemToInvoice(product) {
                const current = this.currentInvoice();
                const existingItemIndex = current.invoiceItems.findIndex(item => item.item_code ===
                    product.item_code);
                if (existingItemIndex !== -1) {
                    current.invoiceItems[existingItemIndex].quantity++;
                } else {
                    current.invoiceItems.unshift({
                        id: product.id,
                        title: product.name,
                        item_code: product.item_code,
                        stock: Number(product.stock || 0),
                        mrp: Number(product.mrp || 0),
                        purchase_price: Number(product.purchase_price || 0),
                        sell_price: Number(product.sell_price || 0),
                        shopping_wallet_price: Number(product.shop_wallet || 0),
                        quantity: 1,
                        gst_rate: Number(product.gst_rate || 0),
                        cgst: Number(product.cgst || 0),
                        sgst: Number(product.sgst || 0),
                        gst_amount_per_unit: (Number(product.purchase_price || 0) * (Number(
                            product.gst_rate || 0) / 100))
                    });
                }
                this.clearSearchResults();
                this.calculateInvoiceTotals();
                this.$nextTick(() => {
                    this.$refs.searchInput.focus();
                });
            },
            selectItem(index) {
                this.currentInvoice().selectedItemIndex = index;
            },
            deleteSelectedItem() {
                const current = this.currentInvoice();
                if (current.selectedItemIndex !== -1) {
                    if (confirm('Are you sure you want to remove this item?')) {
                        current.invoiceItems.splice(current.selectedItemIndex, 1);
                        current.selectedItemIndex = -1; // Deselect after removal
                        this.calculateInvoiceTotals();
                    }
                } else {
                    alert('Please select an item to delete.');
                }
            },
            updateItemDetails(index) {
                const item = this.currentInvoice().invoiceItems[index];
                item.quantity = Math.max(1, item.quantity);
                item.purchase_price = Math.max(0, item
                    .purchase_price);
                item.sell_price = Math.max(0, item.sell_price);
                this.calculateInvoiceTotals();
            },
            calculateInvoiceTotals() {
                const current = this.currentInvoice();
                let subTotalExcludingGST =
                    0; // This will still be based on purchase_price for the subtotal of goods purchased
                let gstTotal = 0;
                let taxableAmount =
                    0; // This will be the subtotal of items that are subject to GST calculation based on sell_price
                current.invoiceItems.forEach(item => {
                    const itemPurchasePrice = Number(item.purchase_price || 0);
                    const itemSellPrice = Number(item.sell_price || 0); // Get sell price
                    const itemQuantity = Number(item.quantity || 0);
                    subTotalExcludingGST += itemPurchasePrice * itemQuantity;
                    const itemGSTAmount = (Number(item.gst_amount_per_unit || 0) *
                        itemQuantity);
                    gstTotal += itemGSTAmount;
                    taxableAmount += itemSellPrice * itemQuantity;
                });
                current.subTotalExcludingGST = subTotalExcludingGST - gstTotal;
                current.gstTotal = gstTotal;
                current.subTotalIncludingGST = subTotalExcludingGST;
                current.totalAmount = current.subTotalIncludingGST - (Number(current
                    .discountAmount) || 0) + (Number(current.additionalCharge) || 0);
                current.totalAmount = Math.max(0, current.totalAmount);
            },
            openPurchasePriceChangeModal() {
                const current = this.currentInvoice();
                if (current.selectedItemIndex !== -1) {
                    this.tempPurchasePrice = current.invoiceItems[current.selectedItemIndex]
                        .purchase_price;
                    this.purchasePriceChangeModal.show();
                } else {
                    alert('Please select an item to change its price.');
                }
            },
            confirmChangePurchasePrice() {
                const current = this.currentInvoice();
                if (current.selectedItemIndex !== -1) {
                    current.invoiceItems[current.selectedItemIndex].purchase_price = this
                        .tempPurchasePrice;
                    this.updateItemDetails(current.selectedItemIndex);
                    this.purchasePriceChangeModal.hide();
                }
            },
            openQuantityChangeModal() {
                const current = this.currentInvoice();
                if (current.selectedItemIndex !== -1) {
                    this.tempQty = current.invoiceItems[current.selectedItemIndex].quantity;
                    this.qtyChangeModal.show();
                } else {
                    alert('Please select an item to change its quantity.');
                }
            },
            confirmChangeQty() {
                const current = this.currentInvoice();
                if (current.selectedItemIndex !== -1) {
                    current.invoiceItems[current.selectedItemIndex].quantity = Math.max(1, this
                        .tempQty);
                    this.updateItemDetails(current.selectedItemIndex);
                    this.qtyChangeModal.hide();
                }
            },
            openDiscountModal() {
                this.tempDiscount = this.currentInvoice().discountAmount;
                this.tempDiscountNote = this.currentInvoice().discountNote;
                this.discountModal.show();
            },
            confirmDiscount() {
                const current = this.currentInvoice();
                current.discountAmount = Math.max(0, this.tempDiscount);
                current.discountNote = this.tempDiscountNote;
                this.calculateInvoiceTotals();
                this.discountModal.hide();
            },
            openAdditionalChargeModal() {
                this.tempCharge = this.currentInvoice().additionalCharge;
                this.tempChargeNote = this.currentInvoice().additionalChargeNote;
                this.chargeModal.show();
            },
            confirmCharge() {
                const current = this.currentInvoice();
                current.additionalCharge = Math.max(0, this.tempCharge);
                current.additionalChargeNote = this.tempChargeNote;
                this.calculateInvoiceTotals();
                this.chargeModal.hide();
            },
            async saveInvoiceAndPrint() {
                const current = this.currentInvoice();
                if (!current.supplierDetails) {
                    alert('Please select a supplier first.');
                    return;
                }
                if (!current.invoiceItems.length) {
                    alert('Please add items to the invoice before saving.');
                    return;
                }
                const invoiceData = {
                    product_purchase_id: current.supplierDetails.id,
                    customer_name: current.supplierDetails.customer_name,
                    gstin: current.supplierDetails.gstin,
                    phone_number: current.supplierDetails.phone_number,
                    address: current.supplierDetails.address,
                    branch: current.supplierDetails.branch,
                    invoice_items: current.invoiceItems.map(item => ({
                        product_id: item.id,
                        quantity: item.quantity,
                        purchase_price: item.purchase_price,
                        sell_price: item.sell_price,
                        gst_rate: item
                            .gst_rate, // Use gst_rate for consistency if present
                        cgst: item.cgst,
                        sgst: item.sgst,
                    })),
                    discount_amount: current.discountAmount,
                    discount_note: current.discountNote,
                    additional_charge: current.additionalCharge,
                    additional_charge_note: current.additionalChargeNote,
                    gst_total: current.gstTotal,
                    sub_total_excluding_gst: current.subTotalExcludingGST,
                    sub_total_including_gst: current.subTotalIncludingGST,
                    total_amount: current.totalAmount,
                    print_type: this.printType, // Include print type
                };
                try {
                    const response = await fetch('{{ route('purchase_store') }}', {
                        method: 'POST',
                        headers: {
                            'Content-Type': 'application/json',
                            'X-CSRF-TOKEN': document.querySelector(
                                'meta[name="csrf-token"]').content
                        },
                        body: JSON.stringify(invoiceData)
                    });
                    const data = await response.json();
                    if (data.success) {
                        alert('Purchase invoice saved successfully! Invoice ID: ' + data
                            .invoice_id);
                        if (data.print_url) {
                            window.open(data.print_url, '_blank');
                        }
                        this.resetCurrentInvoice();
                        this.loadHeldInvoices();
                    } else {
                        alert('Failed to save purchase invoice: ' + (data.message ||
                            'Unknown error.'));
                    }
                } catch (error) {
                    console.error('Error saving purchase invoice:', error);
                    alert('An error occurred while saving the purchase invoice.');
                }
            },
            async saveHeldInvoice() {
                const current = this.currentInvoice();
                if (!current.invoiceItems.length) {
                    alert('Cannot hold an empty invoice.');
                    return;
                }
                const heldInvoiceData = {
                    invoice_id: current.id,
                    supplierDetails: current.supplierDetails,
                    invoice_items: current.invoiceItems,
                    discount_amount: current.discountAmount,
                    discount_note: current.discountNote,
                    additional_charge: current.additionalCharge,
                    additional_charge_note: current.additionalChargeNote,
                    gst_total: current.gstTotal,
                    sub_total_excluding_gst: current.subTotalExcludingGST,
                    sub_total_including_gst: current.subTotalIncludingGST,
                    total_amount: current.totalAmount,
                    timestamp: new Date().toLocaleString()
                };
                this.heldInvoices.push(heldInvoiceData);
                localStorage.setItem('heldPurchaseInvoices', JSON.stringify(this.heldInvoices));
                alert('Invoice held successfully!');
                this.resetCurrentInvoice(); // Clear current invoice after holding
                this.heldInvoicesModal.hide();
            },
            loadHeldInvoices() {
                const storedHeldInvoices = localStorage.getItem('heldPurchaseInvoices');
                this.heldInvoices = storedHeldInvoices ? JSON.parse(storedHeldInvoices) : [];
            },
            showHeldInvoicesModal() {
                this.loadHeldInvoices();
                this.heldInvoicesModal.show();
            },
            resumeInvoice(invoiceIdToResume) {
                const invoiceIndex = this.heldInvoices.findIndex(hi => hi.invoice_id ===
                    invoiceIdToResume);
                if (invoiceIndex !== -1) {
                    const heldInvoice = this.heldInvoices[invoiceIndex];
                    let targetInvoice = this.invoices.find(inv => inv.id === heldInvoice
                        .invoice_id);
                    if (!targetInvoice) {
                        // If the invoice tab for this ID doesn't exist, create a new one
                        const newId = heldInvoice.invoice_id; // Use the held invoice's ID
                        this.invoices.push({
                            id: newId,
                            name: `Invoice ${newId}`,
                            supplierIdentifier: heldInvoice.supplierDetails ? heldInvoice
                                .supplierDetails.phone_number || heldInvoice.supplierDetails
                                .customer_name : '', // Use customer_name
                            supplierDetails: heldInvoice.supplierDetails,
                            supplierDetailsLoading: false,
                            searchQuery: '',
                            searchResults: [],
                            selectedSearchResultIndex: -1,
                            invoiceItems: heldInvoice.invoice_items || [],
                            selectedItemIndex: -1,
                            discountAmount: heldInvoice.discount_amount || 0,
                            discountNote: heldInvoice.discount_note || '',
                            additionalCharge: heldInvoice.additional_charge || 0,
                            additionalChargeNote: heldInvoice.additional_charge_note || '',
                            gstTotal: heldInvoice.gst_total || 0,
                            subTotalExcludingGST: heldInvoice.sub_total_excluding_gst || 0,
                            subTotalIncludingGST: heldInvoice.sub_total_including_gst || 0,
                            totalAmount: heldInvoice.total_amount || 0,
                        });
                        targetInvoice = this.invoices.find(inv => inv.id === heldInvoice
                            .invoice_id);
                    } else {
                        // If the invoice tab already exists, update its details
                        Object.assign(targetInvoice, {
                            supplierIdentifier: heldInvoice.supplierDetails ? heldInvoice
                                .supplierDetails.phone_number || heldInvoice.supplierDetails
                                .customer_name : '', // Use customer_name
                            supplierDetails: heldInvoice.supplierDetails,
                            invoiceItems: heldInvoice.invoice_items || [],
                            discountAmount: heldInvoice.discount_amount || 0,
                            discountNote: heldInvoice.discount_note || '',
                            additionalCharge: heldInvoice.additional_charge || 0,
                            additionalChargeNote: heldInvoice.additional_charge_note || '',
                        });
                    }
                    this.activateInvoice(heldInvoice.invoice_id);
                    this.heldInvoices.splice(invoiceIndex, 1);
                    localStorage.setItem('heldPurchaseInvoices', JSON.stringify(this.heldInvoices));
                    this.calculateInvoiceTotals();
                    this.heldInvoicesModal.hide();
                    alert('Invoice resumed successfully!');
                }
            },
            deleteHeldInvoice(invoiceIdToDelete) {
                if (confirm(
                        'Are you sure you want to delete this held invoice? This action cannot be undone.'
                    )) {
                    this.heldInvoices = this.heldInvoices.filter(hi => hi.invoice_id !==
                        invoiceIdToDelete);
                    localStorage.setItem('heldPurchaseInvoices', JSON.stringify(this.heldInvoices));
                    alert('Held invoice deleted.');
                    this.loadHeldInvoices(); // Refresh the modal list
                }
            },
openBarcodeModal(item) {
    this.barcodeModalItem = item;
    this.numberOfLabelsToPrint = item.quantity || 1;
    this.barcodeModal.show();
    setTimeout(() => {
        JsBarcode("#purchaseBarcodeModalSvg", item.item_code, {
            format: "CODE128",
            lineColor: "#000",
            width: 2,
            height: 50,
            displayValue: false
        });
    }, 300);
},
printBarcode() {
    if (!this.barcodeModalItem) return;
    const item = this.barcodeModalItem;
    const companyName = "{{ Auth::user()->name ?? 'Company Name' }}";
    // Generate barcode as PNG (stable for print)
    const canvas = document.createElement("canvas");
    JsBarcode(canvas, item.item_code, {
        format: "CODE128",
        lineColor: "#000",
        width: 2,
        height: 50,
        displayValue: false
    });
    const barcodeImg = canvas.toDataURL("image/png");
    const labelHTML = `
    <!DOCTYPE html>
    <html>
    <head>
      <meta charset="UTF-8">
      <title>Barcode Labels</title>
      <style>
        @page {
            size: 110mm 25mm; 
            margin: 0; 
        }
        body {
            margin: 0; 
            padding: 0; 
            font-family: Arial, sans-serif;
            font-size: 6px;
            width: 110mm;
        }
        .label-container {
            display: grid;
            grid-template-columns: repeat(3, 1fr); /* always 3 per row */
            width: 110mm;
            height: 25mm;
            margin: 0;
            padding: 0;
        }
        .label {
            width: 36.66mm; 
            height: 25mm;
            display: flex;
            flex-direction: column;
            align-items: center;
            justify-content: flex-start;
            text-align: center;
            overflow: hidden;
            padding: 0.2mm;
        }
        .company { font-weight: bold; font-size: 6.5px; }
        .item-code { font-size: 7px; font-weight: bold; }
        .title { font-size: 6px; }
        .price { font-size: 7px; font-weight: bold; }
        img.barcode { width: 100px; }
        @media print {
            html, body {
                width: 110mm!important;
                height: 25mm!important;
                margin: 0;
                padding: 0;
            }
        }
      </style>
    </head>
    <body>
      <div class="label-container">
        ${Array.from({ length: 3 }).map(() => `
          <div class="label">
            <div class="company">${companyName.slice(0, 25)}</div>
            <img class="barcode" src="${barcodeImg}" />
            <div class="item-code">${item.item_code}</div>
            <div class="title">${item.title || item.name}</div>
            <div class="price">SP: ₹${Number(item.sell_price || 0).toFixed(2)}</div>
          </div>
        `).join('')}
      </div>
    </body>
    </html>
    `;
    const printWindow = window.open('', '_blank');
    printWindow.document.write(labelHTML);
    printWindow.document.close();
    printWindow.onload = () => {
        printWindow.print();
        this.barcodeModal.hide();
    };
},
            resetCurrentInvoice() {
                const current = this.currentInvoice();
                Object.assign(current, {
                    supplierIdentifier: '',
                    supplierDetails: null,
                    supplierDetailsLoading: false,
                    searchQuery: '',
                    searchResults: [],
                    selectedSearchResultIndex: -1,
                    invoiceItems: [],
                    selectedItemIndex: -1,
                    discountAmount: 0,
                    discountNote: '',
                    additionalCharge: 0,
                    additionalChargeNote: '',
                    gstTotal: 0,
                    subTotalExcludingGST: 0,
                    subTotalIncludingGST: 0,
                    totalAmount: 0,
                });
            },
            openSettings() {
                this.settingsModal.show();
            },
            saveSettings() {
                localStorage.setItem('purchasePrintType', this.printType);
                this.settingsModal.hide();
                alert('Settings saved!');
            }
        }));
    });
</script>
