<!DOCTYPE html>
<html lang="en" x-data="posBillingEdit({{ $offlineSale }})" @keydown.window="handleGlobalKeys($event)">

<head>
    <meta charset="UTF-8" />
    <meta name="viewport" content="width=device-width, initial-scale=1.0" />
    <title>Edit Invoice #{{ $offlineSale->order_id }}</title>
    <link href="https://cdn.jsdelivr.net/npm/bootstrap@5.3.0/dist/css/bootstrap.min.css" rel="stylesheet" />
    <link href="https://cdn.jsdelivr.net/npm/bootstrap-icons@1.10.5/font/bootstrap-icons.css" rel="stylesheet" />
    <script defer src="https://cdn.jsdelivr.net/npm/alpinejs@3/dist/cdn.min.js"></script>
    <meta name="csrf-token" content="{{ csrf_token() }}">
   

   <style>
        .pos-search-results {
            max-height: 200px;
            overflow-y: auto;
            z-index: 1000;
            position: absolute;
            width: calc(100% - 2rem);
            background-color: white;
            border: 1px solid #dee2e6;
            border-top: none;
            left: 1rem;
            right: 1rem;
        }

        .table-scroll-container {
            max-height: calc(100vh - 450px);
            overflow-y: auto;
        }

        .table thead th {
            position: sticky;
            top: 0;
            background-color: #f8f9fa;
            z-index: 10;
        }

        .customer-details-compact p {
            margin-bottom: 0;
            line-height: 1.2;
        }
    </style>


</head>

<body>

    <div class="container-fluid bg-light border-bottom py-2 px-3 d-flex justify-content-between align-items-center">
        <a href="{{ url()->previous() }}" class="btn btn-outline-secondary btn-sm">
            <i class="bi bi-arrow-left"></i> <strong>Back</strong> [ESC]
        </a>
        <h6 class="m-0">Editing Invoice #<span x-text="bill.order_id"></span></h6>
        <div class="d-flex gap-2">
            <button class="btn btn-outline-secondary btn-sm" @click="openSettings()">
                <strong>Settings</strong> [CTRL+S]
            </button>
        </div>
    </div>

    <div class="container-fluid p-3">
        <div class="d-flex flex-wrap align-items-center gap-3 px-2 py-2 bg-light border rounded">
            <div class="card-header bg-primary text-white py-2 px-3 rounded-top">
                <h6 class="mb-0">Customer Details</h6>
            </div>
            <div class="flex-grow-1">
                <input type="text" class="form-control" x-model.debounce.500ms="bill.customerIdentifier"
                    @input.debounce.500ms="fetchCustomerDetails" placeholder="Type customer phone or name to search...">
            </div>
            <template x-if="bill.customerDetails">
                <div class="d-flex flex-wrap align-items-center gap-3">
                    <div>
                        <strong x-text="bill.customerDetails.name"></strong>
                        <small class="text-muted">(ID: <span x-text="bill.customerDetails.user_id"></span>)</small>
                    </div>
                    <div>
                        <span class="badge bg-success">
                            ₹<span x-text="bill.customerDetails.cash_wallet_balance.toFixed(2)"></span> Cash
                        </span>
                        <span class="badge bg-info text-dark">
                            ₹<span x-text="bill.customerDetails.shopping_wallet_balance.toFixed(2)"></span> Shopping
                        </span>
                    </div>
                    <div class="text-muted small">
                        <i class="bi bi-telephone"></i>
                        <span x-text="bill.customerDetails.phone_number"></span>
                    </div>
                    <div class="text-muted small">
                        <i class="bi bi-geo-alt"></i>
                        <span x-text="bill.customerDetails.address"></span>
                    </div>
                </div>
            </template>

            <div class="text-muted small fst-italic"
                x-show="bill.customerIdentifier && !bill.customerDetails && !bill.customerDetailsLoading">
                No customer found. <a href="#" @click.prevent="openNewCustomerModal">Register new customer?</a>
            </div>
            <div class="text-info small fst-italic" x-show="bill.customerDetailsLoading">
                Searching...
            </div>
        </div>

        <div class="row mt-3">
            <div class="col-md-8 border-end position-relative">
                <div class="d-flex gap-2 mb-3">
                    <button class="btn btn-outline-secondary btn-sm" @click="openPriceChangeModal">Change Price
                        [CTRL+P]</button>
                    <button class="btn btn-outline-secondary btn-sm" @click="openQuantityChangeModal">Change QTY
                        [CTRL+Q]</button>
                    <button class="btn btn-outline-secondary btn-sm" @click="deleteSelectedItem()">Delete Item
                        [DEL]</button>
                </div>
                <div class="input-group mb-3">
                    <span class="input-group-text"><i class="bi bi-search"></i></span>
                    <input x-ref="searchInput" class="form-control" type="text" placeholder="Search item/code"
                        x-model.debounce.300ms="bill.searchQuery" @input.debounce.300ms="searchProducts"
                        @keydown.enter.prevent="handleSearchEnter" @keydown.arrow-down.prevent="selectNextResult"
                        @keydown.arrow-up.prevent="selectPrevResult" @keydown.escape.stop="clearSearchResults"
                        @focus="bill.searchQuery = ''">
                    <span class="input-group-text">[F1]</span>
                </div>
                <ul class="list-group pos-search-results" x-show="bill.searchResults.length">
                    <template x-for="(prod, idx) in bill.searchResults" :key="prod.item_identifier">
                        <li class="list-group-item list-group-item-action"
                            :class="{ 'active': idx === bill.selectedSearchResultIndex }"
                            @click="addItemToBill(prod)" @mouseover="bill.selectedSearchResultIndex = idx">
                            <span x-text="prod.name"></span> - <small x-text="prod.item_code"></small> (₹ <span
                                x-text="Number(prod.sell_price || 0).toFixed(2)"></span>)
                        </li>
                    </template>
                </ul>

                <div class="table">
                    <table class="table table-bordered text-center">
                        <thead class="table-light">
                            <tr>
                                <th>#</th>
                                <th>Item</th>
                                <th>Code</th>
                                <th>MRP</th>
                                <th>SP</th>
                                <th>QTY</th>
                                <th>Amount</th>
                            </tr>
                        </thead>
                        <tbody>
                            <template x-if="!bill.billItems.length">
                                <tr>
                                    <td colspan="7" class="py-5 text-muted">
                                        <i class="bi bi-box-seam fs-1"></i>
                                        <p>Add items...</p>
                                    </td>
                                </tr>
                            </template>

                            <template x-for="(item, idx) in bill.billItems" :key="item.item_identifier + idx">
                                <tr :class="{ 'table-primary': bill.selectedItemIndex === idx }"
                                    @click="selectItem(idx)">
                                    <td x-text="idx + 1"></td>
                                    <td x-text="item.name"></td>
                                    <td x-text="item.item_code"></td>
                                    <td x-text="(Number(item.mrp) || 0).toFixed(2)"></td>
                                    <td x-text="(Number(item.sell_price) || 0).toFixed(2)"></td>
                                    <td>
                                        <input type="number" class="form-control form-control-sm text-center"
                                            style="width:80px;margin:auto;" x-model.number="item.quantity"
                                            @change="updateItemQuantity(idx)" min="1">
                                    </td>
                                    <td x-text="((Number(item.sell_price) || 0) * item.quantity).toFixed(2)"></td>
                                </tr>
                            </template>
                        </tbody>
                    </table>
                </div>
            </div>

            <div class="col-md-4">
                <div class="d-flex gap-2 mb-2">
                    <button class="btn btn-outline-secondary btn-sm flex-fill" @click="openDiscountModal">Add Discount
                        [F2]</button>
                    <button class="btn btn-outline-secondary btn-sm flex-fill" @click="openAdditionalChargeModal">Add
                        Charge [F3]</button>
                </div>

                <div class="card p-3 mb-2">
                    <h6>Bill Details</h6>
                    <div class="d-flex justify-content-between">
                        <span>Sub Total</span>
                        <span>₹ <span x-text="bill.subTotal.toFixed(2)"></span></span>
                    </div>
                    <div class="d-flex justify-content-between">
                        <span>GST</span>
                        <span>₹ <span x-text="bill.gstTotal.toFixed(2)"></span></span>
                    </div>

                    <template x-if="bill.discountAmount > 0">
                        <div class="d-flex justify-content-between">
                            <span>Discount</span>
                            <span>- ₹ <span x-text="bill.discountAmount.toFixed(2)"></span></span>
                        </div>
                    </template>

                    <template x-if="bill.additionalCharge > 0">
                        <div class="d-flex justify-content-between">
                            <span>Charge</span>
                            <span>+ ₹ <span x-text="bill.additionalCharge.toFixed(2)"></span></span>
                        </div>
                    </template>
                    <hr>
                    <div class="d-flex justify-content-between fw-bold text-success">
                        <span>Total</span>
                        <span>₹ <span x-text="bill.totalAmount.toFixed(2)"></span></span>
                    </div>
                </div>

                <div class="card p-3 mb-2">
                    <h6>Wallet Use</h6>
                    <div class="form-check mb-2">
                        <input class="form-check-input" type="checkbox" id="use_cash_wallet" x-model="bill.useCashWallet"
                            @change="calculateAmountPayable"
                            :disabled="!bill.customerDetails || bill.customerDetails.cash_wallet_balance <= 0">
                        <label class="form-check-label">Use Cash Wallet (Bal ₹<span
                                x-text="bill.customerDetails?.cash_wallet_balance.toFixed(2) || '0.00'"></span>)</label>
                        <small x-show="bill.cashWalletDeduction > 0" class="text-muted">Applied ₹<span
                                x-text="bill.cashWalletDeduction.toFixed(2)"></span></small>
                    </div>
                    <div class="form-check mb-2">
                        <input class="form-check-input" type="checkbox" id="use_shopping_wallet"
                            x-model="bill.useShoppingWallet" @change="calculateAmountPayable"
                            :disabled="!bill.customerDetails || (bill.customerDetails.shopping_wallet_balance <= 0 && bill.totalApplicableShoppingWalletFromItems <= 0)">
                        <label class="form-check-label">Use Shopping Wallet (Bal ₹<span
                                x-text="bill.customerDetails?.shopping_wallet_balance.toFixed(2) || '0.00'"></span>)</label>
                        <small
                            x-show="bill.totalApplicableShoppingWalletFromItems > 0 && !bill.useShoppingWallet"
                            class="text-muted"> (Applicable ₹<span
                                x-text="bill.totalApplicableShoppingWalletFromItems.toFixed(2)"></span>)</small>
                        <small x-show="bill.shoppingWalletDeduction > 0" class="text-muted"> Applied ₹<span
                                x-text="bill.shoppingWalletDeduction.toFixed(2)"></span></small>
                    </div>
                    <div class="mb-3">
                        <label for="payment_mode" class="form-label fw-bold">Select Payment Mode</label>
                        <select id="payment_mode" class="form-select" x-model="bill.paymentMode">
                            <option value="">--Select--</option>
                            <option value="upi">UPI</option>
                            <option value="cash">Cash</option>
                            <option value="card">Card</option>
                        </select>
                    </div>
                    <div class="d-flex justify-content-between fw-bold"><span>Payable</span><span>₹ <span
                                x-text="bill.amountPayable.toFixed(2)"></span></span></div>
                </div>


                <div class="d-grid gap-2">
    <button class="btn btn-primary btn-lg" @click="updateAndPrintBill" :disabled="!bill.billItems.length">Update & Print [F6]</button>
    <button class="btn btn-secondary btn-lg" @click="updateBill" :disabled="!bill.billItems.length">Save Changes</button>
</div>
            </div>
        </div>
    </div>

    <div class="modal fade" id="priceChangeModal" tabindex="-1">
        <div class="modal-dialog modal-sm modal-dialog-centered">
            <div class="modal-content">
                <div class="modal-header">
                    <h5>Change Price</h5><button class="btn-close" data-bs-dismiss="modal"></button>
                </div>
                <div class="modal-body"><input type="number" class="form-control" x-model.number="tempPrice"></div>
                <div class="modal-footer"><button class="btn btn-primary" @click="confirmChangePrice">Apply</button><button
                        class="btn btn-outline-secondary" data-bs-dismiss="modal">Cancel</button></div>
            </div>
        </div>
    </div>
    <div class="modal fade" id="qtyChangeModal" tabindex="-1">
        <div class="modal-dialog modal-sm modal-dialog-centered">
            <div class="modal-content">
                <div class="modal-header">
                    <h5>Change Quantity</h5><button class="btn-close" data-bs-dismiss="modal"></button>
                </div>
                <div class="modal-body"><input type="number" class="form-control" x-model.number="tempQty" min="1"></div>
                <div class="modal-footer"><button class="btn btn-primary" @click="confirmChangeQty">Apply</button><button
                        class="btn btn-outline-secondary" data-bs-dismiss="modal">Cancel</button></div>
            </div>
        </div>
    </div>
    <div class="modal fade" id="discountModal" tabindex="-1">
        <div class="modal-dialog modal-sm modal-dialog-centered">
            <div class="modal-content">
                <div class="modal-header">
                    <h5>Add Discount</h5><button class="btn-close" data-bs-dismiss="modal"></button>
                </div>
                <div class="modal-body"><input type="number" class="form-control mb-2" x-model.number="tempDiscount"
                        placeholder="Amount"><input class="form-control" x-model="tempDiscountNote"
                        placeholder="Reason (opt)"></div>
                <div class="modal-footer"><button class="btn btn-primary" @click="confirmDiscount">Apply</button><button
                        class="btn btn-outline-secondary" data-bs-dismiss="modal">Cancel</button></div>
            </div>
        </div>
    </div>
    <div class="modal fade" id="chargeModal" tabindex="-1">
        <div class="modal-dialog modal-sm modal-dialog-centered">
            <div class="modal-content">
                <div class="modal-header">
                    <h5>Add Charge</h5><button class="btn-close" data-bs-dismiss="modal"></button>
                </div>
                <div class="modal-body"><input type="number" class="form-control mb-2" x-model.number="tempCharge"
                        placeholder="Amount"><input class="form-control" x-model="tempChargeNote"
                        placeholder="Reason (opt)"></div>
                <div class="modal-footer"><button class="btn btn-primary" @click="confirmCharge">Apply</button><button
                        class="btn btn-outline-secondary" data-bs-dismiss="modal">Cancel</button></div>
            </div>
        </div>
    </div>
    <div class="modal fade" id="settingsModal" tabindex="-1">
        <div class="modal-dialog modal-sm modal-dialog-centered">
            <div class="modal-content">
                <div class="modal-header">
                    <h5>Select Print Type</h5>
                    <button class="btn-close" data-bs-dismiss="modal"></button>
                </div>
                <div class="modal-body">
                    <div class="form-check">
                        <input class="form-check-input" type="radio" x-model="printType" value="a4"
                            id="print_a4">
                        <label class="form-check-label" for="print_a4">A4 Invoice</label>
                    </div>
                    <div class="form-check">
                        <input class="form-check-input" type="radio" x-model="printType" value="thermal"
                            id="print_thermal">
                        <label class="form-check-label" for="print_thermal">Thermal (3 inch)</label>
                    </div>
                </div>
                <div class="modal-footer">
                    <button class="btn btn-primary" @click="saveSettings">Save</button>
                    <button class="btn btn-outline-secondary" data-bs-dismiss="modal">Close</button>
                </div>
            </div>
        </div>
    </div>
    <div class="modal fade" id="newCustomerModal" tabindex="-1">
        <div class="modal-dialog modal-sm modal-dialog-centered">
            <div class="modal-content">
                <div class="modal-header">
                    <h5>Register New Customer</h5>
                    <button class="btn-close" data-bs-dismiss="modal"></button>
                </div>
                <div class="modal-body">
                    <div class="mb-3">
                        <label for="new_customer_phone" class="form-label">Phone Number</label>
                        <input type="text" class="form-control" id="new_customer_phone"
                            x-model="newCustomer.phone_number" disabled>
                    </div>
                    <div class="mb-3">
                        <label for="new_customer_name" class="form-label">Name <span
                                class="text-danger">*</span></label>
                        <input type="text" class="form-control" id="new_customer_name" x-model="newCustomer.name">
                    </div>
                    <div class="mb-3">
                        <label for="new_customer_email" class="form-label">Email (Optional)</label>
                        <input type="email" class="form-control" id="new_customer_email"
                            x-model="newCustomer.email">
                    </div>
                </div>
                <div class="modal-footer">
                    <button class="btn btn-primary" @click="confirmNewCustomer">Register</button>
                    <button class="btn btn-outline-secondary" data-bs-dismiss="modal">Cancel</button>
                </div>
            </div>
        </div>
    </div>


   
</body>

</html>

<script src="https://cdn.jsdelivr.net/npm/bootstrap@5.3.0/dist/js/bootstrap.bundle.min.js"></script>
<script>
    function posBillingEdit(invoiceData) {
        return {
            // Initial State
            invoice: invoiceData,
            bill: {
                order_id: invoiceData.order_id,
                billItems: invoiceData.sale_items.map(item => {
                    const sellPrice = Number(item.sell_price || 0);
                    const gstRate = Number(item.gst_rate || 0);
                    const basePrice = sellPrice / (1 + (gstRate / 100));
                    return {
                        ...item,
                        sell_price: sellPrice,
                        base_price: basePrice,
                        cgst: (basePrice * (gstRate / 200)),
                        sgst: (basePrice * (gstRate / 200)),
                        quantity: Number(item.quantity),
                        shopping_wallet_price: Number(item.shopping_wallet_price || 0)
                    };
                }),
                customerDetails: invoiceData.customer_details,
                customerIdentifier: invoiceData.customer_details?.phone_number || '',
                customerDetailsLoading: false,

                subTotal: 0,
                gstTotal: 0,
                discountAmount: Number(invoiceData.discount),
                additionalCharge: Number(invoiceData.additional_charge),
                cashWalletDeduction: Number(invoiceData.cash_wallet_deduction),
                shoppingWalletDeduction: Number(invoiceData.shopping_wallet_deduction),
                totalAmount: 0,
                amountPayable: 0,
                paymentMode: invoiceData.payment_mode,
                useCashWallet: invoiceData.cash_wallet_deduction > 0,
                useShoppingWallet: invoiceData.shopping_wallet_deduction > 0,
                totalApplicableShoppingWalletFromItems: 0,
                soldBy: invoiceData.pos_user_id,

                searchQuery: '',
                searchResults: [],
                selectedSearchResultIndex: -1,

                selectedItemIndex: -1,
            },

            // Modal State
            tempPrice: 0,
            tempQty: 0,
            tempDiscount: 0,
            tempDiscountNote: '',
            tempCharge: 0,
            tempChargeNote: '',
            printType: localStorage.getItem('posPrintType') || 'thermal',
            newCustomer: {
                phone_number: '',
                name: '',
                email: ''
            },
            init() {
                this.calculateTotals();
            },

            // calculateTotals() {
            //     const bill = this.bill;
            //     bill.subTotal = bill.billItems.reduce((sum, item) =>
            //         sum + (Number(item.base_price || 0) * Number(item.quantity || 0)), 0
            //     );
            //     bill.gstTotal = bill.billItems.reduce((sum, item) =>
            //         sum + ((Number(item.cgst || 0) + Number(item.sgst || 0)) * Number(item.quantity || 0)), 0
            //     );

            //     let calculatedTotal = Number(bill.subTotal || 0) + Number(bill.gstTotal || 0);

            //     calculatedTotal = calculatedTotal - Number(bill.discountAmount || 0) + Number(bill.additionalCharge || 0);
            //     bill.totalAmount = Math.max(0, calculatedTotal);

            //     this.calculateAmountPayable();
            // },

            calculateTotals() {
    const bill = this.bill;

    bill.subTotal = bill.billItems.reduce((sum, item) =>
        sum + (Number(item.base_price || 0) * Number(item.quantity || 0)), 0
    );

    bill.gstTotal = 0; // since already included in base_price

    let calculatedTotal = Number(bill.subTotal || 0);

    calculatedTotal = calculatedTotal - Number(bill.discountAmount || 0) + Number(bill.additionalCharge || 0);
    bill.totalAmount = Math.max(0, calculatedTotal);

    this.calculateAmountPayable();
},





            calculateAmountPayable() {
                let payable = Number(this.bill.totalAmount || 0);
                this.bill.cashWalletDeduction = 0;
                this.bill.shoppingWalletDeduction = 0;

                if (this.bill.useCashWallet && this.bill.customerDetails) {
                    const maxDeduction = Number(this.bill.customerDetails.cash_wallet_balance || 0);
                    this.bill.cashWalletDeduction = Math.min(payable, maxDeduction);
                    payable = payable - this.bill.cashWalletDeduction;
                }

                if (this.bill.useShoppingWallet && this.bill.customerDetails) {
                    const maxDeduction = Number(this.bill.customerDetails.shopping_wallet_balance || 0);

                    const applicableAmount = this.bill.billItems.reduce((sum, item) => {
                        const sellPrice = Number(item.sell_price || 0);
                        const quantity = Number(item.quantity || 0);
                        const swPercentage = Number(item.shopping_wallet_percentage || 0);
                        return sum + (sellPrice * quantity * (swPercentage / 100));
                    }, 0);

                    this.bill.totalApplicableShoppingWalletFromItems = applicableAmount;

                    const actualDeduction = Math.min(payable, maxDeduction, applicableAmount);
                    this.bill.shoppingWalletDeduction = actualDeduction;
                    payable = payable - this.bill.shoppingWalletDeduction;
                }

                this.bill.amountPayable = Math.max(0, payable);
            },

            selectItem(index) {
                this.bill.selectedItemIndex = index;
            },

            deleteSelectedItem() {
                if (this.bill.selectedItemIndex !== -1) {
                    this.bill.billItems.splice(this.bill.selectedItemIndex, 1);
                    this.bill.selectedItemIndex = -1;
                    this.calculateTotals();
                }
            },
            async fetchCustomerDetails() {
                if (this.bill.customerIdentifier.length < 3) {
                    this.bill.customerDetails = null;
                    this.bill.useCashWallet = false;
                    this.bill.useShoppingWallet = false;
                    this.calculateTotals();
                    return;
                }
                this.bill.customerDetailsLoading = true;
                try {
                    const response = await fetch(`/api/customers/search?query=${this.bill.customerIdentifier}`);
                    const data = await response.json();
                    if (data.length > 0) {
                        this.bill.customerDetails = data[0];
                    } else {
                        this.bill.customerDetails = null;
                    }
                } catch (error) {
                    console.error('Error fetching customer:', error);
                } finally {
                    this.bill.customerDetailsLoading = false;
                    this.calculateAmountPayable();
                }
            },
            async searchProducts() {
                const bill = this.bill;
                const q = bill.searchQuery.trim();
                if (!q) {
                    bill.searchResults = [];
                    return;
                }
                try {
                    const res = await fetch('{{ route('search-products') }}', {
                        method: 'POST',
                        headers: {
                            'Content-Type': 'application/json',
                            'X-CSRF-Token': document.querySelector('meta[name=csrf-token]').content
                        },
                        body: JSON.stringify({
                            query: q
                        })
                    });

                    const products = await res.json();
                    bill.searchResults = products.map(p => {
                        const sellPrice = Number(p.sell_price || 0);
                        const gstRate = Number(p.gst_rate || 0);
                        const basePrice = sellPrice / (1 + (gstRate / 100));
                        return {
                            ...p,
                            sell_price: sellPrice,
                            gst_rate: gstRate,
                            base_price: basePrice,
                            cgst: (basePrice * (gstRate / 200)),
                            sgst: (basePrice * (gstRate / 200)),
                            shopping_wallet_price: Number(p.shopping_wallet_price || 0)
                        };
                    });
                    bill.selectedSearchResultIndex = 0;

                    const lowerQuery = q.toLowerCase();
                    const exactMatch = bill.searchResults.find(result =>
                        lowerQuery === result.item_code?.toLowerCase() ||
                        lowerQuery === result.name?.toLowerCase() ||
                        lowerQuery === result.hsn_code?.toLowerCase() ||
                        lowerQuery === result.barcode1?.toLowerCase() ||
                        lowerQuery === result.barcode2?.toLowerCase()
                    );

                    if (exactMatch) {
                        this.addItemToBill(exactMatch);
                        bill.searchQuery = '';
                        bill.searchResults = [];
                    }

                } catch (e) {
                    console.error("Error searching products:", e);
                    bill.searchResults = [];
                }
            },

            addItemToBill(prod) {
                const bill = this.bill;
                const existingItem = bill.billItems.find(item =>
                    item.item_identifier === prod.item_identifier && item.type === prod.type
                );

                if (existingItem) {
                    existingItem.quantity++;
                } else {
                    bill.billItems.unshift({
                        ...prod,
                        quantity: 1,
                    });
                }

                bill.searchQuery = '';
                bill.searchResults = [];
                this.calculateTotals();

                this.$nextTick(() => {
                    const container = this.$el.querySelector('.table-scroll-container');
                    if (container) {
                        container.scrollTop = container.scrollHeight;
                    }
                });
            },

            // Modals
            openPriceChangeModal() {
                if (this.bill.selectedItemIndex === -1) return;
                this.tempPrice = this.bill.billItems[this.bill.selectedItemIndex].sell_price;
                const modal = new bootstrap.Modal(document.getElementById('priceChangeModal'));
                modal.show();
            },
            confirmChangePrice() {
                if (this.bill.selectedItemIndex !== -1 && this.tempPrice > 0) {
                    const item = this.bill.billItems[this.bill.selectedItemIndex];
                    const newPrice = Number(this.tempPrice);
                    item.sell_price = newPrice;

                    const gstRate = Number(item.gst_rate || 0);
                    const basePrice = newPrice / (1 + (gstRate / 100));
                    item.base_price = basePrice;
                    item.cgst = (basePrice * (gstRate / 200));
                    item.sgst = (basePrice * (gstRate / 200));

                    this.calculateTotals();
                }
                bootstrap.Modal.getInstance(document.getElementById('priceChangeModal')).hide();
            },
            openQuantityChangeModal() {
                if (this.bill.selectedItemIndex === -1) return;
                this.tempQty = this.bill.billItems[this.bill.selectedItemIndex].quantity;
                const modal = new bootstrap.Modal(document.getElementById('qtyChangeModal'));
                modal.show();
            },
            confirmChangeQty() {
                if (this.bill.selectedItemIndex !== -1 && this.tempQty > 0) {
                    this.bill.billItems[this.bill.selectedItemIndex].quantity = Number(this.tempQty);
                    this.calculateTotals();
                }
                bootstrap.Modal.getInstance(document.getElementById('qtyChangeModal')).hide();
            },
            openDiscountModal() {
                this.tempDiscount = this.bill.discountAmount;
                this.tempDiscountNote = '';
                const modal = new bootstrap.Modal(document.getElementById('discountModal'));
                modal.show();
            },
            confirmDiscount() {
                this.bill.discountAmount = Number(this.tempDiscount);
                this.calculateTotals();
                bootstrap.Modal.getInstance(document.getElementById('discountModal')).hide();
            },
            openAdditionalChargeModal() {
                this.tempCharge = this.bill.additionalCharge;
                this.tempChargeNote = '';
                const modal = new bootstrap.Modal(document.getElementById('chargeModal'));
                modal.show();
            },
            confirmCharge() {
                this.bill.additionalCharge = Number(this.tempCharge);
                this.calculateTotals();
                bootstrap.Modal.getInstance(document.getElementById('chargeModal')).hide();
            },
            openSettings() {
                const modal = new bootstrap.Modal(document.getElementById('settingsModal'));
                modal.show();
            },
            saveSettings() {
                localStorage.setItem('posPrintType', this.printType);
                bootstrap.Modal.getInstance(document.getElementById('settingsModal')).hide();
            },

            openNewCustomerModal() {
                this.newCustomer.phone_number = this.bill.customerIdentifier;
                const modal = new bootstrap.Modal(document.getElementById('newCustomerModal'));
                modal.show();
            },

            async confirmNewCustomer() {
                try {
                    const response = await fetch('/api/customers/register', {
                        method: 'POST',
                        headers: {
                            'Content-Type': 'application/json',
                            'X-CSRF-TOKEN': document.querySelector('meta[name="csrf-token"]').getAttribute('content'),
                        },
                        body: JSON.stringify(this.newCustomer),
                    });
                    const newCust = await response.json();
                    this.bill.customerDetails = newCust;
                    this.bill.customerIdentifier = newCust.phone_number;
                    bootstrap.Modal.getInstance(document.getElementById('newCustomerModal')).hide();
                } catch (error) {
                    console.error('Error registering new customer:', error);
                    // Use a custom message box instead of alert
                    this.showMessage('Failed to register new customer.', 'danger');
                }
            },

            // Keyboard shortcuts
            handleGlobalKeys(event) {
                if (event.key === 'Escape') {
                    // Close modals first
                    // You might need to add logic here to close active Bootstrap modals
                    if (!this.bill.searchQuery) {
                        window.history.back();
                    } else {
                        this.bill.searchQuery = '';
                        this.bill.searchResults = [];
                    }
                }
                if (event.key === 'F1') {
                    event.preventDefault();
                    this.$refs.searchInput.focus();
                }
                if (event.key === 'F2') {
                    event.preventDefault();
                    this.openDiscountModal();
                }
                if (event.key === 'F3') {
                    event.preventDefault();
                    this.openAdditionalChargeModal();
                }
                if (event.key === 'F6') {
                    event.preventDefault();
                    this.updateAndPrintBill(); // Call the update and print function
                }
                if (event.key === 'Delete') {
                    event.preventDefault();
                    this.deleteSelectedItem();
                }
            },
            handleSearchEnter() {
                if (this.bill.searchResults.length > 0) {
                    this.addItemToBill(this.bill.searchResults[this.bill.selectedSearchResultIndex]);
                }
            },
            selectNextResult() {
                if (this.bill.searchResults.length > 0) {
                    this.bill.selectedSearchResultIndex = Math.min(this.bill.selectedSearchResultIndex + 1, this.bill.searchResults.length - 1);
                }
            },
            selectPrevResult() {
                if (this.bill.searchResults.length > 0) {
                    this.bill.selectedSearchResultIndex = Math.max(this.bill.selectedSearchResultIndex - 1, 0);
                }
            },
            clearSearchResults() {
                this.bill.searchQuery = '';
                this.bill.searchResults = [];
            },
            updateItemQuantity(idx) {
                if (this.bill.billItems[idx].quantity < 1) {
                    this.bill.billItems[idx].quantity = 1;
                }
                this.calculateTotals();
            },

            // Main Actions
            async updateAndPrintBill() {
                // This is the function that the F6 button will call.
                // It calls the main update function and tells it to print.
                await this.updateBill(true);
            },



            async updateBill(print = false) {

    this.calculateTotals();

    if (this.bill.billItems.length === 0) {
        this.showMessage('Cannot update an empty invoice. Please add items.', 'warning');
        return;
    }

    if (!this.bill.paymentMode) {
        this.showMessage('Please select a payment mode.', 'warning');
        return;
    }

    const payload = {
        customerDetails: this.bill.customerDetails,
        billItems: this.bill.billItems,
        subTotal: this.bill.subTotal,
        gstTotal: this.bill.gstTotal,
        discountAmount: this.bill.discountAmount,
        additionalCharge: this.bill.additionalCharge,
        cashWalletDeduction: this.bill.cashWalletDeduction,
        shoppingWalletDeduction: this.bill.shoppingWalletDeduction,
        amountPayable: this.bill.amountPayable,
        paymentMode: this.bill.paymentMode,
        sold_by: this.bill.soldBy,
    };

     try {
                    const response = await fetch(`/admin/update_invoice/${this.bill.order_id}`, {
                        method: 'POST',
                        headers: {
                            'Content-Type': 'application/json',
                            'X-CSRF-TOKEN': document.querySelector('meta[name="csrf-token"]').getAttribute('content'),
                        },
                        body: JSON.stringify(payload),
                    });

                    if (!response.ok) {
                        let errorText = 'Failed to update invoice.';
                        const contentType = response.headers.get("content-type");
                        if (contentType && contentType.indexOf("application/json") !== -1) {
                            const error = await response.json();
                            errorText = error.message || errorText;
                        } else {
                            errorText = `Server error: ${response.status} ${response.statusText}. Please check the server logs.`;
                        }
                        throw new Error(errorText);
                    }

                    const result = await response.json();
                    const updatedOrderId = result.order_id; 

                    if (print) {
                        const printRoute = this.printType === 'a4' ? '{{ route("invoice_a4", ["orderId" => ":orderId"]) }}' : '{{ route("invoice_thermal", ["orderId" => ":orderId"]) }}';
                        window.location.href = printRoute.replace(':orderId', updatedOrderId);
                    } else {
                        this.showMessage('Invoice updated successfully!', 'success');
                    }
                } catch (error) {
                    console.error('Update failed:', error);
                    this.showMessage(error.message || 'An unexpected error occurred. Please try again.', 'danger');
                }
            },


            // async updateBill(print = false) {
            //     if (this.bill.billItems.length === 0) {
            //         this.showMessage('Cannot update an empty invoice. Please add items.', 'warning');
            //         return;
            //     }
            //     if (!this.bill.paymentMode) {
            //         this.showMessage('Please select a payment mode.', 'warning');
            //         return;
            //     }

            //     const payload = {
            //         customerDetails: this.bill.customerDetails,
            //         billItems: this.bill.billItems,
            //         subTotal: this.bill.subTotal,
            //         gstTotal: this.bill.gstTotal,
            //         discountAmount: this.bill.discountAmount,
            //         additionalCharge: this.bill.additionalCharge,
            //         cashWalletDeduction: this.bill.cashWalletDeduction,
            //         shoppingWalletDeduction: this.bill.shoppingWalletDeduction,
            //         amountPayable: this.bill.amountPayable,
            //         paymentMode: this.bill.paymentMode,
            //         sold_by: this.bill.soldBy,
            //     };

            //     try {
            //         const response = await fetch(`/admin/update_invoice/${this.bill.order_id}`, {
            //             method: 'POST',
            //             headers: {
            //                 'Content-Type': 'application/json',
            //                 'X-CSRF-TOKEN': document.querySelector('meta[name="csrf-token"]').getAttribute('content'),
            //             },
            //             body: JSON.stringify(payload),
            //         });

            //         if (!response.ok) {
            //             let errorText = 'Failed to update invoice.';
            //             const contentType = response.headers.get("content-type");
            //             if (contentType && contentType.indexOf("application/json") !== -1) {
            //                 const error = await response.json();
            //                 errorText = error.message || errorText;
            //             } else {
            //                 errorText = `Server error: ${response.status} ${response.statusText}. Please check the server logs.`;
            //             }
            //             throw new Error(errorText);
            //         }

            //         const result = await response.json();
            //         const updatedOrderId = result.order_id; 

            //         if (print) {
            //             const printRoute = this.printType === 'a4' ? '{{ route("invoice_a4", ["orderId" => ":orderId"]) }}' : '{{ route("invoice_thermal", ["orderId" => ":orderId"]) }}';
            //             window.location.href = printRoute.replace(':orderId', updatedOrderId);
            //         } else {
            //             this.showMessage('Invoice updated successfully!', 'success');
            //         }
            //     } catch (error) {
            //         console.error('Update failed:', error);
            //         this.showMessage(error.message || 'An unexpected error occurred. Please try again.', 'danger');
            //     }
            // },



            // Custom Message Box (replaces alert)
            showMessage(message, type = 'info') {
                const messageBox = document.createElement('div');
                messageBox.className = `alert alert-${type} alert-dismissible fade show fixed-top w-75 mx-auto mt-3`;
                messageBox.setAttribute('role', 'alert');
                messageBox.innerHTML = `
                    ${message}
                    <button type="button" class="btn-close" data-bs-dismiss="alert" aria-label="Close"></button>
                `;
                document.body.appendChild(messageBox);

                // Automatically close after 5 seconds
                setTimeout(() => {
                    const bsAlert = bootstrap.Alert.getInstance(messageBox);
                    if (bsAlert) {
                        bsAlert.dispose();
                    } else {
                        messageBox.remove();
                    }
                }, 5000);
            }
        };
    }
</script>