<!DOCTYPE html>
<html lang="en" x-data="posBilling()" @keydown.window="handleGlobalKeys($event)">
<head>
    <meta charset="UTF-8" />
    <meta name="viewport" content="width=device-width, initial-scale=1.0" />
    <title>POS Billing</title>
    <link href="https://cdn.jsdelivr.net/npm/bootstrap@5.3.0/dist/css/bootstrap.min.css" rel="stylesheet" />
    <link href="https://cdn.jsdelivr.net/npm/bootstrap-icons@1.10.5/font/bootstrap-icons.css" rel="stylesheet" />
    <script defer src="https://cdn.jsdelivr.net/npm/alpinejs@3/dist/cdn.min.js"></script>
    <meta name="csrf-token" content="{{ csrf_token() }}">
    <style>
        .pos-search-results {
            max-height: 200px;
            overflow-y: auto;
            z-index: 1000;
            position: absolute;
            width: calc(100% - 2rem);
            /* Adjust based on padding/margin of parent */
            background-color: white;
            border: 1px solid #dee2e6;
            border-top: none;
            left: 1rem;
            /* Adjust to align with input */
            right: 1rem;
            /* Adjust to align with input */
        }
        .table-scroll-container {
            max-height: calc(100vh - 450px);
            /* Adjust based on header/footer height */
            overflow-y: auto;
        }
        .table thead th {
            position: sticky;
            top: 0;
            background-color: #f8f9fa;
            z-index: 10;
        }
        .customer-details-compact p {
            margin-bottom: 0;
            line-height: 1.2;
        }
    </style>
</head>
<body>
    <div class="position-fixed top-0 end-0 p-3" style="z-index: 1100">
        <div id="stockToast" class="toast align-items-center text-white bg-danger border-0" role="alert">
            <div class="d-flex">
                <div class="toast-body" id="stockToastMessage"></div>
                <button type="button" class="btn-close btn-close-white me-2 m-auto" data-bs-dismiss="toast"></button>
            </div>
        </div>
    </div>
    <div class="container-fluid bg-light border-bottom py-2 px-3 d-flex justify-content-between align-items-center">
        <button class="btn btn-outline-secondary btn-sm" @click="exitPOS">
            <i class="bi bi-arrow-left"></i> <strong>Exit POS</strong> [ESC]
        </button>
        <h6 class="m-0">POS Billing</h6>
        <div class="d-flex gap-2">
            <button class="btn btn-outline-secondary btn-sm" @click="showHeldBillsModal">
                <strong>Held Bills</strong> [F8]
            </button>
            <button class="btn btn-outline-secondary btn-sm" @click="openSettings">
                <strong>Settings</strong> [CTRL+S]
            </button>
        </div>
    </div>
    <div class="container-fluid p-3">
        <div class="d-flex flex-wrap align-items-center gap-3 px-2 py-2 bg-light border rounded">
            <div class="card-header bg-primary text-white py-2 px-3 rounded-top">
                <h6 class="mb-0">Customer Details</h6>
            </div>
            <div class="flex-grow-1">
                <input type="text" class="form-control" x-model.debounce.500ms="currentBill().customerIdentifier"
                    @input.debounce.500ms="fetchCustomerDetails" placeholder="Type and wait...">
            </div>
            <template x-if="currentBill().customerDetails">
                <div class="d-flex flex-wrap align-items-center gap-3">
                    <div>
                        <strong x-text="currentBill().customerDetails.name"></strong>
                        <small class="text-muted">(ID: <span
                                x-text="currentBill().customerDetails.user_id"></span>)</small>
                    </div>
                    <div>
                        <span class="badge bg-success">
                            ₹<span x-text="currentBill().customerDetails.cash_wallet_balance.toFixed(2)"></span> Cash
                        </span>
                        <span class="badge bg-info text-dark">
                            ₹<span x-text="currentBill().customerDetails.shopping_wallet_balance.toFixed(2)"></span>
                            Shopping
                        </span>
                    </div>
                    <div class="text-muted small">
                        <i class="bi bi-telephone"></i>
                        <span x-text="currentBill().customerDetails.phone_number"></span>
                    </div>
                    <div class="text-muted small">
                        <i class="bi bi-geo-alt"></i>
                        <span x-text="currentBill().customerDetails.address"></span>
                    </div>
                </div>
            </template>
            <!-- Status Messages -->
            <div class="text-muted small fst-italic"
                x-show="currentBill().customerIdentifier && !currentBill().customerDetails && !currentBill().customerDetailsLoading">
                No customer found.
            </div>
            <div class="text-info small fst-italic" x-show="currentBill().customerDetailsLoading">
                Searching...
            </div>
        </div>
        <ul class="nav nav-tabs mb-1">
            <template x-for="bill in bills" :key="bill.id">
                <li class="nav-item">
                    <a href="#" class="nav-link" :class="{ active: bill.id === activeBillId }"
                        @click.prevent="activateBill(bill.id)">
                        <span x-text="bill.name"></span>
                        <i class="bi bi-x-lg ms-2" x-show="bills.length>1" @click.stop="closeBill(bill.id)"></i>
                    </a>
                </li>
            </template>
            <li class="nav-item">
                <a href="#" class="nav-link" @click.prevent="addNewBill"><i class="bi bi-plus"></i> New Bill
                    [CTRL+B]</a>
            </li>
        </ul>
        <div class="row">
            <div class="col-md-8 border-end position-relative">
                <div class="d-flex gap-2 mb-3">
                    {{-- <button class="btn btn-outline-secondary btn-sm" @click="resetPOS"><i class="bi bi-plus"></i> Clear Bill [CTRL+C]</button> --}}
                    <button class="btn btn-outline-secondary btn-sm" @click="openPriceChangeModal">Change Price
                        [CTRL+P]</button>
                    <button class="btn btn-outline-secondary btn-sm" @click="openQuantityChangeModal">Change QTY
                        [CTRL+Q]</button>
                    <button class="btn btn-outline-secondary btn-sm" @click="deleteSelectedItem()">Delete Item
                        [DEL]</button>
                </div>
                <div class="input-group mb-3">
                    <span class="input-group-text"><i class="bi bi-search"></i></span>
                    <input x-ref="searchInput" class="form-control" type="text" placeholder="Search item/code"
                        x-model.debounce.300ms="currentBill().searchQuery" @input.debounce.300ms="searchProducts"
                        @keydown.enter.prevent="handleSearchEnter" @keydown.arrow-down.prevent="selectNextResult"
                        @keydown.arrow-up.prevent="selectPrevResult" @keydown.escape.stop="clearSearchResults">
                    <span class="input-group-text">[F1]</span>
                </div>
                <ul class="list-group pos-search-results" x-show="currentBill().searchResults.length">
                    <template x-for="(prod, idx) in currentBill().searchResults"
                        :key="prod.item_identifier + prod.type">
                        <li class="list-group-item list-group-item-action"
                            :class="{ active: idx === currentBill().selectedSearchResultIndex }"
                            @click="addItemToBill(prod)" @mouseover="currentBill().selectedSearchResultIndex = idx">
                            <span x-text="prod.name"></span> - <small x-text="prod.item_code"></small> (₹ <span
                                x-text="Number(prod.sell_price || 0).toFixed(2)"></span>)
                        </li>
                    </template>
                </ul>
                <div class="table">
                    <table class="table table-bordered text-center">
                        <thead class="table-light">
                            <tr>
                                <th>#</th>
                                <th>Item</th>
                                <th>Code</th>
                                <th>MRP</th>
                                <th>SP</th>
                                <th>QTY</th>
                                <th>GST</th>
                                <th>Amount</th>
                            </tr>
                        </thead>
                        <tbody>
                            <template x-if="!currentBill().billItems.length">
                                <tr>
                                    <td colspan="8" class="py-5 text-muted">
                                        <i class="bi bi-box-seam fs-1"></i>
                                        <p>Add items...</p>
                                    </td>
                                </tr>
                            </template>
                            <template x-for="(item, idx) in currentBill().billItems" :key="idx">
                                <tr :class="{ 'table-primary': currentBill().selectedItemIndex === idx }"
                                    @click="selectItem(idx)">
                                    <td x-text="idx + 1"></td>
                                    <td x-text="item.name"></td>
                                    <td x-text="item.item_code"></td>
                                    <td x-text="(Number(item.mrp) || 0).toFixed(2)"></td>
                                    <td x-text="(Number(item.sell_price) || 0).toFixed(2)"></td>
                                    <td>
                                        <input type="number" class="form-control form-control-sm text-center"
                                            style="width:80px;margin:auto;" x-model.number="item.quantity"
                                            @change="updateItemQuantity(idx)" min="1">
                                    </td>
                                    <td>
                                        <template x-if="item.gst_price">
                                            <span x-text="(item.gst_price * item.quantity).toFixed(2)"></span>
                                        </template>
                                        <template x-if="!item.gst_price">
                                            <span
                                                x-text="((Number(item.cgst || 0) + Number(item.sgst || 0)) * item.quantity).toFixed(2)"></span>
                                        </template>
                                    </td>
                                    <td x-text="((Number(item.sell_price) || 0) * item.quantity).toFixed(2)"></td>
                                </tr>
                            </template>
                        </tbody>
                    </table>
                </div>
            </div>
            <div class="col-md-4">
                <div class="d-flex gap-2 mb-2">
                    <button class="btn btn-outline-secondary btn-sm flex-fill" @click="openDiscountModal">Add Discount
                        [F2]</button>
                    <button class="btn btn-outline-secondary btn-sm flex-fill" @click="openAdditionalChargeModal">Add
                        Charge [F3]</button>
                </div>
                <div class="card p-3 mb-2">
                    <h6>Bill Details</h6>
                    <!-- ✅ GST total -->
                    <div class="d-flex justify-content-between">
                        <span>Sub Total</span>
                        <span>₹ <span x-text="currentBill().subTotal.toFixed(2)"></span></span>
                    </div>
                    <div class="d-flex justify-content-between">
                        <span>GST</span>
                        <span>₹ <span x-text="currentBill().gstTotal.toFixed(2)"></span></span>
                    </div>
                    <template x-if="currentBill().discountAmount > 0">
                        <div class="d-flex justify-content-between">
                            <span>Discount</span>
                            <span>₹ <span x-text="currentBill().discountAmount.toFixed(2)"></span></span>
                        </div>
                    </template>
                    <template x-if="currentBill().additionalCharge > 0">
                        <div class="d-flex justify-content-between">
                            <span>Charge</span>
                            <span>₹ <span x-text="currentBill().additionalCharge.toFixed(2)"></span></span>
                        </div>
                    </template>
                    <hr>
                    <div class="d-flex justify-content-between fw-bold text-success">
                        <span>Total</span>
                        <span>₹ <span x-text="currentBill().totalAmount.toFixed(2)"></span></span>
                    </div>
                </div>
                <div class="card p-3 mb-2">
                    <h6>Wallet Use</h6>
                    <div class="form-check mb-2">
                        <input class="form-check-input" type="checkbox" id="use_cash_wallet"
                            x-model="currentBill().useCashWallet" @change="calculateAmountPayable"
                            :disabled="!currentBill().customerDetails || currentBill().customerDetails.cash_wallet_balance <= 0">
                        <label class="form-check-label">Use Cash Wallet (Bal ₹<span
                                x-text="currentBill().customerDetails?.cash_wallet_balance.toFixed(2) || '0.00'"></span>)</label>
                        <small x-show="currentBill().cashWalletDeduction>0" class="text-muted">Applied ₹<span
                                x-text="currentBill().cashWalletDeduction.toFixed(2)"></span></small>
                    </div>
                    <div class="form-check mb-2">
                        <input class="form-check-input" type="checkbox" id="use_shopping_wallet"
                            x-model="currentBill().useShoppingWallet" @change="calculateAmountPayable"
                            :disabled="!currentBill().customerDetails || (currentBill().customerDetails.shopping_wallet_balance <=
                                0 && currentBill().totalApplicableShoppingWalletFromItems <= 0)">
                        <label class="form-check-label">Use Shopping Wallet (Bal ₹<span
                                x-text="currentBill().customerDetails?.shopping_wallet_balance.toFixed(2) || '0.00'"></span>)</label>
                        <small
                            x-show="currentBill().totalApplicableShoppingWalletFromItems > 0 && !currentBill().useShoppingWallet"
                            class="text-muted"> (Applicable ₹<span
                                x-text="currentBill().totalApplicableShoppingWalletFromItems.toFixed(2)"></span>)</small>
                        <small x-show="currentBill().shoppingWalletDeduction > 0" class="text-muted"> Applied ₹<span
                                x-text="currentBill().shoppingWalletDeduction.toFixed(2)"></span></small>
                    </div>
                    <div class="mb-3">
                        <label for="payment_mode" class="form-label fw-bold">Select Payment Mode</label>
                        <select id="payment_mode" class="form-select"
                            x-model="bills.find(b => b.id === activeBillId).paymentMode">
                            <option value="">-- Select --</option>
                            <option value="upi">UPI</option>
                            <option value="cash">Cash</option>
                            <option value="card">Card</option>
                        </select>
                    </div>
                    <div class="d-flex justify-content-between fw-bold"><span>Payable</span><span>₹ <span
                                x-text="currentBill().amountPayable.toFixed(2)"></span></span></div>
                </div>
                <div class="d-flex gap-2">
                    <button class="btn btn-primary flex-fill" @click="saveBillAndPrint">Save & Print [F6]</button>
                    <button class="btn btn-outline-secondary flex-fill" @click="saveHeldBill">Hold Bill [F7]</button>
                </div>
            </div>
        </div>
    </div>
    <div class="modal fade" id="priceChangeModal" tabindex="-1">
        <div class="modal-dialog modal-sm modal-dialog-centered">
            <div class="modal-content">
                <div class="modal-header">
                    <h5>Change Price</h5><button class="btn-close" data-bs-dismiss="modal"></button>
                </div>
                <div class="modal-body"><input type="number" class="form-control" x-model.number="tempPrice"></div>
                <div class="modal-footer"><button class="btn btn-primary"
                        @click="confirmChangePrice">Apply</button><button class="btn btn-outline-secondary"
                        data-bs-dismiss="modal">Cancel</button></div>
            </div>
        </div>
    </div>
    <div class="modal fade" id="qtyChangeModal" tabindex="-1">
        <div class="modal-dialog modal-sm modal-dialog-centered">
            <div class="modal-content">
                <div class="modal-header">
                    <h5>Change Quantity</h5><button class="btn-close" data-bs-dismiss="modal"></button>
                </div>
                <div class="modal-body"><input type="number" class="form-control" x-model.number="tempQty"
                        min="1"></div>
                <div class="modal-footer"><button class="btn btn-primary"
                        @click="confirmChangeQty">Apply</button><button class="btn btn-outline-secondary"
                        data-bs-dismiss="modal">Cancel</button></div>
            </div>
        </div>
    </div>
    <div class="modal fade" id="discountModal" tabindex="-1">
        <div class="modal-dialog modal-sm modal-dialog-centered">
            <div class="modal-content">
                <div class="modal-header">
                    <h5>Add Discount</h5><button class="btn-close" data-bs-dismiss="modal"></button>
                </div>
                <div class="modal-body"><input type="number" class="form-control mb-2"
                        x-model.number="tempDiscount" placeholder="Amount"><input class="form-control"
                        x-model="tempDiscountNote" placeholder="Reason (opt)"></div>
                <div class="modal-footer"><button class="btn btn-primary"
                        @click="confirmDiscount">Apply</button><button class="btn btn-outline-secondary"
                        data-bs-dismiss="modal">Cancel</button></div>
            </div>
        </div>
    </div>
    <div class="modal fade" id="chargeModal" tabindex="-1">
        <div class="modal-dialog modal-sm modal-dialog-centered">
            <div class="modal-content">
                <div class="modal-header">
                    <h5>Add Charge</h5><button class="btn-close" data-bs-dismiss="modal"></button>
                </div>
                <div class="modal-body"><input type="number" class="form-control mb-2" x-model.number="tempCharge"
                        placeholder="Amount"><input class="form-control" x-model="tempChargeNote"
                        placeholder="Reason (opt)"></div>
                <div class="modal-footer"><button class="btn btn-primary"
                        @click="confirmCharge">Apply</button><button class="btn btn-outline-secondary"
                        data-bs-dismiss="modal">Cancel</button></div>
            </div>
        </div>
    </div>
    <div class="modal fade" id="heldBillsModal" tabindex="-1">
        <div class="modal-dialog modal-lg modal-dialog-centered">
            <div class="modal-content">
                <div class="modal-header">
                    <h5>Held Bills</h5><button class="btn-close" data-bs-dismiss="modal"></button>
                </div>
                <div class="modal-body">
                    <ul class="list-group"><template x-for="hb in heldBills" :key="hb.order_id">
                            <li class="list-group-item d-flex justify-content-between align-items-center"><span><strong
                                        x-text="hb.order_id"></strong> – ₹<span
                                        x-text="hb.total_amount.toFixed(2)"></span></span>
                                <div><button class="btn btn-secondary btn-sm me-2"
                                        @click="resumeBill(hb.order_id)">Resume</button><button
                                        class="btn btn-danger btn-sm"
                                        @click="deleteHeldBill(hb.order_id)">Delete</button></div>
                            </li>
                        </template></ul>
                </div>
            </div>
        </div>
    </div>
    <div class="modal fade" id="settingsModal" tabindex="-1">
        <div class="modal-dialog modal-sm modal-dialog-centered">
            <div class="modal-content">
                <div class="modal-header">
                    <h5>Select Print Type</h5>
                    <button class="btn-close" data-bs-dismiss="modal"></button>
                </div>
                <div class="modal-body">
                    <div class="form-check">
                        <input class="form-check-input" type="radio" x-model="printType" value="a4"
                            id="print_a4">
                        <label class="form-check-label" for="print_a4">A4 Invoice</label>
                    </div>
                    <div class="form-check">
                        <input class="form-check-input" type="radio" x-model="printType" value="thermal"
                            id="print_thermal">
                        <label class="form-check-label" for="print_thermal">Thermal (3 inch)</label>
                    </div>
                </div>
                <div class="modal-footer">
                    <button class="btn btn-primary" @click="saveSettings">Save</button>
                    <button class="btn btn-outline-secondary" data-bs-dismiss="modal">Close</button>
                </div>
            </div>
        </div>
    </div>
    <div class="modal fade" id="newCustomerModal" tabindex="-1">
        <div class="modal-dialog modal-sm modal-dialog-centered">
            <div class="modal-content">
                <div class="modal-header">
                    <h5>Register New Customer</h5>
                    <button class="btn-close" data-bs-dismiss="modal"></button>
                </div>
                <div class="modal-body">
                    <div class="mb-3">
                        <label for="new_customer_phone" class="form-label">Phone Number</label>
                        <input type="text" class="form-control" id="new_customer_phone"
                            x-model="newCustomer.phone_number" disabled>
                    </div>
                    <div class="mb-3">
                        <label for="new_customer_name" class="form-label">Name <span
                                class="text-danger">*</span></label>
                        <input type="text" class="form-control" id="new_customer_name"
                            x-model="newCustomer.name">
                    </div>
                    <div class="mb-3">
                        <label for="new_customer_email" class="form-label">Email (Optional)</label>
                        <input type="email" class="form-control" id="new_customer_email"
                            x-model="newCustomer.email">
                    </div>
                </div>
                <div class="modal-footer">
                    <button class="btn btn-primary" @click="confirmNewCustomer">Register</button>
                    <button class="btn btn-outline-secondary" data-bs-dismiss="modal">Cancel</button>
                </div>
            </div>
        </div>
    </div>
</body>
</html>
<script src="https://cdn.jsdelivr.net/npm/bootstrap@5.3.0/dist/js/bootstrap.bundle.min.js"></script>
<script>
    function posBilling() {
        return {
            bills: [{
                id: Date.now(),
                name: 'Bill 1',
                orderId: null,
                customerIdentifier: '',
                customerDetails: null,
                customerDetailsLoading: false,
                billItems: [],
                searchQuery: '',
                searchResults: [],
                selectedSearchResultIndex: 0,
                selectedItemIndex: null,
                subTotal: 0,
                discountAmount: 0,
                additionalCharge: 0,
                totalAmount: 0,
                useCashWallet: false,
                cashWalletDeduction: 0,
                useShoppingWallet: false,
                shoppingWalletDeduction: 0,
                totalApplicableShoppingWalletFromItems: 0,
                amountPayable: 0,
                // paymentMode: '',
                paymentMode: 'cash',
                paymentReceived: 0,
                changeAmount: 0,
                tempDiscountNote: '',
                tempChargeNote: '',
                gstTotal: 0,
            }],
            activeBillId: null,
            heldBills: [],
            tempPrice: 0,
            tempQty: 1,
            tempDiscount: 0,
            tempDiscountNote: '',
            tempCharge: 0,
            tempChargeNote: '',
            printType: 'thermal',
            newCustomer: {
                phone_number: '',
                name: '',
                email: ''
            },
            openSettings() {
                new bootstrap.Modal(document.getElementById('settingsModal')).show();
            },
            saveSettings() {
                const modal = bootstrap.Modal.getInstance(document.getElementById('settingsModal'));
                modal.hide();
                console.log('Saved print type:', this.printType);
            },
            init() {
                this.activeBillId = this.bills[0].id;
                this.$watch('activeBillId', () => {
                    const bill = this.currentBill();
                    this.tempDiscount = bill.discountAmount;
                    this.tempDiscountNote = bill.tempDiscountNote;
                    this.tempCharge = bill.additionalCharge;
                    this.tempChargeNote = bill.tempChargeNote;
                    this.calculateBill();
                });
            },
            currentBill() {
                return this.bills.find(b => b.id === this.activeBillId);
            },
            handleGlobalKeys(e) {
                const map = {
                    Escape: this.exitPOS.bind(this),
                    F1: this.focusSearchInput.bind(this),
                    F2: this.openDiscountModal.bind(this),
                    F3: this.openAdditionalChargeModal.bind(this),
                    F6: this.saveBillAndPrint.bind(this),
                    F7: this.saveHeldBill.bind(this),
                    F8: this.showHeldBillsModal.bind(this)
                };
                if (e.ctrlKey && e.key === 's') {
                    e.preventDefault();
                    this.openSettings();
                } else if (e.ctrlKey && e.key === 'b') {
                    e.preventDefault();
                    this.addNewBill();
                } else if (e.ctrlKey && e.key === 'cl') {
                    e.preventDefault();
                    this.resetPOS();
                } else if (map[e.key]) {
                    e.preventDefault();
                    map[e.key]();
                }
                if (e.ctrlKey && (e.key === 'p' || e.key === 'P')) {
                    e.preventDefault();
                    this.openPriceChangeModal();
                }
                if (e.ctrlKey && (e.key === 'q' || e.key === 'Q')) {
                    e.preventDefault();
                    this.openQuantityChangeModal();
                }
                if (e.key === 'Delete') {
                    e.preventDefault();
                    this.deleteSelectedItem();
                }
            },
            resetPOS() {
                console.log('POS reset');
                const currentBillId = this.activeBillId;
                const currentBillIndex = this.bills.findIndex(b => b.id === currentBillId);
                if (currentBillIndex !== -1) {
                    const newBill = {
                        id: Date.now(),
                        name: `Bill ${this.bills.length + 1}`,
                        orderId: null,
                        customerIdentifier: '',
                        customerDetails: null,
                        customerDetailsLoading: false,
                        billItems: [],
                        searchQuery: '',
                        searchResults: [],
                        selectedSearchResultIndex: 0,
                        selectedItemIndex: null,
                        subTotal: 0,
                        discountAmount: 0,
                        additionalCharge: 0,
                        totalAmount: 0,
                        useCashWallet: false,
                        cashWalletDeduction: 0,
                        useShoppingWallet: false,
                        shoppingWalletDeduction: 0,
                        totalApplicableShoppingWalletFromItems: 0,
                        amountPayable: 0,
                        // paymentMode: 'cash',
                        paymentMode: '',
                        paymentReceived: 0,
                        changeAmount: 0,
                        tempDiscountNote: '',
                        tempChargeNote: '',
                    };
                    this.bills.splice(currentBillIndex, 1, newBill);
                    this.activeBillId = newBill.id;
                } else {
                    this.bills = [];
                    this.addNewBill();
                }
            },
            exitPOS() {
                console.log('Exit POS');
            },
            addNewBill() {
                const idx = this.bills.length + 1;
                const nb = {
                    id: Date.now(),
                    name: `Bill ${idx}`,
                    orderId: null,
                    customerIdentifier: '',
                    customerDetails: null,
                    customerDetailsLoading: false,
                    billItems: [],
                    searchQuery: '',
                    searchResults: [],
                    selectedSearchResultIndex: 0,
                    selectedItemIndex: null,
                    subTotal: 0,
                    discountAmount: 0,
                    additionalCharge: 0,
                    totalAmount: 0,
                    useCashWallet: false,
                    cashWalletDeduction: 0,
                    useShoppingWallet: false,
                    shoppingWalletDeduction: 0,
                    totalApplicableShoppingWalletFromItems: 0,
                    amountPayable: 0,
                    // paymentMode: 'cash',
                    paymentMode: '',
                    paymentReceived: 0,
                    changeAmount: 0,
                    tempDiscountNote: '',
                    tempChargeNote: '',
                };
                this.bills.push(nb);
                this.activeBillId = nb.id;
            },
            activateBill(id) {
                this.activeBillId = id;
            },
            closeBill(id) {
                this.bills = this.bills.filter(b => b.id !== id);
                if (this.activeBillId === id && this.bills.length) {
                    this.activeBillId = this.bills[0].id;
                } else if (this.bills.length === 0) {
                    this.addNewBill();
                }
            },
            async fetchCustomerDetails() {
                const bill = this.currentBill();
                bill.customerDetails = null;
                bill.customerDetailsLoading = true;
                const id = bill.customerIdentifier.trim();
                if (!id) {
                    bill.customerDetailsLoading = false;
                    this.calculateAmountPayable();
                    return;
                }
                try {
                    const res = await fetch('{{ route('search-customer') }}', {
                        method: 'POST',
                        headers: {
                            'Content-Type': 'application/json',
                            'X-CSRF-Token': document.querySelector('meta[name=csrf-token]').content
                        },
                        body: JSON.stringify({
                            identifier: id
                        })
                    });
                    const data = await res.json();
                    console.log("Fetched Customer:", data);
                    if (data && data.user_id) {
                        bill.customerDetails = data;
                    } else {
                        bill.customerDetails = null;
                        if (/^\d{10}$/.test(id)) {
                            this.newCustomer.phone_number = id;
                            this.newCustomer.name = '';
                            this.newCustomer.email = '';
                            // Delay opening modal slightly to ensure DOM readiness
                            setTimeout(() => {
                                const modal = new bootstrap.Modal(document.getElementById(
                                    'newCustomerModal'));
                                modal.show();
                            }, 100);
                        }
                    }
                } catch (e) {
                    console.error("Error fetching customer details:", e);
                    // Optional detailed error handling
                    if (e.response) {
                        e.response.json().then(err => {
                            console.log("Server responded with error:", err);
                            alert('Server Error:\n' + JSON.stringify(err));
                        });
                    } else {
                        alert('Client Error:\n' + e.message);
                    }
                }
                bill.customerDetailsLoading = false;
                this.calculateAmountPayable();
            },
            async confirmNewCustomer() {
                try {
                    const res = await fetch('{{ route('register-new-customer') }}', {
                        method: 'POST',
                        headers: {
                            'Content-Type': 'application/json',
                            'X-CSRF-Token': document.querySelector('meta[name=csrf-token]').content
                        },
                        body: JSON.stringify(this.newCustomer)
                    });
                    let data;
                    try {
                        data = await res.clone().json(); // Use clone in case it's not valid JSON
                    } catch (err) {
                        const text = await res.text();
                        console.error("Invalid JSON response from server:", text);
                        alert('Server error occurred. Please check logs or contact admin.');
                        return;
                    }
                    if (res.ok) {
                        const bill = this.currentBill();
                        bill.customerDetails = {
                            name: data.name,
                            user_id: data.user_id,
                            phone_number: data.phone_number,
                            email: data.email,
                            address: data.address,
                            cash_wallet_balance: data.cash_wallet_balance,
                            shopping_wallet_balance: data.shopping_wallet_balance,
                        };
                        bill.customerIdentifier = data.user_id;
                        bootstrap.Modal.getInstance(document.getElementById('newCustomerModal')).hide();
                        alert('Customer registered and applied to bill!');
                        this.calculateAmountPayable();
                    } else {
                        let errorMessage = 'Failed to register customer.';
                        if (data && data.errors) {
                            errorMessage += '\n' + Object.values(data.errors).flat().join('\n');
                        }
                        alert(errorMessage);
                    }
                } catch (e) {
                    console.error("Error registering new customer:", e);
                    alert('Unexpected error occurred during registration.');
                }
            },
            async searchProducts() {
                const bill = this.currentBill();
                const q = bill.searchQuery.trim();
                if (!q) {
                    bill.searchResults = [];
                    return;
                }
                try {
                    const res = await fetch('{{ route('search-products') }}', {
                        method: 'POST',
                        headers: {
                            'Content-Type': 'application/json',
                            'X-CSRF-Token': document.querySelector('meta[name=csrf-token]').content
                        },
                        body: JSON.stringify({
                            query: q
                        })
                    });
                    const products = await res.json();
                    bill.searchResults = products.map(p => ({
                        ...p,
                        shopping_wallet_price: Number(p.shopping_wallet_price || 0)
                    }));
                    bill.selectedSearchResultIndex = 0;
                    // Auto-add product if one exact match found
                    const lowerQuery = q.toLowerCase();
                    const exactMatch = bill.searchResults.find(result =>
                        lowerQuery === result.item_code?.toLowerCase() ||
                        lowerQuery === result.name?.toLowerCase() ||
                        lowerQuery === result.hsn_code?.toLowerCase() ||
                        lowerQuery === result.barcode1?.toLowerCase() ||
                        lowerQuery === result.barcode2?.toLowerCase()
                    );
                    if (exactMatch) {
                        this.addItemToBill(exactMatch);
                        bill.searchQuery = '';
                        bill.searchResults = [];
                    }
                } catch (e) {
                    console.error("Error searching products:", e);
                    bill.searchResults = [];
                }
            },
            handleSearchEnter() {
                const bill = this.currentBill();
                const i = bill.selectedSearchResultIndex;
                const item = bill.searchResults[i];
                if (item) {
                    this.addItemToBill(item);
                    bill.searchQuery = '';
                }
            },
            selectNextResult() {
                const bill = this.currentBill();
                if (bill.selectedSearchResultIndex < bill.searchResults.length - 1)
                    bill.selectedSearchResultIndex++;
            },
            selectPrevResult() {
                const bill = this.currentBill();
                if (bill.selectedSearchResultIndex > 0) bill.selectedSearchResultIndex--;
            },
            clearSearchResults() {
                this.currentBill().searchResults = [];
            },
            addItemToBill(prod) {
                const bill = this.currentBill();
                if (Number(prod.stock || 0) <= 0) {
                    this.showToast(`"${prod.name}" is out of stock and cannot be added to the bill.`);
                    return;
                }
                const existingItem = bill.billItems.find(item =>
                    item.item_identifier === prod.item_identifier && item.type === prod.type
                );
                if (existingItem) {
                    existingItem.quantity++;
                } else {
                    bill.billItems.unshift({
                        ...prod,
                        quantity: 1,
                        shopping_wallet_price: Number(prod.shopping_wallet_price || 0)
                    });
                }
                bill.searchQuery = '';
                bill.searchResults = [];
                this.calculateBill();
                this.$nextTick(() => {
                    const container = this.$el.querySelector('.table-scroll-container');
                    if (container) {
                        container.scrollTop = 0;
                    }
                });
            },
            showToast(message) {
                const toastMessage = document.getElementById('stockToastMessage');
                const toastElement = document.getElementById('stockToast');
                if (toastMessage && toastElement) {
                    toastMessage.innerText = message;
                    const toast = new bootstrap.Toast(toastElement, {
                        delay: 3000,
                        autohide: true
                    });
                    toast.show();
                }
            },
            selectItem(idx) {
                this.currentBill().selectedItemIndex = idx;
            },
            updateItemQuantity(idx) {
                const bill = this.currentBill();
                if (bill.billItems[idx].quantity < 1) {
                    bill.billItems[idx].quantity = 1;
                }
                this.calculateBill();
            },
            deleteSelectedItem() {
                const bill = this.currentBill();
                if (bill.selectedItemIndex !== null && bill.selectedItemIndex >= 0 && bill.selectedItemIndex < bill
                    .billItems.length) {
                    bill.billItems.splice(bill.selectedItemIndex, 1);
                    bill.selectedItemIndex = null;
                    this.calculateBill();
                } else {
                    alert('Please select an item to delete.');
                }
            },
            calculateBill() {
                const bill = this.currentBill();
                bill.subTotal = bill.billItems.reduce((s, i) => s + Number(i.sell_price || 0) * i.quantity, 0);
                // Recalculate total applicable shopping wallet from items
                bill.totalApplicableShoppingWalletFromItems = bill.billItems.reduce((sum, item) => {
                    return sum + (Number(item.shopping_wallet_price || 0) * item.quantity);
                }, 0);
                bill.totalAmount = Math.max(0, bill.subTotal - bill.discountAmount + bill.additionalCharge);
                this.calculateAmountPayable();
            },
            calculateAmountPayable() {
                const bill = this.currentBill();
                bill.cashWalletDeduction = 0;
                bill.shoppingWalletDeduction = 0;
                let remainingAmountForPayment = bill.totalAmount;
                let totalApplicableShoppingWalletFromItems = 0;
                bill.billItems.forEach(item => {
                    totalApplicableShoppingWalletFromItems += (Number(item.shopping_wallet_price || 0) * item
                        .quantity);
                });
                bill.totalApplicableShoppingWalletFromItems = totalApplicableShoppingWalletFromItems;
                if (bill.customerDetails) {
                    const customerCashWallet = Number(bill.customerDetails.cash_wallet_balance || 0);
                    const customerShoppingWallet = Number(bill.customerDetails.shopping_wallet_balance || 0);
                    if (bill.useCashWallet) {
                        bill.cashWalletDeduction = Math.min(customerCashWallet, remainingAmountForPayment);
                        remainingAmountForPayment -= bill.cashWalletDeduction;
                    }
                    if (bill.useShoppingWallet) {
                        const shoppingWalletCanApply = Math.min(
                            totalApplicableShoppingWalletFromItems,
                            customerShoppingWallet,
                            remainingAmountForPayment
                        );
                        bill.shoppingWalletDeduction = shoppingWalletCanApply;
                        remainingAmountForPayment -= shoppingWalletCanApply;
                    }
                } else {
                    // If no customer is selected, wallets cannot be used.
                    bill.useCashWallet = false;
                    bill.useShoppingWallet = false;
                    bill.cashWalletDeduction = 0;
                    bill.shoppingWalletDeduction = 0;
                    bill.totalApplicableShoppingWalletFromItems = 0; // Reset if no customer
                }
                bill.amountPayable = Math.max(0, remainingAmountForPayment);
                this.calculateChange();
            },
            calculateChange() {
                const bill = this.currentBill();
                bill.changeAmount = bill.paymentReceived - bill.amountPayable;
            },
            openPriceChangeModal() {
                const b = this.currentBill(),
                    idx = b.selectedItemIndex;
                if (idx !== null) {
                    this.tempPrice = b.billItems[idx].sell_price;
                    new bootstrap.Modal(document.getElementById('priceChangeModal')).show();
                } else {
                    alert('Please select an item in the bill to change its price.');
                }
            },
            confirmChangePrice() {
                const b = this.currentBill(),
                    idx = b.selectedItemIndex;
                if (idx !== null) {
                    b.billItems[idx].sell_price = Number(this.tempPrice); // Ensure it's a number
                    this.calculateBill();
                }
                bootstrap.Modal.getInstance(document.getElementById('priceChangeModal')).hide();
            },
            openQuantityChangeModal() {
                const b = this.currentBill(),
                    idx = b.selectedItemIndex;
                if (idx !== null) {
                    this.tempQty = b.billItems[idx].quantity;
                    new bootstrap.Modal(document.getElementById('qtyChangeModal')).show();
                } else {
                    alert('Please select an item in the bill to change its quantity.');
                }
            },
            confirmChangeQty() {
                const b = this.currentBill(),
                    idx = b.selectedItemIndex;
                if (idx !== null) {
                    b.billItems[idx].quantity = Number(this.tempQty); // Ensure it's a number
                    this.calculateBill();
                }
                bootstrap.Modal.getInstance(document.getElementById('qtyChangeModal')).hide();
            },
            openDiscountModal() {
                this.tempDiscount = this.currentBill().discountAmount;
                this.tempDiscountNote = this.currentBill().tempDiscountNote;
                new bootstrap.Modal(document.getElementById('discountModal')).show();
            },
            confirmDiscount() {
                const b = this.currentBill();
                b.discountAmount = Number(this.tempDiscount); // Ensure it's a number
                b.tempDiscountNote = this.tempDiscountNote;
                this.calculateBill();
                bootstrap.Modal.getInstance(document.getElementById('discountModal')).hide();
            },
            openAdditionalChargeModal() {
                this.tempCharge = this.currentBill().additionalCharge;
                this.tempChargeNote = this.currentBill().tempChargeNote;
                new bootstrap.Modal(document.getElementById('chargeModal')).show();
            },
            confirmCharge() {
                const b = this.currentBill();
                b.additionalCharge = Number(this.tempCharge); // Ensure it's a number
                b.tempChargeNote = this.tempChargeNote;
                this.calculateBill();
                bootstrap.Modal.getInstance(document.getElementById('chargeModal')).hide();
            },
            async saveBill(status) {
                const b = this.currentBill();
                if (b.billItems.length === 0) {
                    alert('Please add items to the bill before saving.');
                    return {
                        success: false
                    };
                }
                if (b.totalAmount <= 0 && b.amountPayable > 0) {
                    alert(
                        'Bill total cannot be zero or negative unless fully covered by wallet. Please check amounts.'
                    );
                    return {
                        success: false
                    };
                }
                const customerDetailsToSend = b.customerDetails ? {
                    name: b.customerDetails.name,
                    user_id: b.customerDetails.user_id,
                    phone_number: b.customerDetails.phone_number,
                    email: b.customerDetails.email || null,
                    address: b.customerDetails.address || 'N/A',
                    cash_wallet_balance: b.customerDetails.cash_wallet_balance,
                    shopping_wallet_balance: b.customerDetails.shopping_wallet_balance,
                } : null;
                const payload = {
                    order_id: b.orderId || null,
                    user_id: b.customerDetails?.user_id || null,
                    customer_details: customerDetailsToSend,
                    bill_items: b.billItems.map(item => ({
                        item_identifier: item.item_identifier,
                        product_id_main: item.product_id_main || null,
                        type: item.type,
                        name: item.name,
                        item_code: item.item_code,
                        mrp: item.mrp,
                        sell_price: item.sell_price,
                        purchase_price: item.purchase_price,
                        quantity: item.quantity,
                        gst: item.gst,
                        cgst: item.cgst,
                        sgst: item.sgst,
                        gst_price: item.gst_price,
                        shopping_wallet_price: item.shopping_wallet_price || 0,
                    })),
                    sub_total: b.subTotal,
                    discount_amount: b.discountAmount,
                    additional_charge: b.additionalCharge,
                    total_amount: b.totalAmount,
                    use_cash_wallet: b.useCashWallet,
                    use_shopping_wallet: b
                        .useShoppingWallet, // This maps to the correct property for Laravel backend
                    cash_wallet_deduction: b.cashWalletDeduction,
                    shopping_wallet_deduction: b.shoppingWalletDeduction,
                    amount_payable: b.amountPayable,
                    payment_mode: b.paymentMode,
                    payment_received: b.paymentReceived ?? 0,
                    change_amount: b.changeAmount ?? 0,
                    status: status,
                    description: (status === 'held' || b.discountAmount > 0 || b.additionalCharge > 0) ?
                        (b.tempDiscountNote || b.tempChargeNote || 'N/A') : null
                };
                const url = '{{ route('save-bill') }}';
                try {
                    const res = await fetch(url, {
                        method: 'POST',
                        headers: {
                            'Content-Type': 'application/json',
                            'X-CSRF-Token': document.querySelector('meta[name=csrf-token]').content
                        },
                        body: JSON.stringify(payload)
                    });
                    const data = await res.json();
                    if (!res.ok) {
                        let messages = 'Something went wrong. Please try again.';
                        if (data.errors) {
                            messages = Object.values(data.errors).flat().join('\n');
                        } else if (data.message) {
                            messages = data.message;
                        }
                        alert('Operation failed:\n' + messages);
                        return {
                            success: false
                        };
                    }
                    alert(data.message);
                    return {
                        success: true,
                        order_id: data.order_id
                    };
                } catch (e) {
                    console.error("❌ Bill Save Error:", e);
                    alert('An unexpected error occurred. Please check console for details.');
                    return {
                        success: false
                    };
                }
            },
            async saveBillAndPrint() {
                const resp = await this.saveBill('completed');
                if (resp.success && resp.order_id) {
                    const printUrl =
                        this.printType === 'thermal' ?
                        `{{ url('admin/invoice_thermal') }}/${resp.order_id}` :
                        `{{ url('admin/invoice_a4') }}/${resp.order_id}`;
                    window.open(printUrl);
                    const printedBillId = this.activeBillId;
                    this.closeBill(printedBillId);
                    if (this.bills.length === 0) {
                        this.addNewBill();
                    }
                }
            },
            async saveHeldBill() {
                const resp = await this.saveBill('held');
                if (resp.success) {
                    // Reset the current bill after holding it
                    this.resetPOS();
                }
            },
            async showHeldBillsModal() {
                try {
                    const res = await fetch('{{ route('held-bills') }}');
                    if (!res.ok) throw new Error('Failed to fetch held bills');
                    this.heldBills = await res.json();
                    new bootstrap.Modal(document.getElementById('heldBillsModal')).show();
                } catch (e) {
                    console.error("Error fetching held bills:", e);
                    alert('Could not load held bills.');
                }
            },
            async resumeBill(orderId) {
                try {
                    const res = await fetch(`{{ url('admin/resume-bill') }}/${orderId}`, {
                        method: 'POST',
                        headers: {
                            'X-CSRF-Token': document.querySelector('meta[name=csrf-token]').content
                        }
                    });
                    const data = await res.json();
                    if (res.ok && data.bill_data) {
                        const newBill = JSON.parse(JSON.stringify(this.bills[0]));
                        Object.assign(newBill, {
                            id: Date.now(), // Assign a new unique ID for the frontend session
                            name: `Resumed Bill (${data.bill_data.order_id})`,
                            orderId: data.bill_data.order_id,
                            customerIdentifier: data.bill_data.customerIdentifier,
                            customerDetails: data.bill_data.customerDetails,
                            billItems: data.bill_data.billItems.map(item => ({
                                ...item,
                                shopping_wallet_price: Number(item.shopping_wallet_price || 0)
                            })),
                            discountAmount: data.bill_data.discountAmount,
                            additionalCharge: data.bill_data.additionalCharge,
                            useCashWallet: data.bill_data.use_cash_wallet,
                            useShoppingWallet: data.bill_data.use_shopping_wallet,
                            cashWalletDeduction: data.bill_data.cash_wallet_deduction,
                            shoppingWalletDeduction: data.bill_data.shopping_wallet_deduction,
                            totalApplicableShoppingWalletFromItems: data.bill_data
                                .total_applicable_shopping_wallet_from_items || 0,
                            paymentMode: data.bill_data.payment_mode,
                            paymentReceived: data.bill_data.payment_received,
                            tempDiscountNote: data.bill_data.description || '',
                            tempChargeNote: data.bill_data.description || '',
                            searchQuery: '',
                            searchResults: [],
                            selectedSearchResultIndex: 0,
                            selectedItemIndex: null,
                        });
                        this.bills.push(newBill);
                        this.activeBillId = newBill.id;
                        this.calculateBill();
                        bootstrap.Modal.getInstance(document.getElementById('heldBillsModal')).hide();
                    } else {
                        alert(data.message || 'Failed to resume bill.');
                    }
                } catch (e) {
                    console.error("Error resuming bill:", e);
                    alert('An error occurred while resuming the bill.');
                }
            },
            async deleteHeldBill(orderId) {
                if (!confirm('Are you sure you want to delete this held bill? This action cannot be undone.')) {
                    return;
                }
                try {
                    const res = await fetch(`{{ url('admin/delete-held-bill') }}/${orderId}`, {
                        method: 'DELETE',
                        headers: {
                            'X-CSRF-Token': document.querySelector('meta[name=csrf-token]').content
                        }
                    });
                    if (res.ok) {
                        this.heldBills = this.heldBills.filter(h => h.order_id !== orderId);
                        alert('Held bill deleted successfully.');
                    } else {
                        const data = await res.json();
                        alert(data.message || 'Failed to delete held bill.');
                    }
                } catch (e) {
                    console.error("Error deleting held bill:", e);
                    alert('An error occurred while deleting the held bill.');
                }
            },
            resetPOS() {
                const current = this.currentBill();
                Object.assign(current, {
                    orderId: null,
                    customerIdentifier: '',
                    customerDetails: null,
                    customerDetailsLoading: false,
                    billItems: [],
                    searchQuery: '',
                    searchResults: [],
                    selectedSearchResultIndex: 0,
                    selectedItemIndex: null,
                    subTotal: 0,
                    discountAmount: 0,
                    additionalCharge: 0,
                    totalAmount: 0,
                    useCashWallet: false,
                    cashWalletDeduction: 0,
                    useShoppingWallet: false,
                    shoppingWalletDeduction: 0,
                    totalApplicableShoppingWalletFromItems: 0,
                    amountPayable: 0,
                    paymentMode: 'cash',
                    // paymentMode: '',
                    paymentReceived: 0,
                    changeAmount: 0,
                    tempDiscountNote: '',
                    tempChargeNote: '',
                });
            }
        };
    }
</script>
