<?php

namespace App\Http\Controllers\Video;

use App\Http\Controllers\Controller;
use Illuminate\Http\Request;
use App\Models\Video;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Validator;
use App\Models\UserVideo;
use App\Models\IncomeWallet;
use App\Models\User;
use App\Models\VideoIncome;
use App\Models\Membership;

class VideoController extends Controller
{

    public function add_video()
    {
        $datatables = Video::orderby('id', 'desc')->get();
        return view('admin.add_video', compact('datatables'));
    }

    public function add_video_form(Request $request)
    {
        try {
            Log::info('Video form submission started.', ['request_data' => $request->all()]);

            // Validation
            $validator = Validator::make($request->all(), [
                'note'        => 'required',
                'task'        => 'required',
                'video_link'  => 'required|url',
                'cover_image' => 'required',
                'time'        => 'required',
                'date'        => 'required',

            ]);

            if ($validator->fails()) {
                Log::error('Validation failed.', ['errors' => $validator->errors()->all()]);
                return redirect()->back()->with('errormessage', $validator->errors()->first());
            }

            // Unique Video ID Generation
            do {
                $videoId = 'VDO' . rand(1000, 9999);
                $exists = DB::table('videos')->where('video_id', $videoId)->exists();
            } while ($exists);

            Log::info('Generated unique video_id.', ['video_id' => $videoId]);

            $imagepath = null;
            if ($request->hasFile('cover_image')) {
                $payment_image     = $request->file('cover_image');
                $payment_imagename = time() . '_' . $payment_image->getClientOriginalName();
                $payment_image->move(public_path('assets/images/cover_images'), $payment_imagename);
                $imagepath = 'assets/images/cover_images/' . $payment_imagename;
            }

            // Save to Database
            $video              = new Video();
            $video->video_id    = $videoId;
            $video->date        = $request->date;
            $video->note        = $request->note;
            $video->task        = $request->task;
            $video->video_link  = $request->video_link;
            $video->time        = $request->time;
            $video->cover_image = $imagepath;
            $video->save();

            Log::info('Video saved successfully.', ['video_id' => $videoId]);

            return redirect()->route('add_video')->with('successmessage', 'Video added successfully.');
        } catch (\Exception $e) {
            Log::error('Exception occurred while adding video.', ['error' => $e->getMessage()]);
            return redirect()->route('add_video')->with('errormessage', "Exception Error: " . $e->getMessage());
        }
    }


    public function video_delete(Request $request)
    {
        $video = Video::findOrFail($request->id);
        if ($video) {
            $video->delete();
            return redirect()->route('add_video')->with('successmessage', 'Notification deleted successfully.');
        }
        return redirect()->back()->with('errormessage', 'Error not found');
    }


    public function daily_task()
    {
        try {
            $today = date('Y-m-d');

            $videocount = Video::whereDate('date', $today)->count();

            $videotask = Video::whereDate('date', $today)->orderBy('id', 'desc')->first();

            $uservideo = UserVideo::whereDate('created_at', $today)
                ->where('user_id', Auth::user()->user_id)
                ->count();

            return response()->json([
                'status'  => 'success',
                'message' => 'Daily Task',
                'data'    => [
                    'date'          => $today,
                    "task"          => $videotask ? $videotask->task : null,
                    "video_count"   => $videocount,
                    "watched_count" => $uservideo,
                ],
                'code' => 200,
            ], 200);
        } catch (\Exception $e) {
            return response()->json([
                'status'  => 'error',
                'message' => 'Something went wrong',
                'error'   => $e->getMessage(),
                'code'    => 500,
            ], 500);
        }
    }

    public function daily_videos()
    {
        $today           = date('Y-m-d');
        $videos          = Video::whereDate('date', $today)->get();
        $watchedVideoIds = UserVideo::whereDate('created_at', $today)
            ->where('user_id', Auth::user()->user_id)
            ->pluck('video_id')
            ->toArray();

        $videos = $videos->map(function ($video) use ($watchedVideoIds) {
            $video->is_watched = in_array($video->video_id, $watchedVideoIds) ? 1 : 0;
            return $video;
        });

        if ($videos->isNotEmpty()) {
            return response()->json([
                'status'  => 'success',
                'message' => 'Daily Task',
                'date'    => $today,
                'data'    => $videos,
                'code'    => 200,
            ], 200);
        } else {
            return response()->json([
                'status'  => 'error',
                'message' => 'No data found!',
                'data'    => [],
                'code'    => 404,
            ], 404);
        }
    }


    public function single_video($video_id = "")
    {
        try {
            $videos = Video::where('video_id', $video_id)->first();

            return response()->json([
                'status'  => 'success',
                'message' => 'Daily Task',
                'data'    => $videos,
                'code' => 200,
            ], 200);
        } catch (\Exception $e) {
            return response()->json([
                'status'  => 'error',
                'message' => 'Something went wrong',
                'error'   => $e->getMessage(),
                'code'    => 500,
            ], 500);
        }
    }


    public function video_form(Request $request)
    {
        try {
            Log::info('Request fields:', [
                'video_id'     => $request->video_id,
                'is_completed' => $request->is_completed
            ]);

            $validator = Validator::make($request->all(), [
                'video_id'     => 'required|exists:videos,video_id',
                'is_completed' => 'required|boolean',
            ]);

            if ($validator->fails()) {
                Log::info('Validation errors:', $validator->errors()->toArray());

                return response()->json([
                    'status'  => 'error',
                    'message' => 'Validation errors',
                    'code'    => 422,
                    'data'    => $validator->errors()
                ], 422);
            }

            Log::info('Validation Passed');

            $watchedVideo = UserVideo::where('video_id', $request->video_id)
                ->where('user_id', Auth::user()->user_id)
                ->first();

            if ($watchedVideo) {
                Log::info('Duplicate video entry found for user', [
                    'user_id'  => Auth::user()->user_id,
                    'video_id' => $request->video_id
                ]);

                return response()->json([
                    'status'  => 'error',
                    'message' => 'Duplicate entry',
                    'code'    => 422,
                    'data'    => 'Video Already Added'
                ], 422);
            }

            Log::info('Video not watched before, proceeding to save.');

            $userId         = Auth::user()->user_id;
            $data           = new UserVideo();
            $data->user_id  = $userId;
            $data->video_id = $request->video_id;
            $data->save();

            Log::info('Video saved successfully', [
                'user_id'  => $userId,
                'video_id' => $request->video_id
            ]);

            if ($request->is_completed == 1) {
                $today = date('Y-m-d');

                Log::info('Marking videos as completed', [
                    'user_id' => $userId,
                    'date'    => $today
                ]);

                UserVideo::where('user_id', $userId)
                    ->whereDate('created_at', $today)
                    ->update(['is_completed' => 1]);

                Log::info('Updated completion status in the database');

                $this->video_income($userId);

                Log::info('Video income function executed');
            }

            return response()->json([
                'status'  => 'success',
                'message' => 'Video status updated successfully',
                'data'    => $data,
                'code'    => 200,
            ], 200);
        } catch (\Exception $e) {
            Log::error('Exception Error in video_form:', [
                'error' => $e->getMessage()
            ]);

            return response()->json([
                'status'  => 'error',
                'message' => 'Exception Error',
                'code'    => 500,
                'data'    => 'Something went wrong! Please try again.'
            ], 500);
        }
    }

    public function video_income($user_id)
    {
        try {
            Log::info("video_income started", ['user_id' => $user_id]);

            $user = User::where('user_id', $user_id)->first();
            if (!$user) {
                Log::info("User not found", ['user_id' => $user_id]);
                return false;
            }
            Log::info("User found", ['user_id' => $user_id]);

            Log::info("User found", ['user_id' => $user_id]);

            $membership = Membership::where('membership_id', $user->membership)->first();
            if (!$membership) {
                Log::info("Membership not found", ['membership_id' => $user->membership]);
                return false;
            }
            Log::info("Membership found", ['membership_id' => $membership->membership_id, 'daily_reward' => $membership->daily_reward]);

            $data                = new VideoIncome();
            $data->user_id       = $user_id;
            $data->membership_id = $membership->membership_id;
            $data->amount        = $membership->daily_reward;
            $data->remark        = "Video Income";
            $data->type          = "video";
            $data->save();
            Log::info("Video income saved", ['user_id' => $user_id, 'amount' => $membership->daily_reward]);

            $income_wallet = IncomeWallet::where('user_id', $user_id)->first();

            if ($income_wallet) {
                $income_wallet->video_income    += $membership->daily_reward;
                $income_wallet->shopping_wallet += $membership->daily_reward;
                $income_wallet->save();
                Log::info("Updated income wallet", ['user_id' => $user_id, 'new_video_income' => $income_wallet->video_income, 'new_shopping_wallet' => $income_wallet->shopping_wallet]);
            } else {
                $newWallet = new IncomeWallet();
                $newWallet->user_id         = $user_id;
                $newWallet->video_income    = $membership->daily_reward;
                $newWallet->shopping_wallet = $membership->daily_reward;
                $newWallet->save();
                Log::info("Created new income wallet", ['user_id' => $user_id, 'video_income' => $newWallet->video_income, 'shopping_wallet' => $newWallet->shopping_wallet]);
            }

            return true;
        } catch (\Exception $e) {
            Log::error("Exception in video_income", [
                'user_id' => $user_id,
                'error'   => $e->getMessage(),
                'trace'   => $e->getTraceAsString(),
            ]);
            return false;
        }
    }
}
